$(document).ready(function() {
    // Prevent multiple initializations
    if (window.earningsHandlerInitialized) {
        return;
    }
    window.earningsHandlerInitialized = true;

    // Variables to track current state
    let currentType = 'service';
    let isLoading = false;
    let ajaxRequest = null; // Track current AJAX request

    // Get base URL for assets
    const baseUrl = window.location.origin;
    const assetPath = baseUrl + 'frontend/assets/img';

    // Initialize page - load services by default
    loadEarningsList('service');

    // Remove any existing event handlers and attach new ones
    $('.nav-tabs .nav-link').off('click.earnings').on('click.earnings', function(e) {
        e.preventDefault();

        // Prevent multiple clicks
        if (isLoading) return;

        const $this = $(this);
        const tabId = $this.attr('href');

        // Don't reload if already active
        if ($this.hasClass('active') && $(tabId).hasClass('show active')) {
            return;
        }

        // Remove active class from all tabs
        $('.nav-tabs .nav-link').removeClass('active');
        // Add active class to clicked tab
        $this.addClass('active');

        // Show/hide tab content
        $('.tab-pane').removeClass('show active');
        $(tabId).addClass('show active');

        // Determine type based on tab
        let newType = 'service';
        if (tabId === '#solid-tab2') {
            newType = 'product';
        }

        // Only load if type changed
        if (newType !== currentType) {
            currentType = newType;
            loadEarningsList(currentType);
        }
    });

    // Search functionality
    $('input[placeholder="Search"]').off('keyup.earnings').on('keyup.earnings', debounce(function() {
        const searchTerm = $(this).val().toLowerCase();
        filterTable(searchTerm);
    }, 300));

    /**
     * Load earnings list via AJAX
     * @param {string} type - 'service' or 'product'
     */
    function loadEarningsList(type) {
        if (isLoading) {
            // Cancel previous request if still running
            if (ajaxRequest) {
                ajaxRequest.abort();
            }
        }

        isLoading = true;
        showLoading(type);

        ajaxRequest = $.ajax({
            url: baseUrl + '/provider/earning/list',
            method: 'GET',
            data: {
                type: type,
                _token: $('meta[name="csrf-token"]').attr('content')
            },
            dataType: 'json',
            timeout: 15000, // 15 second timeout
            success: function(response) {
                if (response.status === 'success') {
                    populateTable(response.data.earnings, type);
                    updateSummary(response.data.summary);
                } else {
                    showError('Failed to load earnings: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                if (status !== 'abort') {
                    showError('Failed to load earnings. Please try again.');
                }
            },
            complete: function(xhr, status) {
                if (status !== 'abort') {
                    isLoading = false;
                    hideLoading(type);
                    ajaxRequest = null;
                }
            }
        });
    }

    /**
     * Populate table with earnings data
     * @param {Array} earnings - Array of earning records
     * @param {string} type - 'service' or 'product'
     */
    function populateTable(earnings, type) {
        const tabId = type === 'service' ? '#solid-tab1' : '#solid-tab2';
        const tbody = $(tabId + ' tbody');
        const itemLabel = type === 'service' ? 'Service' : 'Product';

        // Clear existing rows
        tbody.empty();

        if (!earnings || earnings.length === 0) {
            tbody.html(`
                <tr>
                    <td colspan="7" class="text-center py-4">
                        <div class="text-muted">
                            <i class="isax isax-empty-wallet fs-24 mb-2 d-block"></i>
                            No ${type}s found
                        </div>
                    </td>
                </tr>
            `);
            return;
        }

        // Populate rows
        earnings.forEach(function(earning, index) {
            // Use default images with correct paths
            const customerImage = getValidImageUrl(earning.user_details?.profile_image, '/frontend/assets/img/default-profile.png');
            const serviceImage = getValidImageUrl(earning.service_image,
                type === 'service' ? '/frontend/assets/img/default-image-02.jpg' : '/frontend/assets/img/default-placeholder-image.png');

            const row = `
                <tr data-id="${earning.id}" data-index="${index}">
                    <td>
                        <a href="/vendor/products/order-invoice/${earning.invoice_id}">#${earning.invoice_id}</a>
                    </td>
                    <td>
                        <div class="d-flex align-items-center">
                            <a href="javascript:void(0);">
                                <img src="${customerImage}"
                                     class="avatar avatar-md rounded-circle me-2"
                                     alt="user"
                                     loading="lazy"
                                     onerror="handleImageError(this, 'user')">
                            </a>
                            <a href="javascript:void(0);" class="fs-14"
                            title="${earning.customer_email || ''}">
                            ${earning.customer ? earning.customer.charAt(0).toUpperCase() + earning.customer.slice(1) : ''}
                            </a>
                        </div>
                    </td>
                    <td>
                        <div class="d-flex align-items-center">
                            <a href="">
                                <img src="${serviceImage}"
                                     class="avatar avatar-md rounded-3 me-2"
                                     alt="${type}"
                                     loading="lazy"
                                     onerror="handleImageError(this, '${type}')">
                            </a>
                            <a href="" class="fs-14 fw-medium" title="${earning.service}">${truncateText(earning.service, 30)}</a>
                        </div>
                    </td>
                    <td>$${earning.amount}</td>
                    <td>
                        <span class="badge bg-light text-dark">${earning.payment_method}</span>
                    </td>
                    <td>${earning.payment_date}</td>
                    <td>
                        <a href="/vendor/products/order-invoice/${earning.invoice_id}?product_id=${earning.id}" class="btn btn-sm btn-light border p-2 d-inline-flex align-items-center justify-content-center rounded-circle" title="View Invoice">
                            <i class="isax isax-eye fs-14"></i>
                        </a>
                    </td>
                </tr>
            `;

            tbody.append(row);
        });
    }

    /**
     * Get valid image URL or fallback
     * @param {string} imageUrl - Original image URL
     * @param {string} fallback - Fallback image path
     * @returns {string} Valid image URL
     */
    function getValidImageUrl(imageUrl, fallback) {
        if (imageUrl && imageUrl.trim() !== '') {
            // If it's already a full URL, use it
            if (imageUrl.startsWith('http')) {
                return imageUrl;
            }
            // If it starts with assets, use baseUrl
            if (imageUrl.startsWith('assets/')) {
                return baseUrl + '/' + imageUrl;
            }
            // Otherwise, assume it's a path from assets/img
            return assetPath + '/' + imageUrl;
        }
        // Return fallback
        return assetPath + '/' + fallback;
    }

    /**
     * Handle image load errors - Global function
     */
    window.handleImageError = function(img, type) {
        // Prevent infinite loops
        if (img.dataset.errorHandled) return;
        img.dataset.errorHandled = 'true';

        let fallbackImage;
        if (type === 'user') {
            fallbackImage = assetPath + '/default-profile.png';
        } else if (type === 'service') {
            fallbackImage = assetPath + '/default-image-02.jpg';
        } else if (type === 'product') {
            fallbackImage = assetPath + '/default-image-02.jpg';
        } else {
            // Ultimate fallback - create a placeholder
            img.style.display = 'none';
            return;
        }

        img.src = fallbackImage;
    };

    /**
     * Update summary information
     * @param {Object} summary - Summary data
     */
    function updateSummary(summary) {
        if (summary) {
            // Update any summary displays here
        }
    }

    /**
     * Show loading state
     * @param {string} type - 'service' or 'product'
     */
    function showLoading(type) {
        const tabId = type === 'service' ? '#solid-tab1' : '#solid-tab2';
        const tbody = $(tabId + ' tbody');

        tbody.html(`
            <tr>
                <td colspan="7" class="text-center py-4">
                    <div class="spinner-border spinner-border-sm me-2" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    Loading ${type}s...
                </td>
            </tr>
        `);
    }

    /**
     * Hide loading state
     */
    function hideLoading(type) {
        // Loading will be hidden when populateTable is called
    }

    /**
     * Show error message
     * @param {string} message - Error message
     */
    function showError(message) {
        // Show error in the active tab
        const activeTab = $('.tab-pane.show.active');
        const tbody = activeTab.find('tbody');

        tbody.html(`
            <tr>
                <td colspan="7" class="text-center py-4">
                    <div class="text-danger">
                        <i class="isax isax-warning-2 fs-24 mb-2 d-block"></i>
                        ${message}
                        <br>
                        <button class="btn btn-sm btn-outline-primary mt-2" onclick="location.reload()">Retry</button>
                    </div>
                </td>
            </tr>
        `);

        // Optional: Use toast if available
        if (typeof toastr !== 'undefined') {
            toastr.error(message);
        }
    }

    /**
     * Filter table based on search term
     * @param {string} searchTerm - Search term
     */
    function filterTable(searchTerm) {
        const activeTab = $('.tab-pane.show.active');
        const rows = activeTab.find('tbody tr[data-id]'); // Only filter data rows

        if (!searchTerm) {
            rows.show();
            return;
        }

        rows.each(function() {
            const row = $(this);
            const text = row.text().toLowerCase();

            if (text.includes(searchTerm)) {
                row.show();
            } else {
                row.hide();
            }
        });
    }

    /**
     * Truncate text to specified length
     * @param {string} text - Text to truncate
     * @param {number} length - Maximum length
     * @returns {string} Truncated text
     */
    function truncateText(text, length) {
        if (!text || text.length <= length) {
            return text || '';
        }
        return text.substring(0, length) + '...';
    }

    /**
     * Debounce function to prevent excessive calls
     * @param {Function} func - Function to debounce
     * @param {number} delay - Delay in milliseconds
     * @returns {Function} Debounced function
     */
    function debounce(func, delay) {
        let timeoutId;
        return function(...args) {
            clearTimeout(timeoutId);
            timeoutId = setTimeout(() => func.apply(this, args), delay);
        };
    }

    /**
     * Refresh current tab data
     */
    function refreshCurrentTab() {
        if (!isLoading) {
            loadEarningsList(currentType);
        }
    }

    // Expose refresh function globally if needed
    window.refreshEarnings = refreshCurrentTab;
});