/* global $, loadTranslationFile, document, Swal, _l */

(async () => {
    "use strict";
    await loadTranslationFile();

    // Variable to hold the entire product data object for easy access
    let productData = null;
    let isRTL = $("body").data("dir") === "rtl";

    // --- 1. INITIAL DATA FETCH ---
    function fetchServicedetails() {
        let lastSegment = getLastUrlSegment();
        $.ajax({
            type: "GET",
            url: "/product/product-details/" + lastSegment,
            dataType: "json",
            success: function (response) {
                if (response.code === 200 && response.data) {
                    productData = response.data; // Store the data
                    initializeProductView(); // Initialize the page with the data
                } else {
                    // Handle product not found
                    $(".product-detail-container").html(
                        `<p class="text-center">${_l("no_product_found")}</p>`
                    ).removeClass("d-none");
                }

                $(".content-loader").addClass("d-none");
            },
            error: function() {
                $("#content-loader").removeClass("d-none");
            }
        });
    }

    // --- 2. INITIAL PAGE SETUP ---
    function initializeProductView() {
        // Load static content that doesn't change with variations
        loadStaticContent(productData);

        // Load color swatches from variations
        if (productData.variations && productData.variations.length > 0) {
            loadVariationColorSwatches(productData.variations);

            // Set the initial view based on the FIRST available color variation
            const initialColor = productData.variations[0].color;
            updateViewForColor(initialColor);
            $("#selected-color-display").text(initialColor); // Show initial color code
            $("#product_color").html(`
                <span class="product-color-code ms-2" style="background-color: ${initialColor};"></span>
            `);

            if (!initialColor) {
                $(".color_data").addClass("d-none");
            } else {
                $(".color_data").removeClass("d-none");
            }
        } else {
             // If no variations, load the main gallery (if it exists)
            if (productData.gallery && productData.gallery.length > 0) {
                loadSliderImages(productData.gallery);
            }
            // Set price from base product if no variations
            $("#amount").text(productData.currency + productData.starts_from).attr("data-amount", productData.starts_from);
            $("#in_stock").text(productData.source_stock + " " + _l('products_in_stock'));
            $(".color_data").addClass("d-none");
        }

        $(".product-detail-container").removeClass("d-none");

        loadRelatedProducts(productData.related_products, productData.currency);
    }

    // --- 3. CORE DYNAMIC UPDATE FUNCTION ---
    /**
     * This is the main function that updates the page when a color is selected.
     * @param {string} selectedColor - The hex code of the color to display.
     */
    function updateViewForColor(selectedColor) {
        if (!productData || !productData.variations) return;

        // Find the specific variation object that matches the selected color
        const selectedVariation = productData.variations.find(v => v.color === selectedColor);

        if (!selectedVariation) return;

        // a) Update the main image slider with this variation's images
        loadSliderImages(selectedVariation.images);

        // b) Update the available size/memory attributes for this color
        loadAttributesForVariation(selectedVariation.attributes);

        // c) Update the price/stock to the first available attribute of this new color
        const firstAttribute = selectedVariation.attributes[0];
        if (firstAttribute) {
            $("#amount").text(productData.currency + firstAttribute.price).attr("data-amount", firstAttribute.price);
            $("#in_stock").text(firstAttribute.stock + " Products In Stock");
        } else {
            $("#amount").text("N/A");
            $("#in_stock").text("Out of Stock");
        }
    }

    // --- 4. HTML RENDERING FUNCTIONS ---
    /**
     * Loads ALL content that does NOT change with variations, like title, description, related products etc.
     */
    function loadStaticContent(data) {
        $("#add-to-cart-btn").attr("data-product-id", data.id);
        $("#add_wishlist").attr("data-product-id", data.id);
        if (data.wishlist == true) {
            $('.favourite i').addClass('text-danger isax-heart5').removeClass('isax-heart');
        } else {
            $('.favourite i').removeClass('text-danger isax-heart5').addClass('isax-heart');
        }

        $("#buy-now-btn").attr("data-product-id", data.id);
        $("#selected_product_name").text(data.title);

        $("#category_name").html(data.category);
        $("#product_name").html(data.title);
        $("#product_description").html(data.shot_description);

        let avgRating = data.rating;

        let starsHtml = "";
        const maxStars = 5;

        for (let i = 1; i <= maxStars; i++) {
            if (i <= Math.floor(avgRating)) {
            starsHtml += `<i class="fa-solid fa-star text-warning"></i>`;
            } else if (i === Math.ceil(avgRating) && avgRating % 1 >= 0.5) {
            starsHtml += `<i class="fa-solid fa-star-half-stroke text-warning"></i>`;
            } else {
            starsHtml += `<i class="fa-regular fa-star text-warning"></i>`;
            }
        }

        $("#star_container").html(`
                <div class="mb-0 d-flex align-items-center gap-2">
                <div class="d-flex align-items-center fs-14">
                    ${starsHtml}
                </div>
                <p class="fs-14 text-body mb-0">${avgRating} (${data.review_count} ${ data.review_count > 1 ? _l("Reviews") : _l("review") })</p>
                </div>
            `);

        $(".overview_details").html(data.overview.description);

        if (data.specification && data.specification.length > 0) {
            let html = '';
            data.specification.forEach((item, index) => {
                const rowClass = index % 2 === 1 ? 'bg-light' : '';
                html += `
                    <tr class="${rowClass}">
                        <td><h6 class="mb-0 fw-semibold fs-16">${item.label}</h6></td>
                        <td>${item.description}</td>
                    </tr>
                `;
            });
            $("#specification_table tbody").html(html);
        } else {
            $("#specification_table tbody").html(`
                <tr>
                    <td colspan="2" class="text-center text-muted">${_l('no_specification_available')}</td>
                </tr>
            `);
        }

        if (data.shipping_details.international_shipping || data.shipping_details.domestic_shipping) {
            if (data.shipping_details.international_shipping) {
                $("#international_shipping").html(data.shipping_details.international_shipping);
            }

            if (data.shipping_details.domestic_shipping) {
                $("#domestic_shipping").html(data.shipping_details.domestic_shipping);
            }
        } else {
            $(".shipping-info").html(`
                <p class="text-center text-muted">${_l('no_shipping_and_delivery_available')}</p>
            `);
        }

        if (data.social_profile && data.social_profile.length > 0) {
            let socialHtml = `<h6 class="mb-0 fw-medium fs-13">${_l('share')} : </h6>`;
            data.social_profile.forEach(function (profile) {
                socialHtml += `
                    <a href="${profile.url}" target="_blank">
                        <i class="fa-brands ${profile.icon} fs-20 me-2 img-fluid"></i>
                    </a>`;
            });
            $(".social-icons").html(socialHtml);
        } else {
            $(".social-icons").addClass('d-none');
        }

        if (data.map_url && data.map_url !== "") {
            $(".custom-map iframe").attr("src", data.map_url);
            $(".custom-map").show();
        } else {
            $(".custom-map").hide();
        }
    }

    /**
     * Renders the clickable color swatches.
     */
    function loadVariationColorSwatches(variations) {
        let gallery_images = "";
        variations.forEach((variation, index) => {
            let firstImage = variation.images.length > 0 ? variation.images[0] : "/path/to/default.jpg";
            gallery_images += `
                <div class="img-view border select-tag ${index === 0 ? "active" : ""}"
                     data-color="${variation.color}">
                    <img src="${firstImage}" class="img-fluid rounded" alt="Color ${variation.color}">
                </div>
            `;
        });
        $("#product_color_images").html(gallery_images);
    }

    /**
     * Renders the main image slider. Can be called repeatedly.
     */
    function loadSliderImages(imagesArray) {
        let slider_images = "";
        let small_images = "";

        imagesArray.forEach((imgUrl, index) => {
            slider_images += `
                <div class="service-img position-relative">
                    <img src="${imgUrl}" class="img-fluid rounded" alt="slide-img-${index + 1}">
                    <div class="d-flex align-items-center justify-content-end gap-2 position-absolute view-item top-0">
                        <a href="${imgUrl}" data-fancybox="gallery" class="btn btn-icon border rounded"> <i class="isax isax-maximize-21"></i></a>
                    </div>
                </div>`;
            small_images += `
                <div class="img-view">
                    <img src="${imgUrl}" class="img-fluid rounded" alt="slider-thumb-${index + 1}">
                </div>`;
        });

        $('#large-img').html(slider_images);
        $('#small-img').html(small_images);

        // Reinitialize sliders (important if using Slick/Owl)
        $('#large-img').slick('unslick');
        $('#small-img').slick('unslick');

        $('#large-img').slick({
            slidesToShow: 1,
            slidesToScroll: 1,
            arrows: true,
            fade: true,
            asNavFor: '#small-img'
        });

        $('#small-img').slick({
            slidesToShow: 5,
            slidesToScroll: 1,
            asNavFor: '#large-img',
            focusOnSelect: true,
            arrows: false
        });

        // Rebind fancybox
        $("[data-fancybox='gallery']").fancybox();
    }

    /**
     * Renders the size/memory attribute buttons for a specific color variation.
     */
    function loadAttributesForVariation(attributes) {
        let addServicesHtml = `<h6 class="fs-14 fw-bold mb-2">Size : </h6>`;
        if (attributes && attributes.length > 0) {
             addServicesHtml += `<div class="d-flex align-items-center flex-wrap gap-2 mb-2">`;
             attributes.forEach((attr, index) => {
                addServicesHtml += `<a href="javascript:void(0);"
                    class="btn btn-sm bg-light border select-tag1 text-body variation_change ${index === 0 ? "active" : ""}"
                    data-size="${attr.size}"
                    data-price="${attr.price}"
                    data-stock="${attr.stock}">
                    ${attr.size}
                </a>`;
            });
            addServicesHtml += `</div>`;
            $(".add_services_container").html(addServicesHtml);
        } else {
            addServicesHtml = "<p>No sizes available for this color.</p>";
        }
    }

    function loadRelatedProducts(related_products, currency) {
        if (related_products && related_products.length > 0) {
            let html = "";
            related_products.forEach((product) => {
                let avgRating = product.rating;

                let starsHTML = "";
                for (let i = 1; i <= 5; i++) {
                    if (avgRating >= i) {
                        starsHTML += '<i class="fas fa-star text-warning"></i>'; // full star
                    } else if (avgRating >= i - 0.5) {
                        starsHTML += '<i class="fas fa-star-half-alt text-warning"></i>'; // half star
                    } else {
                        starsHTML += '<i class="far fa-star text-warning"></i>'; // empty star
                    }
                }

                html += `
                <div class="product-slide">
                    <div class="product-item mb-0 flex-fill">
                        <div class="product-img position-relative">
                            <a href="/productdetail/${product.slug}">
                                <img src="${product.image}" alt="${product.title}" class="img-fluid w-100">
                            </a>
                            <div class="avatar user-avatar position-absolute bottom-0 left-0">
                                <img src="${product.profile_image}" alt="brand-img-01" class="rounded-circle">
                            </div>
                            <div class="product-layout d-flex flex-column align-items-center justify-content-center gap-2">
                                <a href="javascript:void(0);" class="btn btn1 btn-icon btn-white frontend-list favourite" data-product-id="${product.id}" data-type="products">
                                    <i class="isax ${product.wishlist ? 'text-danger isax-heart5' : 'isax-heart'}"></i>
                                </a>
                                <a href="/productdetail/${product.slug}" class="btn btn3 btn-icon btn-white"><i class="isax isax-eye"></i></a>
                            </div>
                        </div>
                        <div class="product-content">
                            <div class="product-info mb-3">
                                <div class="d-flex align-items-center flex-wrap gap-2 mb-2">
                                    <div class="d-flex align-items-center fs-14">
                                        ${starsHTML}
                                    </div>
                                    <p class="fs-14 text-body mb-0">${avgRating} (${product.review_count} ${product.review_count > 1 ? _l("Reviews") : _l("review")})</p>
                                </div>
                                <h6 class="mb-0"><a href="/productdetail/${product.slug}" class="text-dark">${product.title}</a></h6>
                            </div>
                            <div class="product-action d-flex align-items-center justify-content-between gap-2 flex-wrap">
                                <h6 class="mb-0">${currency}${product.starts_from}</h6>
                                <div class="d-flex align-items-center justify-content-between gap-2 flex-wrap">
                                    <a href="/productdetail/${product.slug}" class="btn btn-icon btn-dark"><i class="isax isax-shopping-cart													"></i></a>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>`;
            });
            $("#related_products_container").html(html);

            initProductSlider();
        }
    }

    function initProductSlider() {
        $('.product-slider').each(function () {
            const $slider = $(this);
            if ($slider.hasClass('slick-initialized')) {
                $slider.slick('unslick'); // destroy if already initialized
            }
            $slider.slick({
                dots: false,
                infinite: true,
                rtl: isRTL,
                speed: 2000,
                slidesToShow: 4,
                slidesToScroll: 1,
                autoplay: false,
                arrows: true,
                prevArrow: '<button type="button" class="slick-prev"><i class="isax isax-arrow-left"></i></button>',
                nextArrow: '<button type="button" class="slick-next"><i class="isax isax-arrow-right-1"></i></button>',
                responsive: [
                    { breakpoint: 1300, settings: { slidesToShow: 3, slidesToScroll: 1 } },
                    { breakpoint: 992,  settings: { slidesToShow: 2, slidesToScroll: 1 } },
                    { breakpoint: 768,  settings: { slidesToShow: 2, slidesToScroll: 1 } },
                    { breakpoint: 576,  settings: { slidesToShow: 1, slidesToScroll: 1 } }
                ]
            });
        });
    }

    // --- 5. EVENT HANDLERS ---
    $(document).on("click", "#product_color_images .select-tag", function () {
        $("#product_color_images .select-tag").removeClass("active");
        $(this).addClass("active");

        const selectedColor = $(this).data("color");
        $("#selected-color-display").text(selectedColor);
        $("#product_color").html(`
            <span class="product-color-code ms-2" style="background-color: ${selectedColor};"></span>
        `);
        if (!selectedColor) {
            $(".color_data").addClass("d-none");
        } else {
            $(".color_data").removeClass("d-none");
        }

        updateViewForColor(selectedColor);
    });

    $(document).on("click", ".variation_change", function () {
        $(".variation_change").removeClass("active");
        $(this).addClass("active");

        $("#amount").text(productData.currency + $(this).data("price")).data("amount", $(this).data("price"));
        $("#in_stock").text($(this).data("stock") + ' ' + _l("products_in_stock"));
    });

    function addToCart(redirectToCart, button) {
        let productId = $("#add-to-cart-btn").data("product-id");
        let quantity = $("#quantity-input").val();
        let color = $("#product_color_images .select-tag.active").data("color");

        let activeVariation = $(".variation_change.active");
        let size = activeVariation.data("size");
        let price = activeVariation.data("price") || $('#amount').data('amount');

        $.ajax({
            url: "/user/cart/add",
            method: "POST",
            data: {
                _token: $('meta[name="csrf-token"]').attr("content"),
                product_id: productId,
                quantity: quantity,
                color: color,
                size: size,
                price: price,
            },
            beforeSend: function () {
                button.prop("disabled", true).html(`
                    <div class="spinner-border spinner-border-md align-middle me-1" role="status" aria-hidden="true"></div>
                `);
            },
            success: function (response) {
                if (response.success) {
                    $(".cart-item-count").text(response.cartCount);
                    if (redirectToCart) {
                        window.location.href = "/user/cart";
                        button.prop("disabled", false).html(`<i class="isax isax-shopping-cart"></i>${_l("buy_now")}`);
                    } else {
                        button.prop("disabled", false).html(`
                            <i class="isax isax-shopping-bag"></i>${_l("go_to_cart")}
                        `).attr("href", "/user/cart").attr('id', 'go-to-cart');
                        $('#buy-now-btn').html(`
                            <i class="isax isax-shopping-cart"></i>${_l("buy_now")}
                        `).attr("href", "/user/cart").removeAttr('id', 'buy-now-btn');
                        showToast("success", response.message);
                    }
                }
            },
            error: function () {
                if (redirectToCart) {
                    showToast("error", _l("An error occurred while adding to cart"));
                    button.prop("disabled", false).html(`<i class="isax isax-shopping-bag"></i>${_l("add_to_cart")}`);
                } else {
                   button.prop("disabled", false).html(`<i class="isax isax-shopping-cart"></i>${_l("buy_now")}`);
                }
            }
        });
    }

    $(document).on("click", "#add-to-cart-btn", (e) => { e.preventDefault(); addToCart(false, $(e.currentTarget)); });
    $(document).on("click", "#buy-now-btn", (e) => { e.preventDefault(); addToCart(true, $(e.currentTarget)); });


    // --- UTILITY & LOADER FUNCTIONS ---
    function getLastUrlSegment() {
        let url = window.location.href.split("?")[0].split("#")[0];
        url = url.replace(/\/+$/, "");
        return url.substring(url.lastIndexOf("/") + 1);
    }

    // --- INITIAL TRIGGER ---
    fetchServicedetails();

    // --- Review Section ---
    $(document).ready(function () {
        $("#reviewForm").validate({
            rules: {
                review: {
                    required: true,
                    minlength: 10,
                },
            },
            messages: {
                review: {
                    required: _l("review_required"),
                    minlength: _l("review_minlength"),
                },
            },
            errorPlacement: function (error, element) {
                var errorId = element.attr("id") + "_error";
                if (element.hasClass("select2-hidden-accessible")) {
                    $("#" + errorId).text(error.text());
                } else {
                    $("#" + errorId).text(error.text());
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").addClass("is-invalid").removeClass("is-valid");
                }
                $(element).addClass("is-invalid").removeClass("is-valid");
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").removeClass("is-invalid").addClass("is-valid");
                }
                $(element).removeClass("is-invalid").addClass("is-valid");
                var errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            onkeyup: function (element) {
                $(element).valid();
            },
            onchange: function (element) {
                $(element).valid();
            },
            submitHandler: function () {
                let formData = new FormData();
                formData.append("review", $("#review").val());
                formData.append("rating", $(".ratings_stars i.filled").length);
                formData.append("product_id", $("#product_id").val());
                formData.append("type", "product");

                $.ajax({
                    type: "POST",
                    url: "/customer/store-review",
                    data: formData,
                    processData: false,
                    contentType: false,
                    headers: {
                        Accept: "application/json",
                        "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr("content"),
                    },
                    beforeSend: function () {
                        $(".submit-review").attr("disabled", true).html(`
                            <span class="spinner-border spinner-border-sm align-middle me-1" role="status" aria-hidden="true"></span> ${_l("submitting")}
                        `);
                    },
                    success: function (resp) {
                        $("#reviewForm")[0].reset();
                        $(".error-text").text("");
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".submit-review").removeAttr("disabled").html(_l("submit"));
                        if (resp.code === 200) {
                            showToast("success", resp.message);
                            $("#leave_review_card").addClass("d-none");
                            listReviews();
                        }
                    },
                    error: function (error) {
                        $(".error-text").text("");
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".submit-review").removeAttr("disabled").html(_l("submit"));
                        if (error.responseJSON.code === 422) {
                            $.each(
                                error.responseJSON.errors,
                                function (key, val) {
                                    $("#" + key).addClass("is-invalid");
                                    $("#" + key + "_error").text(val[0]);
                                }
                            );
                        } else {
                            showToast("error", error.responseJSON.message);
                        }
                    },
                });
            },
        });

        $("#replyReviewForm").validate({
            rules: {
                reply_message: {
                    required: true,
                    minlength: 10,
                },
            },
            messages: {
                reply_message: {
                    required: _l("reply_message_required"),
                    minlength: _l("reply_message_minlength"),
                },
            },
            errorPlacement: function (error, element) {
                var errorId = element.attr("id") + "_error";
                if (element.hasClass("select2-hidden-accessible")) {
                    $("#" + errorId).text(error.text());
                } else {
                    $("#" + errorId).text(error.text());
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element)
                        .next(".select2-container")
                        .addClass("is-invalid")
                        .removeClass("is-valid");
                }
                $(element).addClass("is-invalid").removeClass("is-valid");
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element)
                        .next(".select2-container")
                        .removeClass("is-invalid")
                        .addClass("is-valid");
                }
                $(element).removeClass("is-invalid").addClass("is-valid");
                var errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            onkeyup: function (element) {
                $(element).valid();
            },
            onchange: function (element) {
                $(element).valid();
            },
            submitHandler: function (form) {
                let formData = new FormData(form);
                formData.append("review_id", $("#review_id").val());
                formData.append("product_id", $("#product_id").val());
                formData.append("reply_message", $("#reply_message").val());
                formData.append("type", "product");

                $.ajax({
                    type: "POST",
                    url: "/customer/store-reply-review",
                    data: formData,
                    processData: false,
                    contentType: false,
                    headers: {
                        Accept: "application/json",
                        "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr(
                            "content"
                        ),
                    },
                    beforeSend: function () {
                        $(".send-reply").attr("disabled", true).html(`
                            <span class="spinner-border spinner-border-sm align-middle me-1" role="status" aria-hidden="true"></span> ${_l("submitting")}
                        `);
                    },
                    success: function (resp) {
                        $(".error-text").text("");
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".send-reply").removeAttr("disabled").html(_l("submit"));
                        $("#replyReviewForm")[0].reset();

                        if (resp.code === 200) {
                            showToast("success", resp.message);
                            listReviews();
                            $("#reply_review_modal").modal("hide");
                        }
                    },
                    error: function (error) {
                        $(".error-text").text("");
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".send-reply").removeAttr("disabled").html(_l("submit"));
                        if (error.responseJSON.code === 422) {
                            $.each(error.responseJSON.errors, function (key, val) {
                                $("#" + key).addClass("is-invalid");
                                $("#" + key + "_error").text(val[0]);
                            });
                        } else {
                            showToast("error", error.responseJSON.message);
                        }
                    },
                });
            },
        });
    });

    listReviews();
    function listReviews(page = 1, reset = true) {
        if (reset) {
            $("#review_list_container").empty();
        }

        $.ajax({
            url: "/reviews/list",
            type: "GET",
            data: {
                product_id: $("#product_id").val(),
                page: page,
                type: "product",
            },
            headers: {
                Accept: "application/json",
                "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr("content"),
            },
            beforeSend: function () {
                $(".load-more-reviews-btn").attr("disabled", true).html(`
                    <span class="spinner-border spinner-border-sm align-middle me-1" role="status" aria-hidden="true"></span> ${_l("loading")}
                `);
            },
            complete: function () {
                $(".load-more-reviews-btn").removeAttr("disabled").html(`${_l("load_more")}`);
            },
            success: function (response) {
                if (response.code === 200 && response.data) {
                    renderReviewsMeta(response.data.reviews_meta);
                    renderReviews(response.data.reviews);
                    if (response.data.next_page) {
                        $(".load-more-reviews-btn").data("page", response.data.next_page).removeClass("d-none");
                    } else {
                        $(".load-more-reviews-btn").addClass("d-none");
                    }
                }
            },
            error: function (res) {
                if (res.responseJSON.code === 500) {
                    showToast("success", res.responseJSON.message);
                }
            },
        });
    }

    function renderReviews(reviews) {
        if (reviews.length > 0) {
            $("#review_list_main_container").removeClass("d-none");
            $.each(reviews, function (index, review) {
                let starsHtml = "";
                let avgRating = review.rating;

                for (let i = 1; i <= 5; i++) {
                    if (avgRating >= i) {
                        starsHtml += '<i class="fas fa-star text-warning"></i>'; // full star
                    } else if (avgRating >= i - 0.5) {
                        starsHtml += '<i class="fas fa-star-half-alt text-warning"></i>'; // half star
                    } else {
                        starsHtml += '<i class="far fa-star text-warning"></i>'; // empty star
                    }
                }

                $("#review_list_container").append(`
                    <div class="review-list">
                        <div class="review-items border-top my-4 pt-4">
                            <div class="d-flex align-center justify-content-between gap-2 flex-wrap">
                                <div class="mb-3 d-flex align-items-center gap-2 flex-wrap flex-lg-nowrap">
                                    <div class="avatar avatar-lg">
                                        <img src="${review.user.profile_image}" alt="user" class="img-fluid rounded-circle">
                                    </div>
                                    <div class="">
                                        <h6 class="fs-16  mb-1">${review.user.name}</h6>
                                        <p class="fs-14 mb-0 text-body">${review.review_date}</p>
                                    </div>
                                </div>
                                <div class="d-flex align-items-center gap-2 flex-wrap">
                                    <div class="d-flex align-items-center justify-content-center fs-14">
                                        ${starsHtml}
                                    </div>
                                    <p class="fs-14 mb-0 text-body">${avgRating}</p>
                                </div>
                            </div>
                            <p class="mb-0 text-body">${review.review}</p>
                            <div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mt-3">
                                ${ $("#auth_id").val() == ""
                                    ? `<a href="/user/login" class="text-secondary fw-medium review_reply_btn">${_l("Reply")}</a>`
                                    : `<a href="javascript:void(0);" class="text-secondary fw-medium review_reply_btn" data-review_id="${review.id}">${_l("Reply")}</a>`
                                }
                            </div>
                        </div>
                        <div class="review_reply_list" id="review_reply_list_${review.id}">
                        </div>
                    </div>
                `);
                if (review.replies) {
                    renderReviewReplies(review.id, review.replies);
                }
            });
        } else {
            $("#review_list_main_container").addClass("d-none");
        }
    }

    function renderReviewReplies(review_id, replies) {
        $.each(replies, function (index, reply) {
            $("#review_reply_list_" + review_id).append(`
                <div class="review-items ps-4 mb-4">
                    <div class="d-flex align-center justify-content-between gap-2 pt-4 border-top flex-wrap">
                        <div class="mb-3 d-flex align-items-center gap-2 flex-wrap flex-lg-nowrap">
                            <div class="avatar avatar-lg">
                                <img src="${reply.user.profile_image}" alt="user" class="img-fluid rounded-circle">
                            </div>
                            <div class="">
                                <h6 class="fs-16  mb-1">${reply.user.name}</h6>
                                <p class="fs-14 mb-0 text-body">${reply.reply_date}</p>
                            </div>
                        </div>
                    </div>
                    <p class="mb-0 text-body">${reply.review}</p>
                </div>
            `);
        });
    }

    function renderReviewsMeta(reviews_meta) {
        let avgRating = reviews_meta.avg_ratings;
        let starRatingCount = reviews_meta.star_ratings_count;
        let starRatingCountPercentage = reviews_meta.star_ratings_percentage;
        $(".total_reviews").text(reviews_meta.total_reviews + " " + (reviews_meta.total_reviews > 1 ? _l("Reviews") : _l("review")));
        $(".total_reviews_count").text(reviews_meta.total_reviews);
        $(".average_ratings").text(reviews_meta.avg_ratings);
        let starsHTML = "";
        for (let i = 1; i <= 5; i++) {
            if (avgRating >= i) {
                starsHTML += '<i class="fas fa-star text-warning"></i>'; // full star
            } else if (avgRating >= i - 0.5) {
                starsHTML += '<i class="fas fa-star-half-alt text-warning"></i>'; // half star
            } else {
                starsHTML += '<i class="far fa-star text-warning"></i>'; // empty star
            }
        }
        $(".star_icons").html(starsHTML);

        $("#1_star_progress").width(starRatingCountPercentage["1_star"] + "%");
        $("#2_star_progress").width(starRatingCountPercentage["2_star"] + "%");
        $("#3_star_progress").width(starRatingCountPercentage["3_star"] + "%");
        $("#4_star_progress").width(starRatingCountPercentage["4_star"] + "%");
        $("#5_star_progress").width(starRatingCountPercentage["5_star"] + "%");

        $("#1_star_count").text(starRatingCount["1_star"]);
        $("#2_star_count").text(starRatingCount["2_star"]);
        $("#3_star_count").text(starRatingCount["3_star"]);
        $("#4_star_count").text(starRatingCount["4_star"]);
        $("#5_star_count").text(starRatingCount["5_star"]);
    }

    $(document).on("click", ".review_reply_btn", function () {
        $("#replyReviewForm")[0].reset();
        $(".error-text").text("");
        $(".form-control").removeClass("is-invalid is-valid");
        if (!$("#review_list_container").data("is_allow_reply")) {
            showToast("error", _l("product_reply_not_allowed"));
        } else {
            $("#review_id").val($(this).data("review_id"));
            $("#reply_review_modal").modal("show");
        }
    });

    $(".load-more-reviews-btn").on("click", function () {
        let page = $(this).data("page");
        listReviews(page, false);
    });
})();