/* global loadTranslationFile, document, _l, FormData, showToast, window, setTimeout, clearInterval, setInterval, jQuery */
(function($) {
    "use strict";
    (async () => {
        await loadTranslationFile();

        $(document).ready(function () {
            $("#signin").attr("disabled", false);
            $("#userLoginForm").validate({
                rules: {
                    email: {
                        required: true,
                        email: true
                    },
                    password: {
                        required: true,
                        minlength: 6
                    }
                },
                messages: {
                    email: {
                        required: _l("email_required"),
                        email: _l("valid_email")
                    },
                    password: {
                        required: _l("password_required"),
                        minlength: _l("password_minlength")
                    }
                },
                errorPlacement: function (error, element) {
                    const errorId = element.attr("id") + "_error";
                    $("#" + errorId).text(error.text());
                },
                highlight: function(element) {
                    const $input = $(element);
                    const $group = $input.closest('.input-group');

                    $group.addClass('is-invalid').removeClass('is-valid');
                    $input.addClass('is-invalid').removeClass('is-valid'); // optional
                },
                unhighlight: function(element) {
                    const $input = $(element);
                    const $group = $input.closest('.input-group');

                    $group.removeClass('is-invalid').addClass('is-valid');
                    $input.removeClass('is-invalid').addClass('is-valid');

                    const errorId = element.id + "_error";
                    $("#" + errorId).text("");
                },
                submitHandler: function (form) {
                    const $btn = $("#signin");
                    $btn.attr("disabled", true).html(
                        "<div class=\"spinner-border text-light\" role=\"status\"></div>"
                    );
                    const formData = new FormData(form);
                    const csrfToken = $("meta[name='csrf-token']").attr("content");
                    formData.append("_token", csrfToken);


                    $.ajax({
                        type: "POST",
                        url: "/userlogin",
                        data: formData,
                        processData: false,
                        contentType: false,
                        headers: {
                            Accept: "application/json",
                            "X-CSRF-TOKEN": csrfToken
                        },
                        success: function (resp) {
                            $btn.text(_l("signin")).prop("disabled", false);

                            if (resp.code === 200) {
                                // reset form
                                $("#userLoginForm")[0].reset();
                                $(".form-control").removeClass("is-invalid is-valid");

                                // toast success
                                showToast("success", resp.message);

                                // redirect safely
                                const BASE_URL = window.location.origin;
                                let redirectUrl = resp.redirect_url || "/";

                                // ensure only relative paths (avoid open redirects)
                                if (redirectUrl.startsWith("/") && !redirectUrl.startsWith("//")) {
                                    window.location.href = BASE_URL + redirectUrl;
                                } else {
                                    window.location.href = BASE_URL + "/";
                                }
                            } else {
                                showToast("error", resp.message || "Login failed");
                            }
                        },
                        error: function (xhr) {
                            setTimeout(() => {
                                $btn.text(_l("signin")).prop("disabled", false);
                            }, 300);

                            $(".error-text").text("");
                            $(".form-control").removeClass("is-invalid is-valid");

                            if (xhr.responseJSON && xhr.responseJSON.code === 422) {
                                // validation errors
                                const errorMessages = [];
                                $.each(xhr.responseJSON.errors, function (key, val) {
                                    $("#" + key).addClass("is-invalid");
                                    $("#" + key + "_error").text(val[0]);
                                    errorMessages.push(val[0]);
                                });
                                showToast("error", errorMessages.join("<br>"));
                            } else {
                                // fallback error
                                const message = xhr.responseJSON ? xhr.responseJSON.message : "An error occurred";
                                showToast("error", message);
                            }
                        }
                    });
                }
            });


            $(document).ready(function () {
                $(".copy-login-details").on("click", function (event) {
                    event.preventDefault();

                    const email = $(this).data("email");
                    const password = $(this).data("password");

                    $("#userLoginForm input[name=\"email\"]").val(email);
                    $("#userLoginForm input[name=\"password\"]").val(password);
                });
            });
        });

        function isValidEmail(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }

        let emailTimerInterval;
        let emailTimerTime;

        function startTimer(expireTime) {
            clearInterval(emailTimerInterval);
            emailTimerTime = expireTime * 60;

            setTimeout(function () {
                const otpTimerDisplay = document.getElementById("otp-timer");

                if (!otpTimerDisplay) {
                    showToast("error", "OTP Timer element not found!");
                    return;
                }

                emailTimerInterval = setInterval(function () {
                    const minutes = Math.floor(emailTimerTime / 60);
                    const seconds = emailTimerTime % 60;

                    otpTimerDisplay.textContent = String(minutes).padStart(2, "0") + ":" + String(seconds).padStart(2, "0");

                    if (emailTimerTime <= 0) {
                        clearInterval(emailTimerInterval);
                        otpTimerDisplay.textContent = "00:00";
                    } else {
                        emailTimerTime--;
                    }
                }, 1000);
            }, 500);
        }

        $(document).ready(function () {
            $(document).on("click", "#login_otp, .resendEmailOtp", function (event) {
                event.preventDefault();

                const username = $("[name=\"email\"]").val().trim();

                if (!username || !isValidEmail(username)) {
                    const errorMessage = _l("email_required") || "Please provide a valid email address.";
                    showToast("error", errorMessage);
                    return;
                }

                $.ajax({
                    url: "/otp-settings",
                    type: "POST",
                    data: { email: username },
                    headers: {
                        "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr("content")
                    },
                    beforeSend: function () {
                        $("#signin").attr("disabled", true).html(
                            "<div class=\"spinner-border text-light\" role=\"status\"></div>"
                        );
                        $("#login_otp").prop("disabled", true);
                    },
                    success: function (data) {
                        if (data.code !== 200) {
                            showToast("error", data.error || _l("failed_to_send_otp"));
                            return;
                        }

                        const otpExpireTime = parseInt(data.otp_expire_time.split(" ")[0], 10);
                        const otpDigitLimit = parseInt(data.otp_digit_limit, 10);
                        const inputContainer = $(".inputcontainer");

                        inputContainer.empty();

                        let inputsHtml = "<div class=\"d-flex align-items-center mb-3\">";
                        for (let i = 1; i <= otpDigitLimit; i++) {
                            const nextId = "digit-" + (i + 1);
                            const prevId = "digit-" + (i - 1);
                            inputsHtml +=
                                "<input type=\"text\" " +
                                "class=\"rounded w-100 py-sm-3 py-2 text-center fs-26 fw-bold me-3 digit-" + i + "\" " +
                                "id=\"digit-" + i + "\" " +
                                "name=\"digit-" + i + "\" " +
                                "data-next=\"" + nextId + "\" " +
                                "data-previous=\"" + prevId + "\" " +
                                "maxlength=\"1\">";
                        }
                        inputsHtml += "</div>";
                        inputContainer.append(inputsHtml);

                        $(".inputcontainer").off("input").on("input", "input", function () {
                            if (this.value.length >= 1) {
                                const next = $(this).data("next");
                                if (next) {
                                    $("#" + next).focus();
                                }
                            }
                        });

                        $(".inputcontainer").off("keydown").on("keydown", "input", function (e) {
                            if (e.key === "Backspace" && this.value === "") {
                                const prev = $(this).data("previous");
                                if (prev) {
                                    $("#" + prev).focus();
                                }
                            }
                        });

                        $(".inputcontainer").off("click").on("click", "input", function () {
                            $(this).select();
                        });

                        $("#otp-email-message").text(_l("otp_sent_to_email") + " " + username);
                        $("#otp-email-modal").modal("show");
                        startTimer(otpExpireTime);
                    },
                    error: function (xhr) {
                        const errorMessage = xhr.responseJSON && xhr.responseJSON.error ? xhr.responseJSON.error : _l("failed_to_send_otp");
                        showToast("error", errorMessage);
                    },
                    complete: function () {
                        $("#signin").attr("disabled", false).html(_l("signin"));
                        $("#login_otp").prop("disabled", false);
                    }
                });
            });

            $("#verify-email-forgot-otp-btn").on("click", function () {
                const email = $("[name=\"email\"]").val();
                const forgotEmail = $("[name=\"forgot_email\"]").val();
                const otpDigitLimit = $(".inputcontainer input").length;
                const otp = [];

                for (let i = 1; i <= otpDigitLimit; i++) {
                    const digit = $("#digit-" + i).val();
                    otp.push(digit);
                }

                const otpString = otp.join("");
                const requestData = { otp: otpString };

                if (email) {
                    requestData.email = email;
                } else if (forgotEmail) {
                    requestData.forgot_email = forgotEmail;
                    requestData.login_type = "forgot_email";
                }

                $.ajax({
                    url: "/verify-otp",
                    type: "POST",
                    data: requestData,
                    headers: {
                        "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr("content")
                    },
                    beforeSend: function () {
                        $(".verify-email-otp-btn").attr("disabled", true).html(
                            "<div class=\"spinner-border text-light\" role=\"status\"></div>"
                        );
                    },
                    success: function (response) {
                        if (response.data === "done") {
                            $("#otp-email-modal").modal("hide");
                            $("#reset-password").modal("show");
                            const responseEmail = response.email;
                            $("#email_id").val(responseEmail);
                        } else {
                            $("#otp-email-modal").modal("hide");
                            $("#success_modal").modal("show");
                            window.location.href = "/";
                        }
                    },
                    error: function (xhr) {
                        const errorMessage = xhr.responseJSON && xhr.responseJSON.error ? xhr.responseJSON.error : _l("otp_required");
                        showToast("error", errorMessage);
                    },
                    complete: function () {
                        $(".verify-email-otp-btn").attr("disabled", false).html(_l("verify_otp"));
                    }
                });
            });
        });
    })();

})(jQuery);
