/* global $, loadTranslationFile, document, _l, showToast, localStorage, window, clearInterval, setTimeout, setInterval */
(async () => {
    "use strict";
    await loadTranslationFile();

    $(document).ready(function () {
        let emailTimerInterval;
        let emailTimerTime;

        // Add custom method if needed (but plugin already has `email`)
        $.validator.addMethod("validEmail", function(value, element) {
            return this.optional(element) || /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value);
        }, "Please provide a valid email address.");

        $("#resetpasswordForm").validate({
            rules: {
                email: {
                    required: true,
                    email: true,       // plugin's built-in email check
                    validEmail: true   // optional stricter custom check
                }
            },
            messages: {
                email: {
                    required: _l("email_required") || "Email is required",
                    email: _l("email_invalid") || "Enter a valid email"
                }
            },
            errorPlacement: function(error, element) {
                const errorId = element.attr("name") + "_error";
                $("#" + errorId).html(error);
            },
            highlight: function(element) {
                const $input = $(element);
                const $group = $input.closest('.input-group');

                $group.addClass('is-invalid').removeClass('is-valid');
                $input.addClass('is-invalid').removeClass('is-valid'); // optional
            },
            unhighlight: function(element) {
                const $input = $(element);
                const $group = $input.closest('.input-group');

                $group.removeClass('is-invalid').addClass('is-valid');
                $input.removeClass('is-invalid').addClass('is-valid');

                const errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            submitHandler: function(form, event) {
                event.preventDefault();

                const username = $("[name='email']").val().trim();

                $.ajax({
                    url: "/otp-settings",
                    type: "POST",
                    data: { email: username, type: "forgot" },
                    headers: {
                        "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                    },
                    beforeSend: function () {
                        $("#forgot_otp").attr("disabled", true).html(
                            `<div class="spinner-border text-light" role="status"></div>`
                        );
                    },
                    success: function (data) {
                        if (data.code !== 200) {
                            showToast("error", data.error || _l("failed_to_send_otp"));
                            return;
                        }

                        const otpExpireTime = parseInt(data.otp_expire_time.split(" ")[0], 10);
                        const otpDigitLimit = parseInt(data.otp_digit_limit, 10);
                        const inputContainer = $(".inputcontainer").empty();

                        let inputsHtml = `<div class="d-flex align-items-center mb-3">`;
                        for (let i = 1; i <= otpDigitLimit; i++) {
                            const nextId = `digit-${i + 1}`;
                            const prevId = `digit-${i - 1}`;
                            inputsHtml += `
                                <input type="text"
                                    class="rounded w-100 py-sm-3 py-2 text-center fs-26 fw-bold me-3 digit-${i}"
                                    id="digit-${i}"
                                    name="digit-${i}"
                                    data-next="${nextId}"
                                    data-previous="${prevId}"
                                    maxlength="1">
                            `;
                        }
                        inputsHtml += `</div>`;
                        inputContainer.append(inputsHtml);

                        // Auto-move focus
                        $(".inputcontainer").on("input", "input", function () {
                            if (this.value.length >= 1) {
                                const next = $(this).data("next");
                                if (next) $("#" + next).focus();
                            }
                        });

                        // Backspace navigation
                        $(".inputcontainer").on("keydown", "input", function (e) {
                            if (e.key === "Backspace" && this.value === "") {
                                const prev = $(this).data("previous");
                                if (prev) $("#" + prev).focus();
                            }
                        });

                        $(".inputcontainer").on("click", "input", function () {
                            $(this).select();
                        });

                        const successMessage = _l("otp_sent_to_email", { username }) || "OTP sent to your Email Address";
                        $("#otp-email-message").text(successMessage);
                        $("#otp-email-modal").modal("show");
                        startTimer(otpExpireTime);
                    },
                    error: function (xhr) {
                        const errorMessage = xhr.responseJSON?.error || _l("failed_fetch_otp") || "Failed to fetch OTP settings. Please try again.";
                        showToast("error", errorMessage);
                    },
                    complete: function () {
                        $("#forgot_otp").attr("disabled", false).html(_l("reset_password"));
                    }
                });
            }
        });

        // Trigger validation+submit when clicking the OTP button
        $(document).on("click", "#forgot_otp, .resendEmailOtpForgot", function (e) {
            e.preventDefault();
            $("#resetpasswordForm").submit(); // runs validation + submitHandler
        });


        function showLoader() {
            const loader = document.getElementById("pageLoader");
            if (loader) {
                loader.style.display = "block";
            }
        }

        function hideLoader() {
            const loader = document.getElementById("pageLoader");
            if (loader) {
                loader.style.display = "none";
            }
        }

        $("#verify-email-forgot-otp-btn").on("click", function () {
            const email = $("[name='email']").val();
            const forgotEmail = $("[name='forgot_email']").val();
            const otpDigitLimit = $(".inputcontainer input").length;
            const loginType = "forgot_email";

            const otp = [];
            for (let i = 1; i <= otpDigitLimit; i++) {
                const digit = $(`#digit-${i}`).val();
                otp.push(digit);
            }
            const otpString = otp.join("");

            const requestData = { otp: otpString };

            if (email || forgotEmail) {
                requestData.forgot_email = email || forgotEmail;
                requestData.login_type = loginType;
            }

            $.ajax({
                url: "/verify-otp",
                type: "POST",
                data: requestData,
                headers: {
                    "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                },
                beforeSend: function () {
                    $("#verify-email-forgot-otp-btn").attr("disabled", true).html(
                        "<div class=\"spinner-border text-light\" role=\"status\"></div>"
                    );
                },
                success: function (response) {
                    $("#verify-email-forgot-otp-btn").html(_l("reset_password")).attr("disabled", false);
                    if (response.data === "done") {
                        $("#otp-email-modal").modal("hide");
                        $("#reset-password").modal("show");
                        const resEmail = response.email;
                        localStorage.setItem("email", resEmail);
                        window.location.href = "/reset-password";
                        $("#email_id").val(resEmail);
                    } else {
                        $("#otp-email-modal").modal("hide");
                        $("#success_modal").modal("show");
                        window.location.href = "/";
                    }
                },
                error: function (xhr) {
                    const errorMessage = xhr.responseJSON?.error || _l("otp_required") || "OTP Required";
                    showToast("error", errorMessage);
                },
                complete: function () {
                    $(".verify-email-otp-btn").attr("disabled", false).html(_l("verify_otp") || "Verify OTP");
                }
            });
        });

        function isValidEmail(email) {
            const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }

        function startTimer(expireTime) {
            clearInterval(emailTimerInterval);
            emailTimerTime = expireTime * 60;

            setTimeout(function () {
                const otpTimerDisplay = document.getElementById("otp-timer");

                if (!otpTimerDisplay) {
                    showToast("error", "OTP Timer element not found!");
                    return;
                }

                emailTimerInterval = setInterval(function () {
                    const minutes = Math.floor(emailTimerTime / 60);
                    const seconds = emailTimerTime % 60;

                    otpTimerDisplay.textContent = 
                        String(minutes).padStart(2, "0") + ":" + String(seconds).padStart(2, "0");

                    if (emailTimerTime <= 0) {
                        clearInterval(emailTimerInterval);
                        otpTimerDisplay.textContent = "00:00";
                    } else {
                        emailTimerTime--;
                    }
                }, 1000);
            }, 500);
        }    
    });
})();