(async () => {
    "use strict";
    await loadTranslationFile();

    $(document).ready(function () {
        $("#edit_review_form").validate({
            rules: {
                review: {
                    required: true,
                    minlength: 10,
                },
            },
            messages: {
                review: {
                    required: _l("review_required"),
                    minlength: _l("review_minlength"),
                },
            },
            errorPlacement: function (error, element) {
                var errorId = element.attr("id") + "_error";
                if (element.hasClass("select2-hidden-accessible")) {
                    $("#" + errorId).text(error.text());
                } else {
                    $("#" + errorId).text(error.text());
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").addClass("is-invalid").removeClass("is-valid");
                }
                $(element).addClass("is-invalid").removeClass("is-valid");
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").removeClass("is-invalid").addClass("is-valid");
                }
                $(element).removeClass("is-invalid").addClass("is-valid");
                var errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            onkeyup: function (element) {
                $(element).valid();
            },
            onchange: function (element) {
                $(element).valid();
            },
            submitHandler: function () {
                let formData = new FormData();
                formData.append("id", $("#review_id").val());
                formData.append("review", $("#review").val());
                formData.append("rating", $(".ratings_stars i.filled").length);

                $.ajax({
                    type: "POST",
                    url: "/customer/store-review",
                    data: formData,
                    processData: false,
                    contentType: false,
                    headers: {
                        Accept: "application/json",
                        "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr("content"),
                    },
                    beforeSend: function () {
                        $(".submit-review").attr("disabled", true).html(`
                            <span class="spinner-border spinner-border-sm align-middle me-1" role="status" aria-hidden="true"></span> ${_l("saving")}
                        `);
                    },
                    success: function (resp) {
                        $("#edit_review_form")[0].reset();
                        $(".error-text").text("");
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".submit-review").removeAttr("disabled").html(_l("save_changes"));
                        if (resp.code === 200) {
                            showToast("success", resp.message);
                            $("#edit_review").modal("hide");
                            listReviews();
                        }
                    },
                    error: function (error) {
                        $(".error-text").text("");
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".submit-review").removeAttr("disabled").html(_l("save_changes"));
                        if (error.responseJSON.code === 422) {
                            $.each(
                                error.responseJSON.errors,
                                function (key, val) {
                                    $("#" + key).addClass("is-invalid");
                                    $("#" + key + "_error").text(val[0]);
                                }
                            );
                        } else {
                            showToast("error", error.responseJSON.message);
                        }
                    },
                });
            },
        });
    });

	listReviews();
    function listReviews(page = 1, reset = true) {
        if (reset) {
            $("#review_list_container").empty();
            $(".spinner-loader").removeClass('d-none');
        }

        $.ajax({
            url: "/customer/reviews-list",
            type: "GET",
            data: {
                product_id: $("#product_id").val(),
                page: page,
            },
            headers: {
                Accept: "application/json",
                "X-CSRF-TOKEN": $("meta[name=\"csrf-token\"]").attr("content"),
            },
            beforeSend: function () {
                $(".load-more-reviews-btn").attr("disabled", true).html(`
                    <span class="spinner-border spinner-border-sm align-middle me-1" role="status" aria-hidden="true"></span> ${_l("loading")}
                `);
            },
            complete: function () {
                $(".load-more-reviews-btn").removeAttr("disabled").html(`${_l("load_more")}`);
                $(".spinner-loader").addClass('d-none');
            },
            success: function (response) {
                if (response.code === 200 && response.data) {
					renderReviews(response.data.reviews);
                    if (response.data.next_page) {
                        $(".load-more-reviews-btn").data("page", response.data.next_page).removeClass("d-none");
                    } else {
                        $(".load-more-reviews-btn").addClass("d-none");
                    }
                }
            },
            error: function (res) {
                if (res.responseJSON.code === 500) {
                    showToast("success", res.responseJSON.message);
                }
            },
        });
    }

    function renderReviews(reviews) {
        if (reviews.length > 0) {
            $("#review_list_main_container").removeClass("d-none");
            $.each(reviews, function (index, review) {
                let starsHtml = "";
                let avgRating = review.rating;

                for (let i = 1; i <= 5; i++) {
                    if (avgRating >= i) {
                        starsHtml += '<i class="fas fa-star text-warning"></i>'; // full star
                    } else if (avgRating >= i - 0.5) {
                        starsHtml += '<i class="fas fa-star-half-alt text-warning"></i>'; // half star
                    } else {
                        starsHtml += '<i class="far fa-star text-muted"></i>'; // empty star
                    }
                }

                let productTitle = (review.product_name || "");
                let shopLogo = review.product && review.product.shop ? review.product.shop.shop_logo : "";
                let shopName = review.product && review.product.shop ? review.product.shop.shop_name : "";

                $("#review_list_container").append(`
					<div class="card mb-4">
						<div class="card-body">
							<div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mb-2">
                                <h6 class="fw-semibold mb-0 fs-16">${review.type} : ${productTitle}</h6>
								<div class="d-flex align-items-center">
									${starsHtml}
									<span>(${avgRating})</span>
								</div>
							</div>
							<p>${review.review}</p>
							<hr>
							<div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
								<div class="d-flex align-items-center">
									<div class="d-flex align-items-center">
										<a href="javascript:void(0);"><img src="${shopLogo}" alt="user" class="avatar avatar-sm rounded-circle me-1"></a>
										<a href="javascript:void(0);" class="fs-14 fw-medium">${shopName}</a>
									</div>
								</div>
								<div class="d-flex align-items-center">
									<a href="javascript:void(0);" class="btn btn-sm btn-white d-inline-flex align-items-center me-2 edit-review-btn"
                                        data-bs-toggle="modal" data-bs-target="#edit_review"
                                        data-id="${review.id}"
                                        data-rating="${review.rating}"
                                        data-review="${review.review}">
                                        <i class="isax isax-edit me-1"></i>${_l("Edit")}
                                    </a>
									<a href="javascript:void(0);" class="btn btn-sm btn-white d-flex align-items-center delete-review-btn"
                                        data-bs-toggle="modal" data-bs-target="#delete_modal" data-id="${review.id}">
                                        <i class="isax isax-trash me-1"></i>${_l("Delete")}
                                    </a>
								</div>
							</div>
						</div><!-- end card body -->
					</div>
                `);
            });
        } else {
            $("#review_list_main_container").addClass("d-none");
            $("#review_list_container").append(`
                <div class="text-center my-5">
                    <h5 class="mb-1">${_l("no_reviews_available")}</h5>
                </div>
            `);
        }
    }

    $(".load-more-reviews-btn").on("click", function () {
        let page = $(this).data("page");
        listReviews(page, false);
    });

    // Edit Review
    $(document).on("click", ".edit-review-btn", function () {
        let reviewId = $(this).data("id");
        let reviewText = $(this).data("review");
        let rating = $(this).data("rating");

        $("#review_id").val(reviewId);
        $("#review").val(reviewText);
        $(".ratings_stars i").each(function (index) {
            if (index < rating) {
                $(this).addClass("filled");
            } else {
                $(this).removeClass("filled");
            }
        });
    });

	// Delete Review
	$('#deleteReviewForm').submit(function(e){
		e.preventDefault();
		$.ajax({
			url: '/customer/delete-review',
			type: 'POST',
			data: {
                id: $('#delete_review_id').val(),
            },
            headers: {
                "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
            },
            dataType: 'json',
            beforeSend: function () {
                $("#delete_confirm").attr("disabled", true).html(`
                    <span class="spinner-border spinner-border-sm align-middle me-1" role="status" aria-hidden="true"></span> ${_l("deleting")}
                `);
            },
            complete: function () {
                $("#delete_confirm").removeAttr("disabled").html(`${_l("deleting")}`);
            },
			success: function(res){
                showToast('success', res.message);
                $('#delete_modal').modal('hide');
                listReviews();
			},
			error: function(xhr){
                showToast('error', (xhr.responseJSON.message || _l('default_delete_error')));
			},
		});
	});

   	$(document).on('click', '.delete-review-btn', function() {
	    let reviewId = $(this).data('id');
	    $('#delete_review_id').val(reviewId);
	});



})();