
(async function () {
    "use strict";
    await loadTranslationFile();

    $(document).ready(function () {
        if ($('.select2').length > 0) {
            $('.select2').select2({
                minimumResultsForSearch: 0,
                width: '100%'
            });
        }

        initIntlTelInput();
        const selectedCountry = $("#country").data("country");
        const selectedState = $("#state").data("state");
        const selectedCity = $("#city").data("city");

        getCountries(selectedCountry, selectedState, selectedCity);

        $("#country").on("change", function () {
            const selectedCountry = $(this).val();
            clearDropdown($("#state"));
            clearDropdown($("#city"));
            if (selectedCountry) {
                getStates(selectedCountry);
            }
        });

        $("#state").on("change", function () {
            const selectedState = $(this).val();
            clearDropdown($("#city"));
            if (selectedState) {
                getCities(selectedState);
            }
        });

        $("#phone_number").on("input", function () {
            $(this).val(
                $(this)
                    .val()
                    .replace(/[^0-9]/g, "")
            );
            if ($(this).val().length > 15) {
                $(this).val($(this).val().slice(0, 15));
            }
        });

        $("#postal_code").on("input", function () {
            $(this).val($(this).val().replace(/[^0-9A-Za-z]/g, "").slice(0, 6));
        });

        $("#userProfileForm").validate({
            rules: {
                first_name: {
                    required: true,
                    maxlength: 100,
                    pattern: /^[a-zA-Z]+$/,
                },
                last_name: {
                    required: true,
                    maxlength: 100,
                    pattern: /^[a-zA-Z]+$/,
                },
                user_name: {
                    required: true,
                    maxlength: 100,
                    remote: {
                        url: "/api/user/check-unique",
                        type: "post",
                        headers: {
                            Accept: "application/json",
                        },
                        data: {
                            user_name: function () {
                                return $("#user_name").val();
                            },
                            id: function () {
                                return $("#id").val();
                            },
                        },
                    },
                },
                email: {
                    required: true,
                    email: true,
                    remote: {
                        url: "/api/user/check-unique",
                        type: "post",
                        headers: {
                            Accept: "application/json",
                        },
                        data: {
                            email: function () {
                                return $("#user_email").val();
                            },
                            id: function () {
                                return $("#id").val();
                            },
                        },
                    },
                },
                phone_number: {
                    required: true,
                    minlength: 10,
                    maxlength: 12,
                },
                gender: {
                    required: true,
                },
                dob: {
                    required: true,
                },
                address: {
                    required: true,
                    maxlength: 150,
                },
                country: {
                    required: true,
                },
                state: {
                    required: true,
                },
                city: {
                    required: true,
                },
                postal_code: {
                    required: true,
                    maxlength: 6,
                    pattern: /^[a-zA-Z0-9]*$/,
                },
                currency_code: {
                    required: true,
                },
                profile_image: {
                    extension: "jpeg|jpg|png",
                    filesize: 2048,
                },
                language: {
                    required: true,
                },
            },
            messages: {
                first_name: {
                    required: _l("first_name_required"),
                    maxlength: _l("first_name_maxlength"),
                    pattern: _l("alphabets_allowed"),
                },
                last_name: {
                    required: _l("last_name_required"),
                    maxlength: _l("last_name_maxlength"),
                    pattern: _l("alphabets_allowed"),
                },
                user_name: {
                    required: _l("user_name_required"),
                    maxlength: _l("user_name_maxlength"),
                    remote: _l("user_name_exists"),
                },
                email: {
                    required: _l("email_required"),
                    email: _l("valid_email"),
                    remote: _l("email_exists"),
                },
                phone_number: {
                    required: _l("phone_number_required"),
                    minlength: _l("phone_number_between"),
                    maxlength: _l("phone_number_between"),
                },
                gender: {
                    required: _l("gender_required")
                },
                dob: {
                    required: _l("dob_required")
                },
                address: {
                    required: _l("address_required"),
                    maxlength: _l("address_maxlength"),
                },
                country: {
                    required: _l("country_required"),
                },
                state: {
                    required: _l("state_required"),
                },
                city: {
                    required: _l("city_required"),
                },
                postal_code: {
                    required: _l("postal_code_required"),
                    maxlength: _l("postal_code_maxlength"),
                    pattern: _l("postal_code_char"),
                },
                currency_code: {
                    required: _l("currency_code_required"),
                },
                profile_image: {
                    extension: _l("image_extension"),
                    filesize: _l("image_filesize"),
                }
            },
            errorPlacement: function (error, element) {
                if (element.hasClass("select2-hidden-accessible")) {
                    var errorId = element.attr("id") + "_error";
                    $("#" + errorId).text(error.text());
                } else {
                    var errorId = element.attr("id") + "_error";
                    $("#" + errorId).text(error.text());
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element)
                        .next(".select2-container")
                        .addClass("is-invalid")
                        .removeClass("is-valid");
                }
                $(element).addClass("is-invalid").removeClass("is-valid");
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element)
                        .next(".select2-container")
                        .removeClass("is-invalid")
                        .addClass("is-valid");
                }
                $(element).removeClass("is-invalid").addClass("is-valid");
                var errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            onkeyup: function (element) {
                $(element).valid();
            },
            onchange: function (element) {
                $(element).valid();
            },
            submitHandler: function (form) {
                var formData = new FormData(form);
                $.ajax({
                    url: "/api/save-profile-details",
                    type: "POST",
                    data: formData,
                    enctype: "multipart/form-data",
                    contentType: false,
                    processData: false,
                    cache: false,
                    beforeSend: function () {
                        $("#saveProfile")
                            .attr("disabled", true)
                            .html(
                                `<span class="spinner-border spinner-border-sm align-middle" role="status" aria-hidden="true"></span> ${_l('saving')}`
                            );
                    },
                    success: function (response) {
                        $(".error-text").text("");
                        $("#saveProfile")
                            .removeAttr("disabled")
                            .html(_l('save_changes'));
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".select2-container").removeClass(
                            "is-invalid is-valid"
                        );
                        if (response.code === 200) {
                            getProfileDetails();
                            showToast('success', response.message);
                        }
                    },
                    error: function (error) {
                        $(".error-text").text("");
                        $("#saveProfile")
                            .removeAttr("disabled")
                            .html(_l('save_changes'));
                        $(".form-control").removeClass("is-invalid is-valid");
                        $(".select2-container").removeClass(
                            "is-invalid is-valid"
                        );
                        if (error.responseJSON.code === 422) {
                            $.each(
                                error.responseJSON.errors,
                                function (key, val) {
                                    $("#" + key).addClass("is-invalid");
                                    $("#" + key + "_error").text(val[0]);
                                }
                            );
                        } else {
                            showToast('error', (error.responseJSON.message));
                        }
                    },
                });
            },
        });

        $.validator.addMethod(
            "filesize",
            function (value, element, param) {
                if (element.files.length === 0) return true;
                return element.files[0].size <= param * 1024;
            },
            "File size must be less than {0} KB."
        );
    });

    $("#gender").on("change", function () {
        $(this).valid();
    });
    $("#country").on("change", function () {
        $(this).valid();
    });
    $("#state").on("change", function () {
        $(this).valid();
    });
    $("#city").on("change", function () {
        $(this).valid();
    });

    function initIntlTelInput() {
        const $userPhoneInput = $(".user_phone_number");
        const $intlPhoneInput = $("#international_phone_number");

        if ($userPhoneInput.length > 0) {
            const iti = intlTelInput($userPhoneInput[0], {
                utilsScript: "/front/plugins/intltelinput/js/utils.js",
                separateDialCode: true,
                placeholderNumberType: "",
                autoPlaceholder: "off",
                formatOnDisplay: false,
            });

            const initialPhoneNumber = $.trim($userPhoneInput.val());
            if (initialPhoneNumber) {
                iti.setNumber(initialPhoneNumber);
            }

            $("#userProfileForm").on("submit", function (e) {
                e.preventDefault();

                // Get the phone number directly without validation
                const intlNumber = iti.getNumber() || $.trim($userPhoneInput.val());
                $intlPhoneInput.val(intlNumber); // Store in hidden input
            });
        }
    }

    function clearDropdown(dropdown) {
        dropdown.empty().append(
            $("<option>", {
                value: "",
                text: "Select",
                disabled: true,
                selected: true,
            })
        );
    }

    function getCountries(
        selectedCountry = null,
        selectedState = null,
        selectedCity = null
    ) {
        $.getJSON("/countries.json", function (data) {
            const countrySelect = $("#country");
            clearDropdown(countrySelect);

            $.each(data.countries, function (index, country) {
                countrySelect.append(
                    $("<option>", {
                        value: country.id,
                        text: country.name,
                        selected: country.id == selectedCountry,
                    })
                );
            });

            if (selectedCountry) {
                getStates(selectedCountry, selectedState, selectedCity);
            }
        }).fail(function () {
            console.error("Error loading country data");
        });
    }

    function getStates(
        selectedCountry,
        selectedState = null,
        selectedCity = null
    ) {
        $.getJSON("/states.json", function (data) {
            const stateSelect = $("#state");
            clearDropdown(stateSelect);

            const states = data.states.filter(
                (state) => state.country_id == selectedCountry
            );
            if (states.length === 1) {
                stateSelect.append(
                    $("<option>", {
                        value: states[0].id,
                        text: states[0].name,
                        selected: true,
                    })
                );
                getCities(states[0].id, selectedCity);
            } else {
                $.each(states, function (index, state) {
                    stateSelect.append(
                        $("<option>", {
                            value: state.id,
                            text: state.name,
                            selected: state.id == selectedState,
                        })
                    );
                });

                if (selectedState) {
                    getCities(selectedState, selectedCity);
                }
            }
        }).fail(function () {
            console.error("Error loading state data");
        });
    }

    function getCities(selectedState, selectedCity = null) {
        $.getJSON("/cities.json", function (data) {
            const citySelect = $("#city");
            clearDropdown(citySelect);

            const cities = data.cities.filter(
                (city) => city.state_id == selectedState
            );
            if (cities.length === 1) {
                citySelect.append(
                    $("<option>", {
                        value: cities[0].id,
                        text: cities[0].name,
                        selected: true,
                    })
                );
            } else {
                $.each(cities, function (index, city) {
                    citySelect.append(
                        $("<option>", {
                            value: city.id,
                            text: city.name,
                            selected: city.id == selectedCity,
                        })
                    );
                });
            }
        }).fail(function () {
            console.error("Error loading city data");
        });
    }

    $("#profileImageBtn").on("click", function () {
        $("#profile_image").trigger("click");
    });

    $("#profile_image").on("change", function (event) {
        if ($(this).val() !== "") {
            $(this).valid();
        }
        let reader = new FileReader();
        reader.onload = function (e) {
            $(".profileImagePreview").attr("src", e.target.result).show();
        };
        reader.readAsDataURL(event.target.files[0]);
    });

    async function getProfileDetails() {
        try {
            const userId = $("#id").val();
            const response = await $.ajax({
                url: "/api/get-profile-details",
                type: "POST",
                dataType: "json",
                data: { id: userId, isMobile: 1 },
            });

            if (response.code === 200) {
                var userData = response.data.user_details;
                if (userData.profile_image != null) {
                    $(".headerProfileImg").attr("src", userData.profile_image);
                }
                if (userData.first_name != null && userData.last_name != null) {
                    $(".headerName").text(
                        userData.first_name + " " + userData.last_name
                    );
                }
            }
        } catch (error) {
            console.error(
                "Error fetching user details:",
                error.responseJSON?.message || error.statusText
            );
        }
    }
})();