(async function () {
    "use strict";
    await loadTranslationFile();

    let initialPhoneNumber = null;
    let international_phone_number = null;
    let billing_address_count = 0;
    $(document).ready(function () {
        getBillingAddress();
        initInternationalPhoneInput();
        initSelect();
        initValidation();
        initEvents();
    });

    function getBillingAddress() {
        $.ajax({
            type: "GET",
            url: "/customer/billing-address/list",
            headers: {
                Accept: "application/json",
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            success: function (response) {
                let addressData = "";
                if (response.data && response.data.length > 0) {
                    billing_address_count = response.billing_address_count ?? 0;

                    response.data.forEach((data, index) => {
                        console.log(data.state.name);
                        let is_default = data.is_default == 1 ? 1 : 0;
                        addressData += `
                            <div class="d-flex align-items-center">
                                <input type="radio" id="${data.name + index}" name="billing_address" class="me-3 billing_address" value="${data.id}" ${is_default == 1 ? 'checked' : ''}>
                                <label for="${data.name + index}" class="w-100">
                                    <span class="d-flex align-items-center justify-content-between">
                                        <span>
                                            <span class="d-flex align-items-center mb-1">
                                                <span class="fs-16 mb-0 fw-semibold me-2">${data.name}</span>
                                                ${data.is_default == 1 ? `<span class="badge badge-sm text-secondary bg-soft-secondary">${_l("Default")}</span>` : ''}
                                            </span>
                                            <span class="d-block text-body fs-14">${data.address_line_one}, ${data.address_line_two ? data.address_line_two + ',' : ''} ${data.city.name ?? ''}, ${data.state.name ?? ''}, ${data.country.name ?? ''}, ${data.zipcode}</span>
                                        </span>
                                        <span class="d-flex align-items-center text-lg-end">
                                            <a href="javascript:void(0);" class="btn btn-white btn-sm d-inline-flex align-items-center me-2 edit-address" data-id="${data.id}" data-bs-toggle="modal" data-bs-target="#billing_address_modal"><i class="isax isax-edit me-1"></i>${_l("Edit")}</a>
                                            <a href="javascript:void(0);" class="btn btn-white btn-sm delete-address" data-bs-toggle="modal" data-bs-target="#delete_address_modal" data-id="${data.id}"><i class="isax isax-trash me-1"></i>${_l("Delete")}</a>
                                        </span>
                                    </span>
                                </label>
                            </div>
                            <hr>
                        `;
                    });
                } else {
                    addressData = `<div class="text-center">
                        <span class="fs-16">${_l("no_billing_address_found")}</span>
                    </div>`;
                }
                $("#billing_address_wrapper").html(addressData);
            },
            error: function (error) {
                if (error.responseJSON.code === 500) {
                    showToast("error", error.responseJSON.message);
                }
            },
        });
    }

    function initInternationalPhoneInput() {
        const userPhoneInput = document.querySelector(".phone_number");
        const intlPhoneInput = document.querySelector("#international_phone_number");

        if (userPhoneInput) {
            const iti = intlTelInput(userPhoneInput, {
                utilsScript: window.location.origin + "/front/plugins/intltelinput/js/utils.js",
                separateDialCode: true,
                placeholderNumberType: "",
                autoPlaceholder: "off",
                formatOnDisplay: false,
            });

            userPhoneInput.classList.add("iti");
            userPhoneInput.parentElement.classList.add("intl-tel-input");

            const updateHiddenPhone = () => {
                let intlNumber = iti.getNumber();

                // if intlNumber is empty, build it manually
                if (!intlNumber) {
                    const countryData = iti.getSelectedCountryData();
                    const dialCode = countryData ? countryData.dialCode : "";
                    intlNumber = dialCode ? `+${dialCode}${userPhoneInput.value.trim()}` : userPhoneInput.value.trim();
                }

                intlPhoneInput.value = intlNumber;
                international_phone_number = intlNumber;
            };

            // Update on typing and when country changes
            userPhoneInput.addEventListener("input", updateHiddenPhone);
            userPhoneInput.addEventListener("countrychange", updateHiddenPhone);

            // Update once on init (useful for edit modal prefilled values)
            updateHiddenPhone();
        }
    }

    function initSelect() {
        $(".select2").select2({
            dropdownParent: $("#billing_address_modal"),
            minimumResultsForSearch: 0,
            width: '100%',
            placeholder: _l("Select"),
        });
    }

    function initValidation(){
        $("#billingAddressForm").validate({
            rules: {
                name: {
                    required: true,
                    maxlength: 50,
                },
                phone_number: {
                    required: true,
                    minlength: 10,
                    maxlength: 15
                },
                email: {
                    required: true,
                    email: true,
                },
                address_line_one: {
                    required: true,
                },
                country_id: {
                    required: true,
                },
                state_id: {
                    required: true,
                },
                city_id: {
                    required: true,
                },
                zipcode: {
                    required: true,
                    maxlength: 6,
                },
            },
            messages:{
                name: {
                    required: _l("name_required"),
                },
                phone_number: {
                    required: _l("phone_number_required"),
                    minlength: _l("phone_number_between"),
                    maxlength: _l("phone_number_between"),
                },
                email: {
                    required: _l("email_required"),
                    email: _l("valid_email"),
                },
                address_line_one: {
                    required: _l("address_line_one_required"),
                },
                country_id: {
                    required: _l("country_required"),
                },
                state_id: {
                    required: _l("state_required"),
                },
                city_id: {
                    required: _l("city_required"),
                },
                zipcode: {
                    required: _l("zip_code_required"),
                    maxlength: _l("zip_code_maxlength"),
                },
            },
            errorPlacement: function (error, element) {
                var errorId = element.attr("id") + "_error";
                if (element.hasClass("select2-hidden-accessible")) {
                    $("#" + errorId).text(error.text());
                } else {
                    $("#" + errorId).text(error.text());
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").addClass("is-invalid").removeClass("is-valid");
                }
                $(element).addClass("is-invalid").removeClass("is-valid");
                $("#" + element.id).siblings("span").addClass("me-3");
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").removeClass("is-invalid").addClass("is-valid");
                }
                $(element).removeClass("is-invalid").addClass("is-valid");
                $("#" + element.id).siblings("span").addClass("me-3");
                var errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            onkeyup: function(element) {
                $(element).valid();
                $("#" + element.id).siblings("span").removeClass("me-3");
            },
            onchange: function(element) {
                $(element).valid();
                $("#" + element.id).siblings("span").removeClass("me-3");
            },
            submitHandler: function(form) {
                let formData = new FormData(form);
                formData.set("phone_number", $("#international_phone_number").val());
                formData.set("is_default", $("#is_default").is(":checked") ? 1 : 0);

                $.ajax({
                    type:"POST",
                    url:"/customer/billing-address/store",
                    data: formData,
                    enctype: "multipart/form-data",
                    processData: false,
                    contentType: false,
                    headers: {
                        "Accept": "application/json",
                        "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                    },
                    beforeSend: function () {
                        $(".submitbtn").attr("disabled", true).html(`
                            <span class="spinner-border spinner-border-sm align-middle" role="status" aria-hidden="true"></span> ${_l("saving")}
                        `);
                    },
                    success:function(resp){
                        $(".error-text").text("");
                        $(".form-control, .select2-container").removeClass("is-invalid is-valid");
                        $(".submitbtn").removeAttr("disabled").html($('#id').val() ? _l("save_changes") : _l("add_address"));
                        if (resp.code === 200) {
                            showToast("success", resp.message);
                            $("#billing_address_modal").modal("hide");
                            getBillingAddress();
                        }
                    },
                    error:function(error){
                        $(".error-text").text("");
                        $(".form-control, .select2-container").removeClass("is-invalid is-valid");
                        $(".submitbtn").removeAttr("disabled").html($('#id').val() ? _l("save_changes") : _l("add_address"));
                        if (error.responseJSON.code === 422) {
                            $.each(error.responseJSON.errors, function(key, val) {
                                $("#" + key).addClass("is-invalid");
                                $("#" + key + "_error").text(val[0]);
                            });
                        } else {
                            showToast("error", error.responseJSON.message);
                        }
                    }
                });
            }
        });
    }

    function initEvents() {
        $("#add_address").on("click", function() {
            $("#billingAddressForm")[0].reset();
            $("#id").val("");
            $(".error-text").text("");
            $(".form-control, .select2-container").removeClass("is-invalid is-valid");
            $(".submitbtn").text(_l("add_address"));
            if (billing_address_count >= 5) {
                showToast("error", _l("billing_address_limit_reached"));
            } else {
                $("#billing_address_modal").modal("show");
            }
        });

        $("#phone_number").on("input", function () {
            $(this).val($(this).val().replace(/[^0-9]/g, "").slice(0, 15));
        });

        $("#zipcode").on("input", function () {
            $(this).val($(this).val().replace(/[^0-9A-Za-z]/g, "").slice(0, 6));
        });

        $("#deleteBillingAddressForm").on("submit", function(e){
            e.preventDefault();
            $.ajax({
                url:"/customer/billing-address/delete",
                type:"POST",
                data: {
                    id: $("#delete_id").val()
                },
                headers: {
                    "Accept": "application/json",
                    "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                },
                success: function(response) {
                    if(response.code === 200){
                        showToast("success", response.message);
                        $("#delete_address_modal").modal("hide");
                        getBillingAddress();
                    }
                },
                error: function(res) {
                    if(res.responseJSON.code === 500){
                        showToast("error", res.responseJSON.message);
                    } else {
                        showToast("error", _l("default_delete_error"));
                    }
                }
            });
        });

        $(document).on("change", ".billing_address", function(e){
            e.preventDefault();
            $.ajax({
                url:"/customer/billing-address/set-default",
                type:"POST",
                data: {
                    id: $(this).val()
                },
                headers: {
                    "Accept": "application/json",
                    "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                },
                success: function(response) {
                    if(response.code === 200){
                        showToast("success", response.message);
                        getBillingAddress();
                    }
                },
                error: function(res) {
                    if(res.responseJSON.code === 500){
                        showToast("error", res.responseJSON.message);
                    }
                }
            });
        });

        $(document).on("click", ".edit-address", function() {
            let id = $(this).data("id");
            $("#billingAddressForm").trigger("reset");
            $(".submitbtn").text(_l("save_changes"));
            $(".modal-title").text(_l("edit_billing_address"));

            $.ajax({
                type:"GET",
                url:"/customer/billing-address/edit/"+id,
                success: function(response) {
                    $(".error-text").text("");
                    $(".form-control, .select2-container").removeClass("is-invalid is-valid");
                    if(response.code === 200){
                        let data = response.data;

                        $("#id").val(data.id);
                        $("#name").val(data.name);
                        $("#email").val(data.email);
                        $("#address_line_one").val(data.address_line_one);
                        $("#address_line_two").val(data.address_line_two);
                        $("#country_id").val(data.country_id).trigger("change");
                        fetchStatesByCountry(data.country_id, data.state_id, data.city_id);
                        $("#zipcode").val(data.zipcode);
                        $("#is_default").prop("checked", data.is_default == 1 ? true : false);

                        const phoneNumber = data.phone_number ? data.phone_number.trim() : data.phone_number;
                        const phoneInput = document.querySelector(".phone_number");
                        const hiddenInput = document.querySelector("#international_phone_number");

                        if ($(phoneInput).data("itiInstance")) {
                            $(phoneInput).data("itiInstance").destroy();
                        }
                        const iti = intlTelInput(phoneInput, {
                            utilsScript: window.location.origin + "/backend/assets/plugins/intltelinput/js/utils.js",
                            separateDialCode: true,
                            placeholderNumberType: "",
                            autoPlaceholder: "off",
                            formatOnDisplay: false,
                        });
                        $(phoneInput).data("itiInstance", iti);

                        if (phoneNumber) {
                            iti.setNumber(phoneNumber);
                            hiddenInput.value = iti.getNumber();
                            initialPhoneNumber = phoneNumber;
                        }
                        const updateHiddenPhoneNumber = () => {
                            const currentPhoneNumber = iti.getNumber();
                            if (currentPhoneNumber !== initialPhoneNumber) {
                                hiddenInput.value = currentPhoneNumber.trim();
                            }
                        };

                        phoneInput.addEventListener("input", updateHiddenPhoneNumber);
                        phoneInput.addEventListener("countrychange", updateHiddenPhoneNumber);

                        if (!hiddenInput.value) {
                            hiddenInput.value = initialPhoneNumber;
                        }
                        $("#billing_address_modal").modal("show");
                    }
                }
            });
        });

        $(document).on("click", ".delete-address", function() {
            let id = $(this).data("id");
            $("#delete_id").val(id);
        });

        $(document).on("change", "#country_id", function () {
            const id = $(this).val();
            id ? fetchStatesByCountry(id) : (updateStateOptions(), updateCityOptions());
        });

        $(document).on("change", "#state_id", function () {
            const id = $(this).val();
            id ? fetchCitiesByState(id) : updateCityOptions();
        });
    }

    function updateStateOptions() {
        $("#state_id").empty().append(`<option value="">${_l("Select")}</option>`);
    }

    function updateCityOptions() {
        $("#city_id").empty().append(`<option value="">${_l("Select")}</option>`);
    }

    function fetchStatesByCountry(countryId, selectedStateId = null, selectedCityId = null) {
        $.get("/api/get-states", { country_id: countryId }, (res) => {
            if (res) {
                const $state = $("#state_id").empty().append(`<option value="">${_l("Select")}</option>`);

                $.each(res, (_, item) => {
                    $state.append(`<option value="${item.id}">${item.name}</option>`);
                });

                // set state if provided
                if (selectedStateId) {
                    $state.val(selectedStateId).trigger("change");
                    fetchCitiesByState(selectedStateId, selectedCityId);
                } else {
                    updateCityOptions();
                }
            }
        }).fail(() => showToast("error", "Failed to fetch states."));
    }

    function fetchCitiesByState(stateId, selectedCityId = null) {
        $.get("/api/get-city", { state_id: stateId }, (res) => {
            if (res) {
                const $city = $("#city_id").empty().append(`<option value="">${_l("Select")}</option>`);

                $.each(res, (_, item) => {
                    $city.append(`<option value="${item.id}">${item.name}</option>`);
                });

                // set city if provided
                if (selectedCityId) {
                    $city.val(selectedCityId).trigger("change");
                }
            }
        }).fail(() => showToast("error", "Failed to fetch cities."));
    }

})();