(async () => {
    "use strict";

    // Toggle view more/less categories
    if ($(".viewall-one").length > 0) {
        $(".viewall-one").hide();
        $(".viewall-button-one").on("click", function () {
            const isLess = $(this).text() === "Less Categories";
            $(this).text(isLess ? "More Categories" : "Less Categories");
            $(".viewall-one").slideToggle(900);
        });
    }

 $(document).ready(function () {
    const blogContainer = document.getElementById("blogListContainer");
    let activeFilters = {};

    // --- Main function to fetch and render blogs ---
    const fetchFilteredBlogs = (params = {}) => {
        activeFilters = { ...params };
        const query = new URLSearchParams(params).toString();

        fetch(`/blog?${query}`, {
            headers: {
                "X-Requested-With": "XMLHttpRequest"
            }
        })
        .then((response) => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.json();
        })
        .then((data) => {
            // **DEBUGGING STEP:** First, let's confirm the data is correct.
            console.log("Data received from server:", data); 

            if (data.html) {
                // Sanitize and update the container's HTML.
                blogContainer.innerHTML = DOMPurify.sanitize(data.html);

                // Re-initialize the Load More functionality on the new content.
                // This is the key part that needs to work with the newly injected HTML.
                initializeLoadMore(); 

                // Scroll to the top of the container for better UX.
                window.scrollTo({
                    top: blogContainer.offsetTop - 100,
                    behavior: "smooth"
                });
            } else {
                 console.error("Server response is missing the 'html' property.");
                 blogContainer.innerHTML = '<p class="text-center">No blogs found.</p>';
            }
        })
        .catch((error) => {
            console.error("Error fetching blogs:", error);
            // Optionally, show an error message to the user in the container.
            blogContainer.innerHTML = '<p class="text-center">Error loading blogs. Please try again.</p>';
        });
    };

    // --- "Load More" Functionality ---
    const initializeLoadMore = () => {
        const items = blogContainer.querySelectorAll(".blog-post-item");
        const loadMoreBtn = blogContainer.querySelector("#load-more-btn");
        
        // If there's no load more button in the new HTML, there's nothing to do.
        if (!loadMoreBtn) {
            // Ensure all items are visible if there's no "Load More" logic
            items.forEach(item => item.style.display = 'block');
            return;
        }

        const itemsPerLoad = 6;
        let visibleCount = 0;

        // Hide all items initially to apply the "load more" effect
        items.forEach(item => item.style.display = 'none');
        
        const showNextItems = () => {
            let newlyShown = 0;
            for (let i = visibleCount; i < items.length && newlyShown < itemsPerLoad; i++) {
                if (items[i]) {
                    items[i].style.display = "block";
                    newlyShown++;
                }
            }
            visibleCount += newlyShown;

            if (visibleCount >= items.length) {
                loadMoreBtn.style.display = "none";
            }
        };

        // Show the first set of items
        showNextItems();

        // Attach event listener to the button
        loadMoreBtn.addEventListener("click", showNextItems);
    };

    // --- Event Listener Bindings ---

    // **FIX 1: Use a more specific selector for category buttons.**
    // Your sidebar links should have the class "category-filter".
    const categoryButtons = document.querySelectorAll(".category-filter[data-category]");
    categoryButtons.forEach((button) => {
        button.addEventListener("click", function () {
            const category = this.dataset.category;
            fetchFilteredBlogs({ category: category, page: 1 }); // Reset to page 1 on new filter
        });
    });

    // Bind blog search
    const searchInput = document.getElementById("blogSearch");
    if (searchInput) {
        searchInput.addEventListener("keypress", function (e) {
            if (e.key === "Enter") {
                const search = this.value.trim();
                fetchFilteredBlogs({ search: search, page: 1 }); // Reset to page 1 on new search
            }
        });
    }
    
    // Initial load for the first time the page is visited.
    initializeLoadMore();
});
})();