$(document).ready(function() {
    /**
     * Helper function to display toast notifications.
     * @param {string} type - 'success', 'error', 'warning', 'info'
     * @param {string} message - The message to display.
     */
    function showToast(type, message) {
        toastr.options = {
            "closeButton": true,
            "progressBar": true,
            "positionClass": "toast-top-right",
            "timeOut": "5000" // 5 seconds
        };
        toastr[type](message);
    }

    // --- INITIALIZATION ---
    $('.searchable-select').select2({
        placeholder: "Please select an option",
        allowClear: true
    });

    const csrfToken = $('meta[name="csrf-token"]').attr('content');
    const form = $('.shop-form');

    // --- JQUERY VALIDATION PLUGIN SETUP ---
    if (form.length === 0) {
        return;
    }

    const validator = form.validate({
        ignore: ':hidden:not(#description)',
        rules: {
            shop_name: {
                required: true,
                minlength: 3,
                remote: {
                    url: "/provider/shops/check-name",
                    type: "post",
                    data: {
                        _token: csrfToken,
                        shop_name: function() {
                            return $("#shop_name").val();
                        },
                        id: function() {
                            return $('#shop_id').val();
                        }
                    }
                }
            },
            owner_name: {
                required: true,
                minlength: 3
            },
            phone: {
                required: true,
                digits: true,
                minlength: 7
            },
            email: {
                required: true,
                email: true
            },
            description: "required",
            tax_number: {
                required: () => $('input[name="tax_status"]:checked').val() === '1'
            },
            address: "required",
            country: "required",
            state: "required",
            city: "required",
            zip_code: {
                required: true,
                digits: true
            },
            shop_logo: {
                required: function(element) {
                    // The logo is only required on the create form, not the edit form.
                    return form.attr('id') === 'create-shop-form';
                }
            }
        },
        messages: {
            shop_name: {
                required: "The Shop Title field is required.",
                minlength: "Shop Title must be at least 3 characters.",
                remote: "This shop name is already taken."
            },
            owner_name: {
                required: "The Owner Name field is required.",
                minlength: "Owner Name must be at least 3 characters."
            },
            phone: "Please enter a valid mobile number.",
            email: "Please enter a valid email address.",
            description: "The Description is required.",
            tax_number: "The Tax Number is required.",
            address: "The Address Line is required.",
            country: "Country is required.",
            state: "State is required.",
            city: "City is required.",
            zip_code: "Please enter a valid postal code.",
            shop_logo: "The Shop Logo is required."
        },
        errorElement: 'span',
        errorClass: 'invalid-feedback',
        errorPlacement: function(error, element) {
            if (element.prop('type') === 'file') {
                 error.insertAfter(element.closest('.file-upload').next('.avatar-preview, .d-flex'));
            } else if (element.hasClass('searchable-select')) {
                 error.insertAfter(element.next('.select2-container'));
            } else if (element.attr('id') === 'description') {
                 error.insertAfter(element.next('.note-editor'));
            } else {
                 error.insertAfter(element);
            }
        },
        highlight: function(element) {
            let el = $(element);
            el.addClass('is-invalid');
            if (el.hasClass('searchable-select')) {
                 el.next('.select2-container').find('.select2-selection').addClass('is-invalid');
            }
            if (el.attr('id') === 'description') {
                 el.next('.note-editor').addClass('is-invalid');
            }
        },
        unhighlight: function(element) {
            let el = $(element);
            el.removeClass('is-invalid');
            if (el.hasClass('searchable-select')) {
                 el.next('.select2-container').find('.select2-selection').removeClass('is-invalid');
            }
            if (el.attr('id') === 'description') {
                 el.next('.note-editor').removeClass('is-invalid');
            }
        }
    });

    // --- SUMMERNOTE INITIALIZATION ---
    if ($.fn.summernote) {
        $('#description').summernote({
            height: 200,
            callbacks: {
                onChange: function(contents) {
                    $('#description').val(contents);
                    validator.element('#description');
                }
            }
        });
    }

    // --- WIZARD NAVIGATION ---
    const infoTab = $('#info-tab');
    const availTab = $('#avail-tab');
    const infoContent = $('#info-content');
    const availContent = $('#avail-content');
    const nextBtn = $('#next-btn');
    const prevBtn = $('#prev-btn');

    nextBtn.on('click', function(event) {
        event.stopImmediatePropagation();
        let isInfoTabValid = true;
        infoContent.find('input, select, textarea').each(function() {
            if ($(this).attr('name') && validator.settings.rules[$(this).attr('name')]) {
                if (!validator.element(this)) {
                    isInfoTabValid = false;
                }
            }
        });
        if (isInfoTabValid) {
            infoContent.removeClass('active');
            availContent.addClass('active');
            infoTab.removeClass('active');
            availTab.addClass('active');
        }
    });

    prevBtn.on('click', function() {
        availContent.removeClass('active');
        infoContent.addClass('active');
        availTab.removeClass('active');
        infoTab.addClass('active');
    });

    // --- DEPENDENT DROPDOWNS ---
    function fetchAndPopulate(url, body, selectElement, placeholder) {
        selectElement.html(`<option value="">${placeholder}</option>`).trigger('change');
        if ((body.country_id === undefined || body.country_id === "") && (body.state_id === undefined || body.state_id === "")) return;
        
        $.ajax({
            url: url,
            method: 'POST',
            data: JSON.stringify(body),
            contentType: 'application/json',
            headers: {
                'X-CSRF-TOKEN': csrfToken
            },
            success: function(data) {
                let options = data.states || data.cities || [];
                options.forEach(item => {
                    const option = new Option(item.name, item.id);
                    selectElement.append(option);
                });
                selectElement.trigger('change.select2');
            }
        });
    }
    $('#country').on('change', function() {
        fetchAndPopulate('/provider/shops/get-states', { country_id: $(this).val() }, $('#state'), 'Select State');
        $('#city').html('<option value="">Select City</option>').trigger('change.select2');
    });
    $('#state').on('change', function() {
        fetchAndPopulate('/provider/shops/get-cities', { state_id: $(this).val() }, $('#city'), 'Select City');
    });

    // --- UI LOGIC ---
    const taxNumberField = $('#tax-number-field');
    const taxNumberInput = taxNumberField.find('input');
    $('input[name="tax_status"]').on('change', function() {
        if ($(this).val() === '1') {
            taxNumberField.removeClass('d-none');
            taxNumberInput.prop('required', true);
        } else {
            taxNumberField.addClass('d-none');
            taxNumberInput.prop('required', false);
        }
    });

    // --- SHOP LOGO PREVIEW ---
    function readURL(input, imgElement) {
        if (input.files && input.files[0]) {
            var reader = new FileReader();
            reader.onload = function(e) {
                $(imgElement).attr('src', e.target.result);
                $(imgElement).parent().removeClass('d-none');
            };
            reader.readAsDataURL(input.files[0]);
        }
    }
    $("#shop-logo-upload").on('change', function() {
        readURL(this, '#shop-logo-preview-img');
        validator.element(this);
    });

    // --- GALLERY PREVIEW & DELETE LOGIC (WITH DIMENSION VALIDATION) ---
    const galleryInput = $("#gallery-upload");
    const previewContainer = $('#gallery-preview-container');
    const removedContainer = $('#removed-images-container');
    let galleryFiles = new DataTransfer();

    galleryInput.on('change', function() {
        const newFiles = this.files;
        const requiredWidth = 636;
        const requiredHeight = 409;
        let validationPromises = [];

        for (let i = 0; i < newFiles.length; i++) {
            const file = newFiles[i];

            const promise = new Promise((resolve, reject) => {
                const image = new Image();
                image.src = URL.createObjectURL(file);

                image.onload = function() {
                    if (this.naturalWidth === requiredWidth && this.naturalHeight === requiredHeight) {
                        resolve(file);
                    } else {
                        reject({
                            name: file.name,
                            message: `Upload Image in ${requiredWidth}x${requiredHeight} dimensions.`
                        });
                    }
                };
                image.onerror = function() {
                    reject({
                        name: file.name,
                        message: `Could not validate image: "${file.name}". It may be corrupt.`
                    });
                };
            });
            validationPromises.push(promise);
        }

        Promise.allSettled(validationPromises)
            .then(results => {
                results.forEach(result => {
                    if (result.status === 'fulfilled') {
                        galleryFiles.items.add(result.value);
                    } else {
                        showToast('error', result.reason.message);
                    }
                });
                galleryInput[0].files = galleryFiles.files;
                renderNewImagePreviews();
            });
    });

    previewContainer.on('click', '.remove-gallery-item-btn', function() {
        const parentItem = $(this).closest('.gallery-item');

        if (parentItem.hasClass('existing-image')) {
            const imagePath = parentItem.data('path');
            removedContainer.append(
                `<input type="hidden" name="removed_gallery_images[]" value="${imagePath}">`
            );
            parentItem.remove();
        } else {
            const indexToRemove = parentItem.data('index');
            const newFiles = new DataTransfer();
            const currentFiles = galleryFiles.files;

            for (let i = 0; i < currentFiles.length; i++) {
                if (i !== indexToRemove) {
                    newFiles.items.add(currentFiles[i]);
                }
            }
            galleryFiles = newFiles;
            galleryInput[0].files = galleryFiles.files;
            renderNewImagePreviews();
        }
    });

    function renderNewImagePreviews() {
        previewContainer.find('.new-image').remove();
        const filesToRender = galleryFiles.files;
        for (let i = 0; i < filesToRender.length; i++) {
            const file = filesToRender[i];
            const reader = new FileReader();
            reader.onload = function(e) {
                const previewHtml = `
                    <div class="gallery-item new-image" data-index="${i}">
                        <img src="${e.target.result}" alt="New Image" class="gallery-thumbnail rounded">
                        <button type="button" class="btn btn-sm btn-danger remove-gallery-item-btn">&times;</button>
                    </div>
                `;
                previewContainer.append(previewHtml);
            };
            reader.readAsDataURL(file);
        }
    }

    // --- AVAILABILITY SWITCHES ---
    $('#all-days-switch').on('change', function() {
        const isChecked = $(this).is(':checked');
        $('.day-switch').each(function() {
            if ($(this).is(':checked') !== isChecked) {
                $(this).prop('checked', isChecked).trigger('change');
            }
        });
    });

    $('.day-switch').on('change', function() {
        const day = $(this).data('day');
        const timeInputs = $(`.time-input[data-day="${day}"]`);
        const collapseTarget = $(`#time-collapse-${day}`);
        if ($(this).is(':checked')) {
            timeInputs.prop('disabled', false).prop('required', true);
            collapseTarget.collapse('show');
        } else {
            timeInputs.prop('disabled', true).prop('required', false);
            collapseTarget.collapse('hide');
        }
    });
});