(function($) {
    "use strict";

    $(document).ready(function() {
        let currentPage = 1;
        let hasMorePages = true;
        let isLoading = false;

        const defaultImage = "{{ asset('assets/img/default-image.png') }}";
        const ratingsContainer = $('#ratings-container');
        const loader = $('#loader');
        const noRatingsMessage = $('#no-ratings-message');
        const loadMoreContainer = $('#load-more-container');
        const loadMoreBtn = $('#load-more-btn');
        const replyModal = new bootstrap.Modal(document.getElementById('add-reply-modal'));
        const deleteModal = new bootstrap.Modal(document.getElementById('delete-rating-modal'));

        function generateStars(rating) {
            let stars = '';
            for (let i = 1; i <= 5; i++) {
                stars += `<span class="me-1"><i class="fa-${i <= rating ? 'solid text-warning' : 'regular'} fa-star"></i></span>`;
            }
            return stars;
        }

        function createRatingCard(rating) {
            const stars = generateStars(rating.rating);
            // First, handle if the image path is null from the database
            const profileImage = rating.customer.profile_image || defaultImage;

            const replyContent = rating.reply ?
                `<div class="card bg-light mb-0 mt-4">
                    <div class="card-body">
                        <h6 class="fw-semibold fs-16 mb-3">Your Reply</h6>
                        <p class="mb-0">${rating.reply.review}</p>
                    </div>
                </div>` : '';

            const actionButtons = `
                <div class="d-flex align-items-center gap-2">
                    ${!rating.reply ? `<a href="javascript:void(0);" class="btn btn-sm btn-white d-flex align-items-center reply-btn" data-rating-id="${rating.id}"><i class="isax isax-back-square me-1"></i>Reply</a>` : ''}
                    <button type="button" class="btn btn-sm btn-danger-light delete-btn" data-rating-id="${rating.id}"><i class="isax isax-trash"></i></button>
                </div>`;

            // **MODIFICATION HERE:**
            // The 'onerror' attribute is added to the <img> tag. If 'profileImage' fails to load,
            // 'onerror' will trigger and set the 'src' to our 'defaultImage'.
            return `
                <div class="card mb-4" id="rating-card-${rating.id}">
                    <div class="card-body">
                        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2 mb-2">
                            <h6 class="mb-0 fw-semibold fs-16">Product: ${rating.product.title}</h6>
                            <div class="d-flex align-items-center">${stars}<span>(${rating.rating.toFixed(1)})</span></div>
                        </div>
                        <p>${rating.review}</p>
                        <hr>
                        <div class="d-flex align-items-center justify-content-between flex-wrap gap-2">
                            <div class="d-flex align-items-center">
                                <img src="${profileImage}"
                                     onerror="this.onerror=null; this.src='${defaultImage}'"
                                     alt="user" class="avatar avatar-sm rounded-circle me-2">
                                <span class="fs-14 fw-medium">${rating.customer.name}</span>
                                <span class="text-muted mx-2">&bull;</span>
                                <span class="fs-12 text-muted">${rating.review_date}</span>
                            </div>
                            ${actionButtons}
                        </div>
                        ${replyContent}
                    </div>
                </div>`;
        }

        function loadRatings(page, append = false) {
            if (isLoading || !hasMorePages && append) return;
            isLoading = true;

            $.ajax({
                url: `/provider/product-ratings?page=${page}`,
                type: 'POST',
                headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
                beforeSend: () => {
                    loader.show();
                    loadMoreBtn.prop('disabled', true);
                    noRatingsMessage.hide();
                },
                success: (response) => {
                    if (response.status === 'success') {
                        if (!append) ratingsContainer.empty();

                        if(response.data.length > 0) {
                             response.data.forEach(rating => ratingsContainer.append(createRatingCard(rating)));
                        } else if (!append) {
                            noRatingsMessage.show();
                        }

                        hasMorePages = response.pagination.has_more_pages;
                        loadMoreContainer.toggle(hasMorePages);
                    }
                },
                error: (err) => {
                    toastr.error('Could not load reviews. Please try again.');
                    if (!append) noRatingsMessage.show();
                },
                complete: () => {
                    loader.hide();
                    loadMoreBtn.prop('disabled', false);
                    isLoading = false;
                }
            });
        }

        loadMoreBtn.on('click', function() {
            currentPage++;
            loadRatings(currentPage, true);
        });

        // Event Delegation for dynamic buttons
        $(document).on('click', '.reply-btn', function() {
            const ratingId = $(this).data('rating-id');
            $('#reply-parent-id').val(ratingId);
            $('#add-reply-form')[0].reset();
            $('#reply-text').removeClass('is-invalid');
            replyModal.show();
        });

        $(document).on('click', '.delete-btn', function() {
            const ratingId = $(this).data('rating-id');
            $('#delete-rating-id').val(ratingId);
            deleteModal.show();
        });

        $('#add-reply-form').on('submit', function(e) {
            e.preventDefault();
            const form = $(this);
            const button = form.find('button[type="submit"]');

            $.ajax({
                url: '/provider/product-ratings/reply',
                type: 'POST',
                data: form.serialize(),
                headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
                beforeSend: () => button.prop('disabled', true).text('Submitting...'),
                success: (response) => {
                    if (response.status === 'success') {
                        toastr.success(response.message);
                        replyModal.hide();
                        currentPage = 1;
                        loadRatings(currentPage, false);
                    }
                },
                error: (xhr) => {
                    const errorMsg = xhr.responseJSON?.message || 'Failed to submit reply.';
                    toastr.error(errorMsg);
                    if (xhr.status === 422) {
                        $('#reply-text').addClass('is-invalid');
                        $('#reply-error').text(xhr.responseJSON.errors.reply_text[0]);
                    }
                },
                complete: () => button.prop('disabled', false).text('Submit Reply')
            });
        });

        $('#delete-rating-form').on('submit', function(e) {
            e.preventDefault();
            const ratingId = $('#delete-rating-id').val();
            const button = $(this).find('button[type="submit"]');

            $.ajax({
                url: `/provider/product-ratings/${ratingId}`,
                type: 'DELETE',
                headers: {'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')},
                beforeSend: () => button.prop('disabled', true).text('Deleting...'),
                success: (response) => {
                    toastr.success(response.message);
                    deleteModal.hide();
                    $(`#rating-card-${ratingId}`).fadeOut(500, function() { $(this).remove(); });
                },
                error: (xhr) => toastr.error(xhr.responseJSON?.message || 'Failed to delete review.'),
                complete: () => button.prop('disabled', false).text('Delete Review')
            });
        });

        // Initial load
        loadRatings(currentPage);
    });

})(jQuery);