$(function () {
    // ===================================================================
    // === CORE FORM & VALIDATION LOGIC ================================
    // ===================================================================

    $("#product-form").validate({
        // Ignore hidden elements, except for the thumbnail file input which might be visually hidden
        ignore: ":hidden:not(#thumbnail-upload)",
        rules: {
            product_name: { required: true },
            slug: { required: true },
            product_code: { required: true },
            category_id: { required: true },
            brand_id: { required: true }, // Assuming you added a brand_id field
            description: { required: true },
            price: { required: true, number: true, min: 0 },
            stock: { required: true, digits: true, min: 0 },
            status: { required: true },
        },
        messages: {
            product_name: { required: "Product name is required." },
            slug: { required: "Slug is required." },
            product_code: { required: "Product code is required." },
            category_id: { required: "Please select a category." },
            brand_id: { required: "Please select a brand." },
            description: { required: "Description is required." },
            price: {
                required: "Price is required.",
                number: "Please enter a valid number.",
                min: "Price cannot be negative.",
            },
            stock: {
                required: "Stock quantity is required.",
                digits: "Please enter whole numbers only.",
                min: "Stock cannot be negative.",
            },
            status: { required: "Product status is required." },
        },
        errorElement: "span", // Use a span for errors
        errorPlacement: function (error, element) {
            // Sanitize the name by removing brackets
            const name = element.attr("name").replace(/\[\]/g, "");
            const errorId = "#" + name + "_error";

            $(errorId).html(error);
        },
        highlight: function (element, errorClass, validClass) {
            $(element)
                .addClass("border-red-500")
                .removeClass("border-slate-300");
        },
        unhighlight: function (element) {
            $(element)
                .removeClass("border-red-500")
                .addClass("border-slate-300");

            // Sanitize the name here as well
            const name = $(element).attr("name").replace(/\[\]/g, "");
            const errorId = "#" + name + "_error";

            $(errorId).empty(); // This will now work correctly
        },
        submitHandler: function (form) {
            const formData = new FormData(form);

            $.ajax({
                url: "/admin/product/store",
                method: "POST",
                data: formData,
                processData: false,
                contentType: false,
                headers: {
                    Accept: "application/json",
                    "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr(
                        "content"
                    ),
                },
                beforeSend: function () {
                    $("#submit-btn").prop("disabled", true).text("Saving...");
                },
                success: function (response) {
                    Swal.fire({
                        icon: "success",
                        title: "Product Saved!",
                        text: response.message,
                        timer: 2000,
                        showConfirmButton: false,
                    }).then(() => {
                        if (response.redirectUrl) {
                            window.location.href = response.redirectUrl;
                        }
                    });
                },
                error: function (jqXHR) {
                    let errorMessage =
                        "An unexpected error occurred. Please try again.";
                    if (jqXHR.status === 422) {
                        const errors = jqXHR.responseJSON.errors;
                        $('span[id$="_error"]').text("");
                        $.each(errors, function (key, value) {
                            // Handle array-based field names like variations[0][sizes][0]
                            const sanitizedKey = key
                                .replace(/\./g, "_")
                                .replace(/\[/g, "_")
                                .replace(/\]/g, "");
                            $("#" + sanitizedKey + "_error").text(value[0]);
                        });
                        errorMessage =
                            "Please correct the form errors and try again.";
                    }
                    Swal.fire({
                        icon: "error",
                        title: "Oops...",
                        text: errorMessage,
                    });
                },
                complete: function () {
                    $("#submit-btn")
                        .prop("disabled", false)
                        .text("Save Product");
                },
            });
        },
    });

    // --- Tab Navigation & Sectional Validation ---
    const tabs = $(".tab-button");
    const contents = $(".tab-content");
    const prevBtn = $("#prev-btn");
    const nextBtn = $("#next-btn");
    const submitBtn = $("#submit-btn");
    let currentTabIndex = 0;
    const totalTabs = tabs.length;

    function updateTab(targetIndex) {
        tabs.removeClass("border-indigo-500 text-indigo-600").addClass(
            "border-transparent text-gray-500"
        );
        tabs.eq(targetIndex)
            .addClass("border-indigo-500 text-indigo-600")
            .removeClass("border-transparent text-gray-500");
        contents.addClass("hidden");
        contents.eq(targetIndex).removeClass("hidden");
        prevBtn.toggleClass("hidden", targetIndex === 0);
        nextBtn.toggleClass("hidden", targetIndex === totalTabs - 1);
        submitBtn.toggleClass("hidden", targetIndex !== totalTabs - 1);
        currentTabIndex = targetIndex;
    }

    nextBtn.on("click", function () {
        const form = $("#product-form");
        const currentTabContent = contents.eq(currentTabIndex);

        const isVariationsTab =
            currentTabContent.attr("id") === "seventh-section";

        if (isVariationsTab || form.valid()) {
            if (currentTabIndex < totalTabs - 1) {
                updateTab(currentTabIndex + 1);
            }
        }
    });

    prevBtn.on(
        "click",
        () => currentTabIndex > 0 && updateTab(currentTabIndex - 1)
    );
    tabs.on("click", function () {
        updateTab($(this).index(".tab-button"));
    });
    updateTab(0); // Initialize first tab

    // ===================================================================
    // === HELPER SCRIPTS (Previews, Slug, etc.) =========================
    // ===================================================================

    // --- Image Previews ---
    $("#thumbnail-upload").on("change", function (event) {
        if (event.target.files && event.target.files[0]) {
            const reader = new FileReader();
            reader.onload = (e) => {
                $("#thumbnail-preview")
                    .attr("src", e.target.result)
                    .removeClass("hidden");
                $("#thumbnail-dropzone").find("svg, p").addClass("hidden");
            };
            reader.readAsDataURL(event.target.files[0]);
        }
    });

    $("#multi-image-upload").on("change", function () {
        const previewContainer = $("#multi-image-preview-container").empty();
        if (this.files) {
            $.each(this.files, (i, file) => {
                previewContainer.append(
                    $("<img>", {
                        src: URL.createObjectURL(file),
                        class: "h-24 w-24 rounded-md object-cover border",
                    })
                );
            });
        }
    });

    // --- Automatic Slug Generation ---
    $("#product_name").on("input", function () {
        const slug = $(this)
            .val()
            .toLowerCase()
            .trim()
            .replace(/\s+/g, "-")
            .replace(/[^\w\-]+/g, "")
            .replace(/\-\-+/g, "-");
        $("#slug").val(slug);
    });

    // ===================================================================
    // === DYNAMIC PRODUCT VARIATIONS LOGIC ==============================
    // ===================================================================
    const variationsContainer = $("#variations-container");
    let variationIndex = 0;

    $("#add-variation-btn").on("click", function () {
        variationIndex++;
        const template = `
            <div class="variation-card bg-white border border-slate-200 p-6 rounded-lg relative" id="variation-${variationIndex}">
                <button type="button" class="remove-variation-btn absolute top-4 right-4 text-slate-400 hover:text-red-600 font-bold text-xl">&times;</button>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-x-8 gap-y-6">
                    <div class="md:col-span-2">
                        <label class="block text-sm font-medium text-slate-700 mb-2">Variation Images</label>
                        <div class="image-preview-container flex flex-wrap gap-3 p-2 border rounded-md bg-slate-50 min-h-[96px] w-full">
                            <span class="placeholder-text text-slate-400 text-sm self-center ml-2">Image previews here</span>
                        </div>
                        <div class="mt-4">
                            <label for="file-input-${variationIndex}" class="cursor-pointer text-sm font-semibold text-indigo-600 bg-indigo-50 px-4 py-2 rounded-md border border-indigo-200 hover:bg-indigo-100">Choose files</label>
                            <input type="file" id="file-input-${variationIndex}" name="variations[${variationIndex}][images][]" class="variation-image-input hidden" multiple/>
                        </div>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-slate-700 mb-2">Color</label>
                        <input type="color" name="variations[${variationIndex}][color]" value="#000000" class="h-10 w-24 cursor-pointer p-1 border rounded">
                    </div>
                </div>
                <hr class="my-6">
                <div>
                    <h4 class="font-semibold text-slate-800 mb-3">Sizes, Prices & Stock</h4>
                    <div class="size-price-container space-y-3"></div>
                    <button type="button" class="add-size-btn mt-4 text-sm font-semibold text-indigo-600 hover:text-indigo-800">+ Add Size</button>
                </div>
            </div>`;
        $(template).appendTo(variationsContainer).find(".add-size-btn").click();
    });

    // --- Event Delegation for Dynamic Elements ---
    variationsContainer.on("click", ".remove-variation-btn", function () {
        $(this).closest(".variation-card").remove();
    });
    variationsContainer.on("click", ".remove-size-btn", function () {
        $(this).closest(".size-price-row").remove();
    });

    variationsContainer.on("change", ".variation-image-input", function () {
        const previewContainer = $(this)
            .closest(".variation-card")
            .find(".image-preview-container")
            .empty();
        if (this.files && this.files.length > 0) {
            $.each(this.files, (i, file) => {
                const reader = new FileReader();
                reader.onload = (e) =>
                    previewContainer.append(
                        $("<img>", {
                            src: e.target.result,
                            class: "h-20 w-20 rounded-md object-cover border",
                        })
                    );
                reader.readAsDataURL(file);
            });
        }
    });

    variationsContainer.on("click", ".add-size-btn", function () {
        const container = $(this).prev(".size-price-container");
        const vIndex = $(this)
            .closest(".variation-card")
            .attr("id")
            .split("-")[1];
        const sizeTemplate = `
            <div class="size-price-row grid grid-cols-1 sm:grid-cols-4 items-center gap-3">
                <input type="text" name="variations[${vIndex}][sizes][]" class="sm:col-span-1 p-2 border border-slate-300 rounded-md" placeholder="e.g., M">
                <input type="number" name="variations[${vIndex}][prices][]" class="sm:col-span-1 p-2 border border-slate-300 rounded-md" placeholder="Price">
                <input type="number" name="variations[${vIndex}][stocks][]" class="sm:col-span-1 p-2 border border-slate-300 rounded-md" placeholder="Stock">
                <div class="sm:col-span-1 text-right">
                    <button type="button" class="remove-size-btn text-slate-400 hover:text-red-600 font-bold text-lg">&times;</button>
                </div>
            </div>`;
        container.append(sizeTemplate);
    });
});