$(document).ready(function() {
    // --- INITIALIZATION ---
    if ($.fn.select2) {
        // FIX: Target a more specific class to avoid conflicts with other scripts.
        $('.shop-select').select2({
            minimumResultsForSearch: Infinity,
            width: '100%'
        });
    }

    const csrfToken = $('meta[name="csrf-token"]').attr('content');
    const form = $('.shop-form');

    // --- JQUERY VALIDATION PLUGIN SETUP ---
    if (form.length === 0) {
        return;
    }

    const validator = form.validate({
        ignore: ':hidden:not(#description)',
        rules: {
            shop_name: {
                required: true,
                minlength: 3,
                remote: {
                    url: "/provider/shops/check-name",
                    type: "post",
                    data: {
                        _token: csrfToken,
                        shop_name: function() {
                            return $("#shop_name").val();
                        },
                        id: function() {
                            return $('#shop_id').val();
                        }
                    }
                }
            },
            owner_name: {
                required: true,
                minlength: 3
            },
            phone: {
                required: true,
                digits: true,
                minlength: 7
            },
            email: {
                required: true,
                email: true
            },
            description: "required",
            tax_number: {
                required: () => $('input[name="tax_status"]:checked').val() === '1'
            },
            address: "required",
            country: "required",
            state: "required",
            city: "required",
            zip_code: {
                required: true,
                digits: true
            },
            // FIX: Make the shop_logo conditionally required
            shop_logo: {
                required: function(element) {
                    // The logo is only required on the create form, not the edit form.
                    return form.attr('id') === 'create-shop-form';
                }
            }
        },
        messages: {
            shop_name: {
                required: "The Shop Title field is required.",
                minlength: "Shop Title must be at least 3 characters.",
                remote: "This shop name is already taken."
            },
            owner_name: {
                required: "The Owner Name field is required.",
                minlength: "Owner Name must be at least 3 characters."
            },
            phone: "Please enter a valid mobile number.",
            email: "Please enter a valid email address.",
            description: "The Description is required.",
            tax_number: "The Tax Number is required.",
            address: "The Address Line is required.",
            country: "Country is required.",
            state: "State is required.",
            city: "City is required.",
            zip_code: "Please enter a valid postal code.",
            shop_logo: "The Shop Logo is required."
        },
        errorElement: 'span',
        errorClass: 'invalid-feedback',
        errorPlacement: function(error, element) {
            if (element.prop('type') === 'file') {
                 error.insertAfter(element.closest('.card-body').find('.avatar-preview').next('.invalid-feedback'));
            } else if (element.hasClass('shop-select')) {
                error.insertAfter(element.next('.select2-container'));
            } else if (element.attr('id') === 'description') {
                error.insertAfter(element.next('.note-editor'));
            } else {
                error.insertAfter(element);
            }
        },
        highlight: function(element) {
            let el = $(element);
            el.addClass('is-invalid');
            if (el.hasClass('shop-select')) {
                el.next('.select2-container').addClass('is-invalid');
            }
            if (el.attr('id') === 'description') {
                el.next('.note-editor').addClass('is-invalid');
            }
        },
        unhighlight: function(element) {
            let el = $(element);
            el.removeClass('is-invalid');
            if (el.hasClass('shop-select')) {
                el.next('.select2-container').removeClass('is-invalid');
            }
            if (el.attr('id') === 'description') {
                el.next('.note-editor').removeClass('is-invalid');
            }
        }
    });

    // --- SUMMERNOTE INITIALIZATION ---
    if ($.fn.summernote) {
        $('#description').summernote({
            height: 200,
            callbacks: {
                onChange: function(contents) {
                    $('#description').val(contents);
                    validator.element('#description');
                }
            }
        });
    }

    // --- WIZARD NAVIGATION ---
    const infoTab = $('#info-tab');
    const availTab = $('#avail-tab');
    const infoContent = $('#info-content');
    const availContent = $('#avail-content');
    const nextBtn = $('#next-btn');
    const prevBtn = $('#prev-btn');

    nextBtn.on('click', function(event) {
        event.stopImmediatePropagation();
        let isInfoTabValid = true;
        infoContent.find('input, select, textarea').each(function() {
            // Only validate elements that have rules defined
            if ($(this).attr('name') && validator.settings.rules[$(this).attr('name')]) {
                if (!validator.element(this)) {
                    isInfoTabValid = false;
                }
            }
        });
        if (isInfoTabValid) {
            infoContent.removeClass('active');
            availContent.addClass('active');
            infoTab.removeClass('active');
            availTab.addClass('active');
        }
    });

    prevBtn.on('click', function() {
        availContent.removeClass('active');
        infoContent.addClass('active');
        availTab.removeClass('active');
        infoTab.addClass('active');
    });

    // --- DEPENDENT DROPDOWNS ---
    function fetchAndPopulate(url, body, selectElement, placeholder) {
        selectElement.html(`<option value="">${placeholder}</option>`).trigger('change');
        if (!body.country_id && !body.state_id) return;
        $.ajax({
            url: url, method: 'POST', data: JSON.stringify(body), contentType: 'application/json', headers: { 'X-CSRF-TOKEN': csrfToken },
            success: function(data) {
                let options = data.states || data.cities || [];
                options.forEach(item => {
                    const option = new Option(item.name, item.id);
                    selectElement.append(option);
                });
                selectElement.trigger('change.select2');
            }
        });
    }
    $('#country').on('change', function() { fetchAndPopulate('/provider/shops/get-states', { country_id: $(this).val() }, $('#state'), 'Select State'); $('#city').html('<option value="">Select City</option>').trigger('change.select2'); });
    $('#state').on('change', function() { fetchAndPopulate('/provider/shops/get-cities', { state_id: $(this).val() }, $('#city'), 'Select City'); });

    // --- UI LOGIC ---
    const taxNumberField = $('#tax-number-field');
    const taxNumberInput = taxNumberField.find('input');
    $('input[name="tax_status"]').on('change', function() {
        if ($(this).val() === '1') {
            taxNumberField.show();
            taxNumberInput.prop('required', true);
        } else {
            taxNumberField.hide();
            taxNumberInput.prop('required', false);
        }
    });
    function readURL(input, imgElement) {
        if (input.files && input.files[0]) {
            var reader = new FileReader();
            reader.onload = function(e) {
                $(imgElement).attr('src', e.target.result);
            };
            reader.readAsDataURL(input.files[0]);
        }
    }
    $("#shop-logo-upload").on('change', function() {
        $('#shop-logo-preview-img').parent().show();
        readURL(this, '#shop-logo-preview-img');
        validator.element(this);
    });
    $("#gallery-upload").on('change', function() {
        const container = $('#gallery-preview-container'); container.empty();
        if (this.files) {
            Array.from(this.files).forEach(file => {
                const reader = new FileReader();
                reader.onload = function(e) { container.append(`<div class="gallery-upload-img mb-2"><img src="${e.target.result}" alt="Img" class="gallery-thumbnail rounded"></div>`); };
                reader.readAsDataURL(file);
            });
        }
    });
    $('#all-days-switch').on('change', function() {
        const isChecked = $(this).is(':checked');
        $('.day-switch').each(function() {
            if ($(this).is(':checked') !== isChecked) { $(this).prop('checked', isChecked).trigger('change'); }
        });
    });
    $('.day-switch').on('change', function() {
        const day = $(this).data('day');
        const timeInputs = $(`.time-input[data-day="${day}"]`);
        const collapseTarget = $(`#time-collapse-${day}`);
        if ($(this).is(':checked')) {
            timeInputs.prop('disabled', false).prop('required', true);
            collapseTarget.collapse('show');
        } else {
            timeInputs.prop('disabled', true).prop('required', false);
            collapseTarget.collapse('hide');
        }
    });
});