(async function () {
    "use strict";
    await loadTranslationFile();

    const csrfToken = $('meta[name="csrf-token"]').attr("content");
    const currencySymbol = $("body").data('currency');
    const formatCurrency = (value) => {
        return currencySymbol + parseFloat(value).toFixed(2);
    };
    let is_billing_empty = $('.billing_address_container').data('is_empty');
    let totalTax = 0;
    let taxDetails = [];
    let couponDetails = [];
    let walletAmount = $("#checkout-order-summary").data("wallet_amount");

    function updateCartTotals() {
        let subtotal = 0;
        $(".cart-product-row").each(function () {
            const price = parseFloat($(this).data("price"));
            const quantity = parseInt($(this).find(".quantity-input").val());

            if (!isNaN(price) && !isNaN(quantity)) {
                const rowSubtotal = price * quantity;
                subtotal += rowSubtotal;
                $(this).find(".row-subtotal").text(formatCurrency(rowSubtotal)).attr("data-subtotal", rowSubtotal);
            }
        });

        $("#cart-subtotal").text(formatCurrency(subtotal));
        $("#cart-total").text(formatCurrency(subtotal));
        $("#checkout-subtotal").text(formatCurrency(subtotal));
        $("#checkout-total").text(formatCurrency(subtotal));

        if ($(".cart-product-row").length === 0) {
            $("#proceedCheckout").addClass("disabled");
            $("#empty-cart-btn, #update-cart-btn").attr("disabled", true);
            const emptyHtml = `<tr><td colspan="5" class="text-center py-4">${_l('cart_empty_info')}</td></tr>`;
            $("#cart-wrap tbody").html(emptyHtml);
        } else {
            $("#proceedCheckout").removeClass("disabled");
        }
    }

    // --- AJAX EVENT HANDLERS FOR CART MANAGEMENT ---

    // 1. UPDATE CART (saves all quantity changes)
    $("#update-cart-btn").on("click", function () {
        let quantities = {};
        $(".quantity-input").each(function () {
            quantities[$(this).data("cart-key")] = $(this).val();
        });

        $.ajax({
            url: "/user/cart/update",
            type: "POST",
            data: {
                _token: csrfToken,
                quantities: quantities,
            },
            success: function (response) {
                if (response.success) {
                    $(".cart-item-count").text(response.cartCount);
                    showToast("success", response.message);
                }
            },
            error: function () {
                showToast("error", _l('cart_update_failed'));
            },
        });
    });

    let cartKeyToRemove = null;
    let rowToRemove = null;

    $(document).on("click", ".delete-item", function () {
        cartKeyToRemove = $(this).data("cart-key");
        rowToRemove = $(this).closest(".cart-product-row");
        $("#remove_cart_modal").modal("show");
    });

    // 2. Confirm delete inside modal
    $(".delete-form").on("submit", function (e) {
        e.preventDefault();

        if (!cartKeyToRemove) return;

        $.ajax({
            url: "/user/cart/remove",
            type: "POST",
            data: {
                _token: csrfToken,
                cart_key: cartKeyToRemove,
            },
            success: function (response) {
                if (response.success) {
                    rowToRemove.fadeOut(300, function () {
                        $(this).remove();
                        updateCartTotals();
                        $(".cart-item-count").text(response.cartCount);
                        showToast("success", response.message);
                    });
                }
                $("#remove_cart_modal").modal("hide");
            },
            error: function () {
                showToast("error", _l('cart_item_remove_failed'));
            },
        });
    });

    // 1. Open modal when clicking "Empty Cart"
    $("#empty-cart-btn").on("click", function () {
        if ($(".cart-product-row").length === 0) return;
        $("#remove_all_cart_modal").modal("show");
    });

    // 2. Handle modal confirm submit
    $("#remove_all_cart_modal .delete-form").on("submit", function (e) {
        e.preventDefault();

        $.ajax({
            url: "/user/cart/empty",
            type: "POST",
            data: { _token: csrfToken },
            success: function (response) {
                if (response.success) {
                    $(".cart-product-row").fadeOut(300, function () {
                        $(this).remove();
                    });
                    setTimeout(() => updateCartTotals(), 350);
                    $(".cart-item-count").text(response.cartCount);
                    showToast("success", response.message);
                }
                $("#remove_all_cart_modal").modal("hide");
            },
            error: function () {
                showToast("error", _l('cart_item_empty_failed'));
                $("#remove_all_cart_modal").modal("hide");
            },
        });
    });

    // --- CLIENT-SIDE UI HANDLERS (for immediate feedback) ---

    $("#cart-wrap").on("click", ".add-btn", function () {
        const input = $(this).siblings(".quantity-input");
        input.val(parseInt(input.val()) + 1).trigger("keyup");
    });

    $("#cart-wrap").on("click", ".minus-btn", function () {
        const input = $(this).siblings(".quantity-input");
        let currentVal = parseInt(input.val());
        if (currentVal > 1) {
            input.val(currentVal - 1).trigger("keyup");
        }
    });

    // Recalculate totals whenever a quantity input changes
    $("#cart-wrap").on("keyup", ".quantity-input", function () {
        if (parseInt($(this).val()) < 1) {
            $(this).val(1);
        }
        updateCartTotals();
    });

    // --- CHECKOUT FLOW & PAYMENT SUBMISSION ---

    $("#proceedCheckout").on("click", function (e) {
        e.preventDefault();
        if ($(this).hasClass("disabled")) return;

        $("#proceedCheckout").attr("disabled", true).html(`
            <span class="spinner-border spinner-border-sm align-middle" role="status" aria-hidden="true"></span> ${_l("processing")}
        `);

        updateCheckoutSummary();
    });

    $("#back_to_cart").on("click", function (e) {
        e.preventDefault();
        if ($(this).hasClass("disabled")) return;

        $("#checkout-section").fadeOut(300, () =>
            $("#cart-section").fadeIn(300)
        );
    });

    // This is your original function, now included
    function updateCheckoutSummary(couponData = [], is_remove_coupon = false) {
        const summaryContainer = $("#checkout-order-summary");
        summaryContainer.empty();

        let subtotal = 0;

        $(".cart-product-row").each(function () {
            const productName = $(this).find(".product-name").text();
            const quantity = $(this).find(".quantity-input").val();
            const subtotalText = $(this).find(".row-subtotal").text();
            const itemHtml = `<div class="d-flex align-items-center justify-content-between mb-3"><p class="mb-0 text-body fs-14">${productName} X ${quantity}</p><p class="fw-semibold mb-0 fs-14">${subtotalText}</p></div>`;
            summaryContainer.append(itemHtml);

            const price = parseFloat($(this).data("price"));
            const quantityVal = parseInt($(this).find(".quantity-input").val());

            if (!isNaN(price) && !isNaN(quantityVal)) {
                const rowSubtotal = price * quantityVal;
                subtotal += rowSubtotal;
            }
        });

        const finalSubtotal = $("#cart-subtotal").text();
        $("#checkout-subtotal").text(finalSubtotal);

        $.ajax({
            url: "/user/product/calculate-summary",
            method: "POST",
            data: {
                subtotal: subtotal,
                couponData: couponData,
            },
            headers: {
                Accept: "application/json",
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            success: function (response) {
                if (response.success) {
                    let data = response.data;
                    totalTax = data.total_tax;
                    taxDetails = data.taxes;
                    couponDetails = data.coupon_details;
                    let taxes = data.taxes;
                    let taxHtml = "";
                    $("#checkout-discount-container").empty();
                    $("#checkout-total").text(data.grand_total);

                    if (data.grand_total >= walletAmount) {
                        $("#payment-wallet").attr("disabled", true);
                        $(".wallet_message").text( "- " + _l("insufficient_wallet_balance"));
                    }

                    if (data.total_coupon_discount_amount > 0) {
                        taxHtml += `
                            <div class="d-flex align-items-center justify-content-between mb-3">
                                <p class="mb-0 text-body fs-14">${_l('Coupon')}</p>
                                <p class="fw-semibold mb-0 fs-14">- ${currencySymbol}${data.total_coupon_discount_amount}</p>
                            </div>
                        `;
                    }

                    $("#total_coupon_discount_amount").val(
                        data.total_coupon_discount_amount
                    );

                    if (taxes.length > 0) {
                        taxes.forEach((tax) => {
                            taxHtml += `
                                <div class="d-flex align-items-center justify-content-between mb-3">
                                    <p class="mb-0 text-body fs-14">Tax (${tax.name} ${tax.rate}%)</p>
                                    <p class="fw-semibold mb-0 fs-14">${currencySymbol}${tax.amount}</p>
                                </div>
                            `;
                        });
                    }
                    $("#checkout-discount-container").html(taxHtml);

                    if (is_remove_coupon) {
                        $("#coupon_remove_btn").attr("disabled", false).html(_l('remove'));
                        $("#coupon_remove_btn").addClass("d-none");
                        $("#coupon_btn").removeClass("d-none");
                        $("#coupon_code").attr("readonly", false);
                        $("#coupon_code").val("");
                    } else {
                        $("#cart-section").fadeOut(300, () =>
                            $("#checkout-section").fadeIn(300)
                        );
                    }
                }
            },
            error: function (xhr) {
                showToast('error', xhr.responseJSON.message);
            },
            complete: function () {
                $("#proceedCheckout").attr("disabled", false).html(_l('proceed_to_checkout'));
            }
        });
    }

    $(document).on("click", "#coupon_btn", function () {
        let coupon_code = $("#coupon_code").val();
        let productData = [];

        $(".cart-product-row").each(function () {
            if ($(this).data("product-id")) {
                let productId = $(this).data("product-id");
                let productPrice = parseFloat(
                    $(this).find(".row-subtotal").text().replace(currencySymbol, "")
                );
                productData.push({
                    product_id: productId,
                    price: productPrice
                });
            }
        });

        if (!coupon_code) {
            showToast("error", _l('please_enter_coupon_code'));
            return;
        }

        $.ajax({
            url: "/product/check-coupon",
            type: "POST",
            data: {
                coupon_code: coupon_code,
                products: productData
            },
            headers: {
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            beforeSend: function () {
                $("#coupon_btn").attr("disabled", true).html(`
                    <span class="spinner-border spinner-border-sm align-middle" role="status" aria-hidden="true"></span>
                `);
            },
            success: function (response) {
                if (response.success) {
                    showToast("success", response.message);
                    $("#coupon_btn").addClass("d-none");
                    $("#coupon_btn").attr("disabled", false).html(_l('apply'));
                    $("#coupon_remove_btn").removeClass("d-none");
                    $("#coupon_code").attr("readonly", true);
                    updateCheckoutSummary(response.data);
                } else {
                    showToast("error", response.message);
                }
            },
            error: function (xhr) {
                let message = _l('coupon_apply_error');
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message = xhr.responseJSON.message;
                } else if (xhr.responseText) {
                    message = xhr.responseText;
                }
                showToast('error', message);
            },
            complete: function () {
                $("#coupon_btn").attr("disabled", false).html(_l('apply'));
                $("coupon_remove_btn").attr("disabled", false).html(_l('remove'));
            }
        });
    });

    $(document).on("click", "#coupon_remove_btn", function () {
        $("#coupon_remove_btn").attr("disabled", true).html(`
            <span class="spinner-border spinner-border-sm align-middle" role="status" aria-hidden="true"></span>
        `);
        updateCheckoutSummary([], true);
    });

    // The main "Pay Now" button is now a router
    $("#confirmPaymentBtn").on("click", function (e) {
        e.preventDefault();
        const paymentType = $('input[name="payment_method"]:checked').val();
        const payButton = $(this);

        if (is_billing_empty) {
            showToast("error", _l('please_add_billing_address'));
            return;
        }

        if ($(".billing_address_container .order-card.active").length === 0) {
            showToast("error", _l('please_select_billing_address'));
            return;
        }

        // Validate a payment method is selected
        if (!paymentType) {
            showToast("error", _l('please_select_payment_method'));
            return;
        }

        // If 'bank' is selected, open the modal instead of submitting
        if (paymentType === "bank") {
            // Show the modal using Bootstrap 5's Javascript API
            var bankModal = new bootstrap.Modal(
                document.getElementById("bankTransferModal")
            );
            bankModal.show();
        } else {
            // For any other payment method (COD, PayPal, etc.), proceed with the original AJAX call
            submitPayment(payButton);
        }
    });

    // New handler for the bank transfer modal form submission
    $("#bankTransfer-form").on("submit", function (e) {
        e.preventDefault();
        const payButton = $("#submit-bank-payment"); // The button inside the modal

        // Check if a file is selected
        if ($("#payment_receipt")[0].files.length === 0) {
            showToast("error", _l('payment_proof_required'));
            return;
        }

        // Submit the payment, but this time with the file data
        submitPayment(payButton, new FormData(this));
    });

    // We move the AJAX logic into a reusable function
    function submitPayment(payButton, formData = null) {
        // 1. Gather all the data from the page (same as your previous code)
        let products = [];
        $(".cart-product-row").each(function () {
            const cartKey = $(this).data("cart-key");
            const [id, color, size] = cartKey.split("-");

            products.push({
                id: id, // product ID
                color: color, // hex color code
                size: size, // product size
                name: $(this).find(".product-name").data('product_name'),
                quantity: $(this).find(".quantity-input").val(),
                price: $(this).data("price"),
                subtotal: parseFloat(
                    $(this)
                        .find(".row-subtotal")
                        .text()
                        .replace(currencySymbol, "")
                ),
            });
        });

        const subTotal = parseFloat(
            $("#checkout-subtotal").text().replace(currencySymbol, "")
        );
        const total = parseFloat(
            $("#checkout-total").text().replace(currencySymbol, "")
        );
        const activeAddressCard = $(".order-card.active");
        const address = {
            address_type: activeAddressCard.data("address-type"),
            email: activeAddressCard.data("email"),
            phone: activeAddressCard.data("phone"),
            address: activeAddressCard.data("address"),
            city_name: activeAddressCard.data("city_name"),
            state_name: activeAddressCard.data("state_name"),
            country_name: activeAddressCard.data("country_name"),
            zipcode: activeAddressCard.data("zipcode"),
        };
        const paymentType = $('input[name="payment_method"]:checked').val();

        // 2. Prepare the data for submission
        let ajaxData;
        let ajaxSettings = {
            url: "/user/product/payment",
            type: "POST",
            headers: { "X-CSRF-TOKEN": csrfToken },
        };

        if (formData) {
            // This is a Bank Transfer with a file upload
            // Append all other data to the FormData object
            formData.append("products", JSON.stringify(products));
            formData.append("sub_total", subTotal);
            formData.append("total_amount", total);
            formData.append("billing_address", JSON.stringify(address));
            formData.append("payment_type", paymentType);
            formData.append("total_tax_amount", totalTax);
            formData.append("tax_details", JSON.stringify(taxDetails));
            formData.append("coupon_details", JSON.stringify(couponDetails));
            formData.append("total_coupon_discount_amount", $("#total_coupon_discount_amount").val());

            ajaxData = formData;
            // These settings are CRUCIAL for file uploads
            ajaxSettings.processData = false;
            ajaxSettings.contentType = false;
        } else {
            // This is for other payment types (COD, PayPal)
            const requestData = {
                products: products,
                sub_total: subTotal,
                total_amount: total,
                billing_address: address,
                payment_type: paymentType,
                total_tax_amount: totalTax,
                tax_details: taxDetails,
                coupon_details: couponDetails,
                total_coupon_discount_amount: $("#total_coupon_discount_amount").val(),
            };
            ajaxData = JSON.stringify(requestData);
            ajaxSettings.contentType = "application/json";
        }

        // 3. Perform the AJAX call
        $.ajax({
            ...ajaxSettings, // Spread the settings object
            data: ajaxData,
            beforeSend: function () {
                payButton.prop("disabled", true).text(_l('processing'));
            },
            success: function (response) {
                // Close the modal if it's open
                var bankModalEl = document.getElementById("bankTransferModal");
                var bankModal = bootstrap.Modal.getInstance(bankModalEl);
                if (bankModal) {
                    bankModal.hide();
                }

                if (response.code === 200) {
                    if (response.stripurl) {
                        window.location.href = response.stripurl;
                    }

                    if (response.paypal_url) {
                        window.location.href = response.paypal_url;
                    } else if (response.data && response.data.order_id) {
                        window.location.href =
                            "/product/booking-success?order_id=" +
                            response.data.order_id;
                    }
                }
            },
            error: function (jqXHR, textStatus, errorThrown) {
                showToast("error", "An error occurred. Please try again.");
                payButton.prop("disabled", false).text("Pay Now");
            },
        });
    }


    let initialPhoneNumber = null;
    let international_phone_number = null;
    let billing_address_count = 0;
    $(document).ready(function () {
        initInternationalPhoneInput();
        initSelect();
        initValidation();
        initEvents();
    });

    document.querySelectorAll('input[name="billing_address"]').forEach((radio) => {
        radio.addEventListener('change', function () {
            document.querySelectorAll('.order-card').forEach(card => card.classList.remove('active'));
            if (this.checked) {
                this.closest('.order-card').classList.add('active');
            }
        });
    });

    function getBillingAddress() {
        $.ajax({
            type: "GET",
            url: "/customer/billing-address/list",
            headers: {
                Accept: "application/json",
                "X-CSRF-TOKEN": $('meta[name="csrf-token"]').attr("content"),
            },
            success: function (response) {
                let addressData = "";
                if (response.data && response.data.length > 0) {
                    is_billing_empty = false;
                    billing_address_count = response.billing_address_count ?? 0;

                    response.data.forEach((data) => {
                        let is_default = data.is_default == 1 ? "checked" : "";

                        addressData += `
                            <div class="col-lg-6 col-md-6 d-flex">
                                <div class="card mb-0 order-card flex-fill ${is_default ? 'active' : ''}"
                                    data-address-type="${data.name ?? ''}"
                                    data-email="${data.email ?? ''}"
                                    data-phone="${data.phone_number ?? ''}"
                                    data-address="${(data.address_line_one ?? '') + ', ' +
                                                    (data.address_line_two ?? '') + ', ' +
                                                    (data.city?.name ?? '') + ', ' +
                                                    (data.state?.name ?? '') + ', ' +
                                                    (data.country?.name ?? '') + ', ' +
                                                    (data.zipcode ?? '')}"
                                    data-city_name="${data.city?.name ?? ''}"
                                    data-state_name="${data.state?.name ?? ''}"
                                    data-country_name="${data.country?.name ?? ''}">
                                    <div class="card-body">
                                        <div class="d-flex align-items-center justify-content-between gap-2 mb-3 pb-3 border-bottom">
                                            <h6 class="mb-0">${data.name ?? ''}</h6>
                                            <div class="form-check">
                                                <input type="radio" name="billing_address" class="form-check-input" value="${data.id}" ${is_default}>
                                            </div>
                                        </div>
                                        <div class="mb-3">
                                            <p class="mb-2 fw-semibold fs-14">${_l("address_type")} <span class="fw-normal">: ${data.name ?? ''}</span></p>
                                            <p class="mb-2 fw-semibold fs-14">${_l("email")} <span class="fw-normal">: ${data.email ?? ''}</span></p>
                                            <p class="mb-2 fw-semibold fs-14">${_l("phone")} <span class="fw-normal">: ${data.phone_number ?? ''}</span></p>
                                            <p class="mb-2 fw-semibold fs-14">${_l("address")}
                                                <span class="fw-normal">:
                                                    ${(data.address_line_one ?? '') + ', ' +
                                                    (data.address_line_two ?? '') + ', ' +
                                                    (data.city?.name ?? '') + ', ' +
                                                    (data.state?.name ?? '') + ', ' +
                                                    (data.country?.name ?? '') + ', ' +
                                                    (data.zipcode ?? '')}
                                                </span>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        `;
                    });
                } else {
                    is_billing_empty = true;
                    addressData = `<p class="mb-0">${_l("no_billing_address_found")}</p>`;
                }

                $(".billing_address_container").html(addressData);
            },
            error: function (error) {
                if (error.responseJSON.code === 500) {
                    showToast("error", error.responseJSON.message);
                }
            },
        });
    }

    function initInternationalPhoneInput() {
        const userPhoneInput = document.querySelector(".phone_number");
        const intlPhoneInput = document.querySelector("#international_phone_number");

        if (userPhoneInput) {
            const iti = intlTelInput(userPhoneInput, {
                utilsScript: window.location.origin + "/front/plugins/intltelinput/js/utils.js",
                separateDialCode: true,
                placeholderNumberType: "",
                autoPlaceholder: "off",
                formatOnDisplay: false,
            });

            userPhoneInput.classList.add("iti");
            userPhoneInput.parentElement.classList.add("intl-tel-input");

            const updateHiddenPhone = () => {
                let intlNumber = iti.getNumber();

                // if intlNumber is empty, build it manually
                if (!intlNumber) {
                    const countryData = iti.getSelectedCountryData();
                    const dialCode = countryData ? countryData.dialCode : "";
                    intlNumber = dialCode ? `+${dialCode}${userPhoneInput.value.trim()}` : userPhoneInput.value.trim();
                }

                intlPhoneInput.value = intlNumber;
                international_phone_number = intlNumber;
            };

            // Update on typing and when country changes
            userPhoneInput.addEventListener("input", updateHiddenPhone);
            userPhoneInput.addEventListener("countrychange", updateHiddenPhone);

            // Update once on init (useful for edit modal prefilled values)
            updateHiddenPhone();
        }
    }

    function initSelect() {
        $(".select2").select2({
            dropdownParent: $("#billing_address_modal"),
            minimumResultsForSearch: 0,
            width: '100%',
            placeholder: _l("Select"),
        });
    }

    function initValidation(){
        $("#billingAddressForm").validate({
            rules: {
                name: {
                    required: true,
                    maxlength: 50,
                },
                phone_number: {
                    required: true,
                    minlength: 10,
                    maxlength: 15
                },
                email: {
                    required: true,
                    email: true,
                },
                address_line_one: {
                    required: true,
                },
                country_id: {
                    required: true,
                },
                state_id: {
                    required: true,
                },
                city_id: {
                    required: true,
                },
                zipcode: {
                    required: true,
                    maxlength: 6,
                },
            },
            messages:{
                name: {
                    required: _l("name_required"),
                },
                phone_number: {
                    required: _l("phone_number_required"),
                    minlength: _l("phone_number_between"),
                    maxlength: _l("phone_number_between"),
                },
                email: {
                    required: _l("email_required"),
                    email: _l("valid_email"),
                },
                address_line_one: {
                    required: _l("address_line_one_required"),
                },
                country_id: {
                    required: _l("country_required"),
                },
                state_id: {
                    required: _l("state_required"),
                },
                city_id: {
                    required: _l("city_required"),
                },
                zipcode: {
                    required: _l("zip_code_required"),
                    maxlength: _l("zip_code_maxlength"),
                },
            },
            errorPlacement: function (error, element) {
                var errorId = element.attr("id") + "_error";
                if (element.hasClass("select2-hidden-accessible")) {
                    $("#" + errorId).text(error.text());
                } else {
                    $("#" + errorId).text(error.text());
                }
            },
            highlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").addClass("is-invalid").removeClass("is-valid");
                }
                $(element).addClass("is-invalid").removeClass("is-valid");
                $("#" + element.id).siblings("span").addClass("me-3");
            },
            unhighlight: function (element) {
                if ($(element).hasClass("select2-hidden-accessible")) {
                    $(element).next(".select2-container").removeClass("is-invalid").addClass("is-valid");
                }
                $(element).removeClass("is-invalid").addClass("is-valid");
                $("#" + element.id).siblings("span").addClass("me-3");
                var errorId = element.id + "_error";
                $("#" + errorId).text("");
            },
            onkeyup: function(element) {
                $(element).valid();
                $("#" + element.id).siblings("span").removeClass("me-3");
            },
            onchange: function(element) {
                $(element).valid();
                $("#" + element.id).siblings("span").removeClass("me-3");
            },
            submitHandler: function(form) {
                let formData = new FormData(form);
                formData.set("phone_number", $("#international_phone_number").val());
                formData.set("is_default", $("#is_default").is(":checked") ? 1 : 0);

                $.ajax({
                    type:"POST",
                    url:"/customer/billing-address/store",
                    data: formData,
                    enctype: "multipart/form-data",
                    processData: false,
                    contentType: false,
                    headers: {
                        "Accept": "application/json",
                        "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                    },
                    beforeSend: function () {
                        $(".submitbtn").attr("disabled", true).html(`
                            <span class="spinner-border spinner-border-sm align-middle" role="status" aria-hidden="true"></span> ${_l("saving")}
                        `);
                    },
                    success:function(resp){
                        $(".error-text").text("");
                        $(".form-control, .select2-container").removeClass("is-invalid is-valid");
                        $(".submitbtn").removeAttr("disabled").html($('#id').val() ? _l("save_changes") : _l("add_address"));
                        if (resp.code === 200) {
                            showToast("success", resp.message);
                            $("#billing_address_modal").modal("hide");
                            getBillingAddress();
                        }
                    },
                    error:function(error){
                        $(".error-text").text("");
                        $(".form-control, .select2-container").removeClass("is-invalid is-valid");
                        $(".submitbtn").removeAttr("disabled").html($('#id').val() ? _l("save_changes") : _l("add_address"));
                        if (error.responseJSON.code === 422) {
                            $.each(error.responseJSON.errors, function(key, val) {
                                $("#" + key).addClass("is-invalid");
                                $("#" + key + "_error").text(val[0]);
                            });
                        } else {
                            showToast("error", error.responseJSON.message);
                        }
                    }
                });
            }
        });
    }

    function initEvents() {
        $("#add_address").on("click", function() {
            $("#billingAddressForm")[0].reset();
            $("#id").val("");
            $(".error-text").text("");
            $(".form-control, .select2-container").removeClass("is-invalid is-valid");
            $(".submitbtn").text(_l("add_address"));
            if (billing_address_count >= 5) {
                showToast("error", _l("billing_address_limit_reached"));
            } else {
                $("#billing_address_modal").modal("show");
            }
        });

        $("#phone_number").on("input", function () {
            $(this).val($(this).val().replace(/[^0-9]/g, "").slice(0, 15));
        });

        $("#zipcode").on("input", function () {
            $(this).val($(this).val().replace(/[^0-9A-Za-z]/g, "").slice(0, 6));
        });

        $("#deleteBillingAddressForm").on("submit", function(e){
            e.preventDefault();
            $.ajax({
                url:"/customer/billing-address/delete",
                type:"POST",
                data: {
                    id: $("#delete_id").val()
                },
                headers: {
                    "Accept": "application/json",
                    "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                },
                success: function(response) {
                    if(response.code === 200){
                        showToast("success", response.message);
                        $("#delete_address_modal").modal("hide");
                        getBillingAddress();
                    }
                },
                error: function(res) {
                    if(res.responseJSON.code === 500){
                        showToast("error", res.responseJSON.message);
                    } else {
                        showToast("error", _l("default_delete_error"));
                    }
                }
            });
        });

        $(document).on("change", ".billing_address", function(e){
            e.preventDefault();
            $.ajax({
                url:"/customer/billing-address/set-default",
                type:"POST",
                data: {
                    id: $(this).val()
                },
                headers: {
                    "Accept": "application/json",
                    "X-CSRF-TOKEN": $("meta[name='csrf-token']").attr("content")
                },
                success: function(response) {
                    if(response.code === 200){
                        showToast("success", response.message);
                        getBillingAddress();
                    }
                },
                error: function(res) {
                    if(res.responseJSON.code === 500){
                        showToast("error", res.responseJSON.message);
                    }
                }
            });
        });

        $(document).on("click", ".edit-address", function() {
            let id = $(this).data("id");
            $("#billingAddressForm").trigger("reset");
            $(".submitbtn").text(_l("save_changes"));
            $(".modal-title").text(_l("edit_billing_address"));

            $.ajax({
                type:"GET",
                url:"/customer/billing-address/edit/"+id,
                success: function(response) {
                    $(".error-text").text("");
                    $(".form-control, .select2-container").removeClass("is-invalid is-valid");
                    if(response.code === 200){
                        let data = response.data;

                        $("#id").val(data.id);
                        $("#name").val(data.name);
                        $("#email").val(data.email);
                        $("#address_line_one").val(data.address_line_one);
                        $("#address_line_two").val(data.address_line_two);
                        $("#country_id").val(data.country_id).trigger("change");
                        fetchStatesByCountry(data.country_id, data.state_id, data.city_id);
                        $("#zipcode").val(data.zipcode);
                        $("#is_default").prop("checked", data.is_default == 1 ? true : false);

                        const phoneNumber = data.phone_number ? data.phone_number.trim() : data.phone_number;
                        const phoneInput = document.querySelector(".phone_number");
                        const hiddenInput = document.querySelector("#international_phone_number");

                        if ($(phoneInput).data("itiInstance")) {
                            $(phoneInput).data("itiInstance").destroy();
                        }
                        const iti = intlTelInput(phoneInput, {
                            utilsScript: window.location.origin + "/backend/assets/plugins/intltelinput/js/utils.js",
                            separateDialCode: true,
                            placeholderNumberType: "",
                            autoPlaceholder: "off",
                            formatOnDisplay: false,
                        });
                        $(phoneInput).data("itiInstance", iti);

                        if (phoneNumber) {
                            iti.setNumber(phoneNumber);
                            hiddenInput.value = iti.getNumber();
                            initialPhoneNumber = phoneNumber;
                        }
                        const updateHiddenPhoneNumber = () => {
                            const currentPhoneNumber = iti.getNumber();
                            if (currentPhoneNumber !== initialPhoneNumber) {
                                hiddenInput.value = currentPhoneNumber.trim();
                            }
                        };

                        phoneInput.addEventListener("input", updateHiddenPhoneNumber);
                        phoneInput.addEventListener("countrychange", updateHiddenPhoneNumber);

                        if (!hiddenInput.value) {
                            hiddenInput.value = initialPhoneNumber;
                        }
                        $("#billing_address_modal").modal("show");
                    }
                }
            });
        });

        $(document).on("click", ".delete-address", function() {
            let id = $(this).data("id");
            $("#delete_id").val(id);
        });

        $(document).on("change", "#country_id", function () {
            const id = $(this).val();
            id ? fetchStatesByCountry(id) : (updateStateOptions(), updateCityOptions());
        });

        $(document).on("change", "#state_id", function () {
            const id = $(this).val();
            id ? fetchCitiesByState(id) : updateCityOptions();
        });
    }

    function updateStateOptions() {
        $("#state_id").empty().append(`<option value="">${_l("Select")}</option>`);
    }

    function updateCityOptions() {
        $("#city_id").empty().append(`<option value="">${_l("Select")}</option>`);
    }

    function fetchStatesByCountry(countryId, selectedStateId = null, selectedCityId = null) {
        $.get("/api/get-states", { country_id: countryId }, (res) => {
            if (res) {
                const $state = $("#state_id").empty().append(`<option value="">${_l("Select")}</option>`);

                $.each(res, (_, item) => {
                    $state.append(`<option value="${item.id}">${item.name}</option>`);
                });

                // set state if provided
                if (selectedStateId) {
                    $state.val(selectedStateId).trigger("change");
                    fetchCitiesByState(selectedStateId, selectedCityId);
                } else {
                    updateCityOptions();
                }
            }
        }).fail(() => showToast("error", "Failed to fetch states."));
    }

    function fetchCitiesByState(stateId, selectedCityId = null) {
        $.get("/api/get-city", { state_id: stateId }, (res) => {
            if (res) {
                const $city = $("#city_id").empty().append(`<option value="">${_l("Select")}</option>`);

                $.each(res, (_, item) => {
                    $city.append(`<option value="${item.id}">${item.name}</option>`);
                });

                // set city if provided
                if (selectedCityId) {
                    $city.val(selectedCityId).trigger("change");
                }
            }
        }).fail(() => showToast("error", "Failed to fetch cities."));
    }

})();