<?php

namespace App\Services;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Intervention\Image\Laravel\Facades\Image;

class ImageResizer
{
    private function createDirectories(string $basePath, array $sizes): void
    {
        foreach ($sizes as $folder => $_) {
            File::ensureDirectoryExists($this->buildPath($basePath, $folder), 0755, true);
        }
    }

    private function deleteOldFiles(string $basePath, array $sizes, string $oldFilename): void
    {
        foreach ($sizes as $folder => $_) {
            $oldPath = $this->buildPath($basePath, $folder) . $oldFilename;
            if (File::exists($oldPath)) {
                File::delete($oldPath);
            }
        }
    }

    private function processSvg(UploadedFile $file, string $basePath, array $sizes, string $uniqueName, ?string $oldFilePath): string
    {
        $file->move($this->buildPath($basePath, 'original'), $uniqueName);

        if ($oldFilePath) {
            $this->deleteOldFiles($basePath, $sizes, basename($oldFilePath));
        }

        return $uniqueName;
    }

    private function processImage(UploadedFile $file, string $basePath, array $sizes, string $uniqueName): ?string
    {
        try {
            $image = Image::read($file);
        } catch (\Exception $e) {
            \Log::error('Image read error: ' . $e->getMessage());
            return null;
        }

        foreach ($sizes as $folder => $dimensions) {
            $targetPath = $this->buildPath($basePath, $folder) . $uniqueName;
            $resized = clone $image;

            if ($dimensions) {
                $resized->resize($dimensions[0], $dimensions[1]);
            }

            try {
                $resized->save($targetPath);
            } catch (\Exception $e) {
                \Log::error("Failed to save image [{$targetPath}]: " . $e->getMessage());
            }
        }

        return $uniqueName;
    }

    private function buildPath(string $basePath, string $folder): string
    {
        return $basePath . ($folder === 'original' ? '' : "{$folder}/");
    }

    /**
     * Upload and resize image (original, large, medium, thumbnail).
     * SVG will be stored as-is (not resized).
     *
     * @param UploadedFile $file
     * @param string $baseFolder
     * @param string|null $oldFilePath
     *
     * @return string|null
     */
    public function uploadFile(UploadedFile $file, string $baseFolder, ?string $oldFilePath = null): ?string
    {
        if (! $file->isValid()) {
            return null;
        }

        $extension = strtolower($file->getClientOriginalExtension());
        $isSvg = $extension === 'svg';

        $uniqueName = Str::uuid() . '_' . time() . '.' . $extension;
        $basePath = storage_path("app/public/{$baseFolder}/");
        $sizes = [
            'original' => null,
            'large' => [1200, 1000],
            'medium' => [900, 600],
            'small' => [690, 420],
            'thumbnail' => [300, 200],
        ];

        $this->createDirectories($basePath, $sizes);

        $result = $isSvg
            ? $this->processSvg($file, $basePath, $sizes, $uniqueName, $oldFilePath)
            : $this->processImage($file, $basePath, $sizes, $uniqueName);

        if (! $result) {
            return null;
        }

        if (! $isSvg && $oldFilePath) {
            $this->deleteOldFiles($basePath, $sizes, basename($oldFilePath));
        }

        return "{$baseFolder}/{$result}";
    }
}
