<?php

namespace App\Repositories\Eloquent;

use App\Models\BookingProduct;
use App\Models\Bookings;
use App\Models\PayoutDetail;
use App\Models\PayoutHistory;
use App\Models\User;
use App\Models\UserDetail;
use App\Repositories\Contracts\TransactionInterface;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Product\app\Models\Product;
use Modules\Product\app\Models\Productmeta;
use Modules\Product\app\Models\Rating;
use Modules\GlobalSetting\app\Models\Currency;
use App\Models\ProviderRequestAmount;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use App\Models\Categories;
use Illuminate\Http\Request;
use Modules\Service\app\Models\Service;

class TransactionRepository implements TransactionInterface
{
    public function listTransactions(
        ?int $userId = null,
        ?int $customerId = null,
        ?int $providerId = null,
        ?string $search = null,
        string $orderBy = 'desc',
        string $sortBy = 'id'
    ): Collection {

        $providerUserType = User::where('id', $providerId)
            ->where('user_type', 4)
            ->exists();

        $query = Bookings::select([
            'id', 'order_id', 'product_id', 'branch_id', 'staff_id', 'slot_id',
            'type', 'booking_date', 'from_time', 'to_time', 'booking_status',
            'amount_tax', 'tax_details', 'user_id', 'provider_id', 'first_name',
            'last_name', 'user_email', 'user_phone', 'user_city', 'user_state',
            'user_address', 'notes', 'tranaction', 'service_offer', 'additional_services',
            'payment_type', 'payment_status', 'payment_proof_path', 'bank_transfer_proof',
            'user_postal', 'service_amount', 'service_qty', 'total_amount',
            'created_by', 'created_at', 'deleted_at', 'updated_at'
        ])
        ->with([
            'user:id,name,email,user_type',
            'user.userDetail:user_id,profile_image,mobile_number,address,city,state,postal_code',
            'product:id,source_name,source_type,source_category,created_by',
            'product.createdBy:id,name,email,user_type',
            'product.createdBy.userDetail:user_id,profile_image,mobile_number,address,city,state,postal_code',
            'staff:id,name,email,user_type', // if staff relationship exists
            'staff.userDetail:user_id,profile_image,mobile_number,address,city,state,postal_code',
            'bookingProducts:id,user_id,provider_id,shop_id,booking_id,product_id,product_name,quantity,variant,color,price,subtotal,cancellation_reason,product_status,created_at,updated_at',
            'bookingProducts.provider:id,name,email,user_type',
            'bookingProducts.provider.userDetail:user_id,profile_image,mobile_number,address,city,state,postal_code',
            'bookingProducts.productDetails:id,source_name,source_type,source_category,created_by'
        ])
        ->when($userId, function ($query, $userId) {
            $query->where('user_id', $userId);
        })
        ->when($customerId, function ($query, $customerId) {
            $query->whereHas('user', function ($q) use ($customerId) {
                $q->where('id', $customerId);
            });
        })
        ->when($providerId, function ($query, $providerId) {
            $query->where(function ($q) use ($providerId) {
                $q->whereHas('product', function ($productQuery) use ($providerId) {
                    $productQuery->where('created_by', $providerId);
                })
                ->orWhere('provider_id', $providerId)
                ->orWhere('staff_id', $providerId);
            });
        })
        ->when($search, function ($query, $search) {
            $query->where(function ($q) use ($search) {
                $q->whereHas('user', function ($userQuery) use ($search) {
                    $userQuery->where('name', 'like', '%' . $search . '%')
                        ->orWhere('email', 'like', '%' . $search . '%');
                })
                ->orWhereHas('product', function ($productQuery) use ($search) {
                    $productQuery->where('name', 'like', '%' . $search . '%')
                        ->orWhere('source_name', 'like', '%' . $search . '%');
                })
                ->orWhere('order_id', 'like', '%' . $search . '%')
                ->orWhere('first_name', 'like', '%' . $search . '%')
                ->orWhere('last_name', 'like', '%' . $search . '%')
                ->orWhere('user_email', 'like', '%' . $search . '%');
            });
        })
        ->withTrashed()
        ->orderBy($sortBy, $orderBy);

        return $query->get();
    }


    public function getTransactionStatusMap(): array
    {
        return [
            1 => 'Open',
            2 => 'Accepted',
            3 => 'Cancelled',
            4 => 'In Progress',
            5 => 'Completed',
        ];
    }

    public function getPaymentTypeMap(): array
    {
        return [
            1 => 'Paypal',
            2 => 'Stripe',
            3 => 'Razorpay',
            4 => 'Bank Transfer',
            5 => 'COD',
            6 => 'Wallet',
            7 => 'Mollie',
        ];
    }

    public function getPaymentStatusMap(): array
    {
        return [
            1 => 'Unpaid',
            2 => 'Paid',
            3 => 'Refund',
        ];
    }

    public function getCommissionRate(): float
    {
        $commissionSetting = GlobalSetting::where('key', 'commission_rate_percentage')->first();
        return $commissionSetting ? (float) $commissionSetting->value : 0;
    }

    public function getProductImages(array $productIds): Collection
    {
        return Productmeta::whereIn('product_id', $productIds)
            ->where('source_key', 'product_image')
            ->whereNull('deleted_at')
            ->get()
            ->groupBy('product_id');
    }

    public function getCurrencySymbol(): string
    {
        $currency = Cache::remember('currency_details', 86400, function () {
            return Currency::select('symbol')
                ->orderBy('id', 'DESC')
                ->where('is_default', 1)
                ->first();
        });

        return $currency->symbol ?? '';
    }

    public function getDateFormat(): string
    {
        $dateformatSetting = GlobalSetting::where('key', 'date_format_view')->first();
        return $dateformatSetting->value ?? 'Y-m-d';
    }

    public function getCustomerProfileImage(?string $image): string
    {
        if (!empty($image)) {
            return file_exists(public_path('storage/profile/' . $image))
                ? url('storage/profile/' . $image)
                : asset('assets/img/profile-default.png');
        }
        return asset('assets/img/profile-default.png');
    }

    public function getProviderProfileImage(?string $image): string
    {
        return $this->getCustomerProfileImage($image);
    }

    public function getCategoryName(?int $categoryId): string
    {
        if (!$categoryId) {
            return 'No Category';
        }

        $category = Categories::where('id', $categoryId)
            ->select('id', 'name')
            ->first();

        return $category ? ucfirst($category->name) : 'No Category';
    }

    public function getProductImageUrl(Collection $images): string
    {
        $validImage = $images->first(function ($img) {
            return !empty($img->source_Values) && file_exists(public_path('storage/' . $img->source_Values));
        });

        return uploadedAsset($validImage->source_Values ?? '', 'default2');
    }
    public function uploadPaymentProof(array $data)
    {
        return DB::transaction(function () use ($data) {
            $booking = Bookings::findOrFail($data['booking_id']);
            $booking->update([
                'payment_proof_path' => $data['file_path'],
                'payment_status' => 2,
            ]);
            return $booking;
        });
    }

    public function getProviderDetails(int $providerId)
    {
        $user = User::where('id', $providerId)
            ->where('user_type', 2)
            ->with(['userDetails.category'])
            ->firstOrFail();

        $products = Product::where('created_by', $providerId)->get();

        foreach ($products as $product) {
            $ratingDetails = Rating::getProductRatingDetails($product->id);
            $product->average_rating = $ratingDetails['average_rating'];
            $product->rating_count = $ratingDetails['rating_count'];
            $productImageMeta = Productmeta::where('product_id', $product->id)
                ->where('source_key', 'product_image')
                ->select('source_Values')
                ->first();
            $product->image_url = $productImageMeta->source_Values ?? null;
        }

        return [
            'user' => $user,
            'products' => $products,
        ];
    }

    public function providerTransaction(?int $providerId = null)
    {
        $commissionRate = 0;
        $commissionSetting = GlobalSetting::where('key', 'commission_rate_percentage')->first();
        if ($commissionSetting) {
            $commissionRate = (float) $commissionSetting->value;
        }

        // Get providers
        $providersQuery = User::query()->where('user_type', 2);
        if ($providerId) {
            $providersQuery->where('id', $providerId);
        }
        $providers = $providersQuery->get();

        // Get currency
        $currencySymbol = Cache::remember('currency_details', 86400, function () {
            return Currency::select('symbol')->orderBy('id', 'DESC')->where('is_default', 1)->first();
        });

        $response = [];

        foreach ($providers as $provider) {

            $providerId = $provider->id; // the provider you want bookings for

            $serviceBookings = Bookings::where('booking_status', [5, 6])
                ->where('type', 'service')
                ->whereHas('product', function ($query) use ($providerId) {
                    $query->where('created_by', $providerId);
                })
                ->get();

            // Get product bookings (This logic remains the same)
            $productBookings = Bookings::where('type', '!=', 'service')
                ->whereHas('bookingProducts', function ($q) use ($provider) {
                    $q->where(function($subQ) use ($provider) {
                        $subQ->where('provider_id', $provider->id)
                            ->orWhere('provider_id', (string)$provider->id);
                    })
                    ->where('product_status', 4); // filter products with status 4
                })
                ->with(['bookingProducts' => function ($q) use ($provider) {
                    $q->where(function($subQ) use ($provider) {
                        $subQ->where('provider_id', $provider->id)
                            ->orWhere('provider_id', (string)$provider->id);
                    })
                    ->where('product_status', 4); // eager load only products with status 4
                }])
                ->get();

            $totals = ['totalGrossAmount' => 0, 'totalCommission' => 0, 'totalReducedAmount' => 0];
            $bookingCount = 0;

            // Calculate service bookings
            foreach ($serviceBookings as $booking) {
                $grossAmount = floatval($booking->total_amount ?? 0);

                if ($grossAmount > 0) {
                    $commissionAmount = ($grossAmount * $commissionRate) / 100;
                    $reducedAmount = $grossAmount - $commissionAmount;

                    $totals['totalGrossAmount'] += $grossAmount;
                    $totals['totalCommission'] += $commissionAmount;
                    $totals['totalReducedAmount'] += $reducedAmount;
                    $bookingCount++;
                }
            }

            // Calculate product bookings
            foreach ($productBookings as $booking) {
                if ($booking->bookingProducts->count() > 0) {
                    $bookingCount++;
                    foreach ($booking->bookingProducts as $bp) {
                        $grossAmount = floatval($bp->gross_total ?? 0);

                        if ($grossAmount > 0) {
                            $commissionAmount = ($grossAmount * $commissionRate) / 100;
                            $reducedAmount = $grossAmount - $commissionAmount;

                            $totals['totalGrossAmount'] += $grossAmount;
                            $totals['totalCommission'] += $commissionAmount;
                            $totals['totalReducedAmount'] += $reducedAmount;
                        }
                    }
                }
            }

            $enteredAmount = PayoutHistory::where('user_id', $provider->id)->sum('process_amount');

            $profileImage = $provider->userDetails->profile_image ?? '';
            $profileImage = !empty($profileImage) && file_exists(public_path('storage/profile/' . $profileImage))
                ? url('storage/profile/' . $profileImage)
                : asset('assets/img/profile-default.png');

            $response[] = [
                'payout_details' => $this->getProviderPayoutDetails($provider->id),
                'provider' => [
                    'id' => $provider->id,
                    'name' => ucfirst($provider->name),
                    'email' => $provider->email,
                    'profile_image' => $profileImage,
                ],
                'transactions' => [
                    'total_bookings' => $bookingCount,
                    'total_gross_amount' => $totals['totalGrossAmount'],
                    'total_commission_amount' => $totals['totalCommission'],
                    'total_reduced_amount' => $totals['totalReducedAmount'],
                    'entered_amount' => $enteredAmount,
                    'remaining_amount' => $totals['totalReducedAmount'] - $enteredAmount,
                    'commission_rate' => $commissionRate,
                ],
                'currencySymbol' => optional($currencySymbol)->symbol ?? '',
            ];
        }

        // Sort by total bookings descending
        usort($response, function ($a, $b) {
            return $b['transactions']['total_bookings'] <=> $a['transactions']['total_bookings'];
        });

        return $response;
    }

    private function getProviderPayoutDetails($providerId, ?int $payoutType = null): array
    {
        $payoutDetails = PayoutDetail::when(empty($payoutType), function ($query) {
            $query->where('status', 1);
        })
            ->when($payoutType, function ($query, $payoutType) {
                $query->where('payout_type', $payoutType);
            })
            ->where('provider_id', $providerId)
            ->first();

        $finalPayoutDetails = [];
        if ($payoutDetails) {
            $details = $payoutDetails->payout_detail;
            if ($payoutDetails && is_string($payoutDetails->payout_detail)) {
                $decoded = json_decode($payoutDetails->payout_detail, true);

                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    $details = $decoded;
                }
            }
            $payoutMethod = '';
            switch ($payoutDetails->payout_type) {
                case 1:
                    $payoutMethod = "PayPal";
                    break;
                case 2:
                    $payoutMethod = "Stripe";
                    break;
                case 4:
                    $payoutMethod = "Bank Transfer";
                    break;
            }

            $finalPayoutDetails = [
                'payment_method' => $payoutMethod,
                'payout_details' => $details
            ];
        }

        return $finalPayoutDetails;
    }

    public function storePayoutHistory(array $data)
    {
        return DB::transaction(function () use ($data) {
            $payoutHistory = PayoutHistory::create([
                'type' => 1,
                'user_id' => $data['provider_id'],
                'total_bookings' => $data['total_bookings'],
                'total_earnings' => $data['total_earnings'],
                'admin_earnings' => $data['admin_earnings'],
                'pay_due' => $data['provider_pay_due'],
                'process_amount' => $data['entered_amount'],
                'payment_proof' => $data['payment_proof_path'],
                'remaining_amount' => $data['remaining_amount'],
                'payment_method' => $data['payment_method'],
            ]);

            try {
                \App\Helpers\InvoiceHelper::generateInvoice(
                    $payoutHistory->id,
                    $data['entered_amount'],
                    '3',
                    $data['provider_id'],
                );
            } catch (\Exception $e) {
                // Log the error but don't fail the transaction
                Log::error('Invoice generation failed: ' . $e->getMessage());
            }

            return $payoutHistory;
        });
    }

    public function savePayouts(array $data)
    {
        return DB::transaction(function () use ($data) {
            $payoutData = [
                'provider_id' => $data['provider_id'],
                'payout_type' => $data['payout_type'],
                'status' => 1,
            ];

            if ($data['payout_type'] == 1) {
                $payoutData['payout_detail'] = $data['paypal_id'];
            } elseif ($data['payout_type'] == 2) {
                $payoutData['payout_detail'] = $data['stripe_id'];
            } elseif ($data['payout_type'] == 4) {
                $bankData = [
                    'holder_name' => $data['holder_name'],
                    'bank_name' => $data['bank_name'],
                    'account_number' => $data['account_number'],
                    'ifsc' => $data['ifsc'],
                ];
                $payoutData['payout_detail'] = json_encode($bankData);
            }

            PayoutDetail::updateOrCreate(['id' => $data['id'] ?? ''], $payoutData);
            PayoutDetail::where('provider_id', $data['provider_id'])
                ->where('payout_type', '!=', $data['payout_type'])
                ->update(['status' => 0]);

            return true;
        });
    }

    public function getPayoutDetails(int $providerId)
    {
        return PayoutDetail::where(['provider_id' => $providerId])
            ->get(['id', 'provider_id', 'payout_type', 'payout_detail', 'status'])
            ->map(function ($data) {
                if ($data['payout_type'] == 4) {
                    $data['payout_detail'] = json_decode($data['payout_detail']);
                }
                return $data;
            });
    }

    public function getProviderPayoutHistory(int $providerId)
    {
        return PayoutHistory::where('user_id', $providerId)
            ->orderBy('id', 'DESC')
            ->get()
            ->map(function ($item) {
                return [
                    'id' => $item->id,
                    'provider_id' => $item->user_id,
                    'total_amount' => ($item->process_amount + $item->remaining_amount),
                    'processed_amount' => $item->process_amount,
                    'available_amount' => $item->remaining_amount,
                    'payment_proof_path' => url('storage/' . $item->payment_proof),
                    'payment_method' => $item->payment_method,
                    'created_at' => formatDateTime($item->created_at),
                ];
            });
    }

    public function getProviderPayoutRequest(?int $providerId)
    {
        return ProviderRequestAmount::where('provider_id', $providerId)
            ->where('status', 0)
            ->orderBy('id', 'DESC')
            ->get()
            ->map(function ($item) {
                $paymentId = $item->payment_id;
                $payoutMethod = $item->payment_method;
                $item->payment_method = $payoutMethod;
                $item->amount = number_format($item->amount, 2, '.', '');
                $item->created_date = formatDateTime($item->created_at);
                return $item;
            });
    }

    public function listProviderRequests()
    {
        return ProviderRequestAmount::with('provider:id,name')
            ->select('id', 'provider_id', 'payment_id', 'amount', 'status', 'created_at')
            ->get()
            ->map(function ($request) {
                return [
                    'id' => $request->id,
                    'provider_id' => $request->provider_id,
                    'provider_name' => $request->provider->name ?? 'N/A',
                    'payment_id' => $request->payment_id,
                    'amount' => $request->amount,
                    'status' => $request->status,
                    'status_label' => $request->status === 0 ? 'UnPaid' : ($request->status === 1 ? 'Paid' : 'Refund'),
                    'created_at' => $request->created_at,
                    'provider_payout_details' => $this->getProviderPayoutDetails($request->provider_id, $request->payment_id),
                ];
            });
    }

    public function updateProviderRequest(array $data)
    {
        return DB::transaction(function () use ($data) {
            $providerRequest = ProviderRequestAmount::findOrFail($data['id']);

            $providerRequest->update([
                'status' => 1,
                'payment_proof_path' => $data['payment_proof_path'],
            ]);

            $commissionRate = GlobalSetting::where('key', 'commission_rate_percentage')
                ->value('value') ?? 0;

            $enteredAmount = PayoutHistory::where('user_id', $data['provider_id'])
                ->sum('process_amount');

            $bookings = Bookings::with(['product.createdBy'])
                ->where('booking_status', 6)
                ->whereHas('product', function ($query) use ($data) {
                    $query->where('created_by', $data['provider_id']);
                })
                ->get();

            $totals = $bookings->reduce(function ($carry, $booking) use ($commissionRate) {
                $totalAmount = $booking->total_amount;
                $commissionAmount = ($totalAmount * $commissionRate) / 100;
                $carry['totalEarnings'] += $totalAmount;
                $carry['adminEarnings'] += $commissionAmount;
                return $carry;
            }, ['totalEarnings' => 0, 'adminEarnings' => 0]);

            $providerPayDue = $totals['totalEarnings'] - $totals['adminEarnings'];
            $remainingAmount = $providerPayDue - $enteredAmount;

            $payoutHistory = PayoutHistory::create([
                'user_id' => $data['provider_id'],
                'type' => 1,
                'total_bookings' => $bookings->count(),
                'total_earnings' => $totals['totalEarnings'],
                'admin_earnings' => $totals['adminEarnings'],
                'pay_due' => $providerPayDue,
                'process_amount' => $data['provider_amount'],
                'payment_proof' => $data['payment_proof_path'],
                'remaining_amount' => $remainingAmount - $data['provider_amount'],
                'payment_method' => $data['payment_method'],
            ]);

            try {
                \App\Helpers\InvoiceHelper::generateInvoice(
                    $payoutHistory->id,
                    $data['provider_amount'],
                    '3',
                    $data['provider_id'],
                );
            } catch (\Exception $e) {
                Log::error($e->getMessage());
            }

            return $payoutHistory;
        });
    }

    public function sendProviderRequestAmount(array $data)
    {
        return DB::transaction(function () use ($data) {
            // Check if payment method is set up
            $payment = PayoutDetail::where([
                'provider_id' => $data['provider_id'],
                'status' => 1
            ])->first();

            if (!$payment) {
                throw new \Exception('Please set your payment method.');
            }

            // Calculate available balance
            $remainingAmount = $this->calculateProviderBalance($data['provider_id']);
            $requestAmount = ProviderRequestAmount::where([
                'provider_id' => $data['provider_id'],
                'status' => 0
            ])->sum('amount');
            $available_amount = $remainingAmount - $requestAmount;

            if ($data['amount'] > $available_amount) {
                throw new \Exception('Amount must be less than available amount.');
            }

            // Prepare payment details based on payment method
            $paymentDetails = [];
            $metadata = [];

            switch ($data['payment_method']) {
                case 'credit_card':
                    $paymentDetails = [
                        'card_holder_name' => $data['card_holder_name'] ?? '',
                        'card_last_four' => $this->getLastFourDigits($data['card_number'] ?? ''),
                        'expiry_date' => $data['expiry_date'] ?? '',
                        'card_type' => $this->detectCardType($data['card_number'] ?? ''),
                        // Never store full card number or CVV
                    ];
                    break;

                case 'paypal':
                    $paymentDetails = [
                        'paypal_email' => $data['paypal_email'] ?? '',
                    ];
                    break;

                case 'stripe':
                    $paymentDetails = [
                        'stripe_email' => $data['stripe_email'] ?? '',
                    ];
                    break;
            }

            // Add metadata
            $metadata = [
                'user_agent' => request()->header('User-Agent'),
                'ip_address' => request()->ip(),
                'request_time' => now()->toDateTimeString(),
                'additional_data' => array_diff_key($data, array_flip([
                    'provider_id',
                    'amount',
                    'payment_method',
                    'card_holder_name',
                    'card_number',
                    'cvv',
                    'expiry_date',
                    'paypal_email',
                    'paypal_password',
                    'stripe_email',
                    'stripe_password'
                ]))
            ];

            // Create withdrawal request
            return ProviderRequestAmount::create([
                'provider_id' => $data['provider_id'],
                'payment_id' => $payment->id,
                'amount' => $data['amount'],
                'payment_method' => $data['payment_method'],
                'payment_details' => json_encode($paymentDetails),
                'request_timestamp' => now(),
                'metadata' => json_encode($metadata)
            ]);
        });
    }

    /**
     * Get last four digits of card number
     */
    private function getLastFourDigits($cardNumber)
    {
        $cleanNumber = preg_replace('/\D/', '', $cardNumber);
        return strlen($cleanNumber) >= 4 ? substr($cleanNumber, -4) : '';
    }

    /**
     * Detect card type from card number
     */
    private function detectCardType($cardNumber)
    {
        $cleanNumber = preg_replace('/\D/', '', $cardNumber);

        if (preg_match('/^4/', $cleanNumber)) {
            return 'Visa';
        } elseif (preg_match('/^5[1-5]/', $cleanNumber)) {
            return 'Mastercard';
        } elseif (preg_match('/^3[47]/', $cleanNumber)) {
            return 'American Express';
        } elseif (preg_match('/^6(?:011|5)/', $cleanNumber)) {
            return 'Discover';
        }

        return 'Unknown';
    }
    public function getProviderBalance(int $providerId)
    {
        $remainingAmount = $this->calculateProviderBalance($providerId);
        $payout = PayoutHistory::select('process_amount')
            ->where('user_id', $providerId)
            ->orderBy('id', 'DESC')
            ->first();
        $requestAmount = ProviderRequestAmount::where(['provider_id' => $providerId, 'status' => 0])->sum('amount');

        $available_amount = $remainingAmount - $requestAmount;
        $last_payout = $payout->process_amount ?? 0;

        return [
            'available_amount' => number_format($available_amount, 2, '.', ''),
            'last_payout' => number_format($last_payout, 2, '.', ''),
        ];
    }

    public function calculateProviderBalance(int $providerId): float
    {
        $commissionRate = (float) (GlobalSetting::where('key', 'commission_rate_percentage')->value('value') ?? 0);
        $productIds = Service::where('user_id', $providerId)->pluck('id')->toArray();

        $transactions = Bookings::with(['product'])
            ->where('booking_status', 6)
            ->where(function ($query) use ($productIds, $providerId) {
                $query->whereIn('product_id', $productIds)
                    ->orWhere('staff_id', $providerId);
            })
            ->get();

        $totals = $transactions->reduce(function ($carry, $booking) use ($commissionRate) {
            $grossAmount = $booking->total_amount ?? 0;
            $commissionAmount = ($grossAmount * $commissionRate) / 100;
            $reducedAmount = $grossAmount - $commissionAmount;

            $carry['totalGrossAmount'] += $grossAmount;
            $carry['totalCommission'] += $commissionAmount;
            $carry['totalReducedAmount'] += $reducedAmount;
            return $carry;
        }, ['totalGrossAmount' => 0, 'totalCommission' => 0, 'totalReducedAmount' => 0]);

        $enteredAmount = PayoutHistory::where('user_id', $providerId)->sum('process_amount');
        return $totals['totalReducedAmount'] - $enteredAmount;
    }

    public function getUserPayoutRequests(Request $request)
    {
        $type = $request->type ?? 'service';

        $bookings = collect();
        $productBookings = collect();

        if ($type == 'service') {
            $bookings = Bookings::select(
                    'bookings.id',
                    'bookings.order_id',
                    'bookings.product_id',
                    'bookings.user_id',
                    'bookings.booking_date',
                    'bookings.total_amount',
                    'bookings.type',
                    'bookings.booking_status',
                )
                ->with([
                    'user.userDetails',
                    'service:id,source_name',
                    'user.additionalSettings' => function ($q) {
                        $q->where('key', 'account_settings');
                    }
                ])
                ->when($type == 'service', function ($query) {
                    return $query->where('type', 'service')
                        ->where('bookings.booking_status', 4);
                })
                ->get()
                ->map(function ($booking) {
                    $serviceArray = [];

                    $productImages = $this->getProductImages([$booking->product_id])->get($booking->product_id, collect());
                    $serviceArray = [
                        'id' => $booking->id,
                        'product_id' => $booking->product_id,
                        'service_name' => ucfirst($booking->service->source_name ?? ''),
                        'service_amount' => number_format($booking->service_amount, 2, '.', ''),
                        'total_amount' => number_format($booking->total_amount, 2, '.', ''),
                        'product_image' => $this->getProductImageUrl($productImages),
                    ];

                    $accountSettings = [];

                    if ($booking->user && $booking->user->additionalSettings && $booking->user->additionalSettings->isNotEmpty()) {
                        $settings = json_decode($booking->user->additionalSettings->first()->value, true);
                        if (!empty($settings['default_type'])) {
                            $type = $settings['default_type'];

                            $accountSettings = [
                                $type => $settings[$type] ?? []
                            ];
                        }
                    }

                    return [
                        'id' => $booking->id,
                        'order_id' => $booking->order_id,
                        'user_id' => $booking->user_id,
                        'type' => ucfirst($booking->type),
                        'booking_date' => formatDateTime($booking->booking_date),
                        'service' => $serviceArray,
                        'customer' => [
                            'id' => $booking->user_id,
                            'name' => $booking->user->userDetails && $booking->user->userDetails->first_name
                                ? ucwords($booking->user->userDetails->first_name  . ' ' . $booking->user->userDetails->last_name) : ucfirst($booking->user->name),
                            'email' => $booking->user->email,
                            'profile_image' => uploadedAsset('profile/' . $booking->user->userDetails->profile_image ?? '', 'profile'),
                            'account_settings' => $accountSettings
                        ],
                    ];
                });
        } else {
            $productBookings = BookingProduct::with([
                    'user.userDetails',
                    'booking:id,order_id,product_id,user_id,booking_date,total_amount,type,booking_status',
                    'product:id,source_name',
                    'user.additionalSettings' => function ($q) {
                        $q->where('key', 'account_settings');
                    }
                ])
                ->where('product_status', 7)
                ->get()
                ->map(function ($bp) {
                    $productArray = [];

                    $productId = $bp->product_id;
                    $productImagesMap = $this->getProductImages([$productId]);
                    $images = $productImagesMap->get($bp->product_id, collect());

                    $productArray = [
                        'product_id' => $bp->product_id,
                        'product_name' => ucfirst($bp->product->source_name ?? ''),
                        'quantity' => $bp->quantity,
                        'price' => number_format($bp->price, 2, '.', ''),
                        'subtotal' => number_format($bp->subtotal, 2, '.', ''),
                        'gross_total' => number_format($bp->gross_total, 2, '.', ''),
                        'product_status' => $bp->product_status,
                        'product_image' => $this->getProductImageUrl($images)
                    ];

                    $accountSettings = [];

                    if ($bp->user && $bp->user->additionalSettings && $bp->user->additionalSettings->isNotEmpty()) {
                        $settings = json_decode($bp->user->additionalSettings->first()->value, true);
                        if (!empty($settings['default_type'])) {
                            $type = $settings['default_type'];

                            $accountSettings = [
                                $type => $settings[$type] ?? []
                            ];
                        }
                    }

                    return [
                        'id' => $bp->id,
                        'booking_id' => $bp->booking_id,
                        'order_id' => $bp->booking->order_id,
                        'user_id' => $bp->user_id,
                        'type' => ucfirst($bp->booking->type),
                        'booking_date' => formatDateTime($bp->booking->booking_date),
                        'product' => $productArray,
                        'customer' => [
                            'id' => $bp->user_id,
                            'name' => $bp->user->userDetails && $bp->user->userDetails->first_name
                                ? ucwords($bp->user->userDetails->first_name  . ' ' . $bp->user->userDetails->last_name) : ucfirst($bp->user->name),
                            'email' => $bp->user->email,
                            'profile_image' => uploadedAsset('profile/' . $bp->user->userDetails->profile_image ?? '', 'profile'),
                            'account_settings' => $accountSettings
                        ],
                    ];
                });
        }

        return [
            'service_bookings' => $bookings,
            'product_bookings' => $productBookings
        ];
    }

    public function updateRefund(array $data)
    {
        return DB::transaction(function () use ($data) {
            if (empty($data['booking_product_id'])) {
                $payoutHistory = PayoutHistory::where('reference_id', $data['bookingid'])
                    ->update([
                        'payment_proof' => $data['payment_proof_path'],
                        'payment_method' => $data['payment_method'],
                        'remaining_amount' => 0
                    ]);
                Bookings::where('id', $data['bookingid'])
                    ->update(['booking_status' => '7']);
            } else {
                $payoutHistory = PayoutHistory::where('reference_id', $data['bookingid'])
                    ->where('booking_product_id', $data['booking_product_id'])
                    ->update([
                        'payment_proof' => $data['payment_proof_path'],
                        'payment_method' => $data['payment_method'],
                        'remaining_amount' => 0
                    ]);
                BookingProduct::where('id', $data['booking_product_id'])
                    ->update(['product_status' => 8]);
            }


            return $payoutHistory;
        });
    }

    public function mapDateFormatToSQL(string $phpFormat): string
    {
        $replacements = [
            'd' => '%d',
            'D' => '%a',
            'j' => '%e',
            'l' => '%W',
            'F' => '%M',
            'm' => '%m',
            'M' => '%b',
            'n' => '%c',
            'Y' => '%Y',
            'y' => '%y',
            'a' => '%p',
            'A' => '%p',
            'g' => '%l',
            'G' => '%k',
            'h' => '%I',
            'H' => '%H',
            'i' => '%i',
            's' => '%S',
        ];

        return strtr($phpFormat, $replacements);
    }

    private function getProfileImageUrl(?string $image): string
    {
        if (!empty($image)) {
            return file_exists(public_path('storage/profile/' . $image))
                ? url('storage/profile/' . $image)
                : asset('assets/img/profile-default.png');
        }
        return asset('assets/img/profile-default.png');
    }
}
