<?php

namespace App\Repositories\Eloquent;

use Modules\GlobalSetting\app\Models\Placeholders;
use Modules\Categories\app\Models\Categories;
use Modules\Product\app\Models\Product;
use Modules\Service\app\Models\Productmeta;
use Modules\Product\app\Models\Rating;
use App\Models\UserDetail;
use App\Models\Country;
use App\Models\State;
use App\Models\City;
use Modules\GlobalSetting\app\Models\Language;
use App\Models\Bookings;
use App\Models\PackageTrx;
use App\Models\ProductVariation;
use Carbon\Carbon;
use Modules\GlobalSetting\app\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;
use Modules\Service\app\Models\AdditionalService;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;
use Modules\GlobalSetting\Entities\GlobalSetting;
use App\Repositories\Contracts\ServiceRepositoryInterface;
use Modules\Shops\app\Models\Shops;

class ServiceRepository implements ServiceRepositoryInterface
{
    public function productlist(Request $request): JsonResponse | View
    {
        $language_id = getLanguageId(app()->getLocale());

        // --- Prepare Static Data for View ---
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('products'),
            'header' => __('products'),
            'banner_visible' => 1,
            'language_id' => $language_id,
            'currency_code' => getDefaultCurrencySymbol(),
            'email' => Auth::check() ? Auth::user()->email : ""
        ];

        $bestSellerCount = Product::join(
            DB::raw('(SELECT product_id, SUM(quantity) as total_sold
                FROM booking_products
                WHERE product_status = 4
                GROUP BY product_id
                ORDER BY total_sold DESC
                LIMIT 5) as bp'), 'products.id', '=', 'bp.product_id')
            ->where('products.source_type', 'product')
            ->where('products.status', 1)
            ->where('products.language_id', $language_id)
            ->whereHas('shop')
            ->count();

        // --- Highlights Filter Data ---
        $highlighted = [
            'all_products' =>  [
                'title' => __('all_products'),
                'count' => Product::whereHas('shop')
                    ->where('source_type', 'product')
                    ->where('status', 1)
                    ->where('language_id', $language_id)
                    ->count(), // Example dynamic count
            ],
            'featured' => [
                'title' => __('featured'),
                'count' => Product::whereHas('shop')
                    ->where('source_type', 'product')
                    ->where('featured', 1)
                    ->where('status', 1)
                    ->where('language_id', $language_id)
                    ->count(), // Example dynamic count
            ],
            'best_seller' =>  [
                'title' => __('best_seller'),
                'count' => $bestSellerCount, // Example dynamic count
            ]
        ];

        // --- DYNAMIC FILTER DATA ---

        // Fetch Categories with product count (Cached for performance)
        $categories = Categories::where(['status' => 1, 'language_id' => $language_id, 'parent_id' => 0, 'source_type' => 'product'])
            ->withCount(['products' => function ($query) {
                $query->where('source_type', 'product')->where('status', 1)
                    ->whereHas('shop');
            }])
            ->get();

        // Fetch Shops with their active product count
        $shops = Shops::where('status', 1)
            ->where('language_id', $language_id)
            ->withCount(['products' => function ($query) {
                $query->where('source_type', 'product')->where('status', 1);
            }])
            ->get();

        // Fetch unique colors from product variations
        $colors = ProductVariation::whereHas('product', function ($query) {
                $query->where('source_type', 'product')->where('status', 1);
            })
            ->whereNotNull('color')
            ->select('color')
            ->distinct()
            ->pluck('color');

        // Max price from source_price
        $maxSourcePrice = Product::where('source_type', 'product')
            ->where('status', 1)
            ->max('source_price');

        // Max price from first variation (for products where source_price is null)
        $maxVariationPrice = ProductVariation::selectRaw('MAX(CAST(JSON_UNQUOTE(JSON_EXTRACT(data, "$[0].price")) AS DECIMAL)) as max_price')
            ->whereHas('product', function ($query) {
                $query->where('source_type', 'product')->where('status', 1);
            })
            ->pluck('max_price')
            ->first();

        // Take the greater of the two
        $maxPrice = max($maxSourcePrice ?? 0, $maxVariationPrice ?? 0);

        $authId  = Auth::id();

        // --- Return View with all data ---
        return view('frontend.pages.products.index', compact('data', 'categories', 'highlighted', 'shops', 'colors', 'authId', 'maxPrice'));
    }

    public function servicelist(Request $request): JsonResponse | View
    {
        $language_id = getLanguageId(app()->getLocale());

        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('services'),
            'header' => __('services'),
        ];

        $data['banner_visible'] = 1;
        $data['language_id'] = $language_id;
        $data['currency_code'] = getDefaultCurrencySymbol();

        $bestSellerCount = Product::join(
            DB::raw('(SELECT product_id, SUM(service_qty) as total_sold
                FROM bookings
                WHERE booking_status = 5
                GROUP BY product_id
                ORDER BY total_sold DESC
                LIMIT 5) as bp'), 'products.id', '=', 'bp.product_id')
            ->where('products.source_type', 'service')
            ->where('products.status', 1)
            ->where('products.language_id', $language_id)
            ->whereHas('shop')
            ->count();

        $highlighted = [
            'all_services' =>  [
                'title' => __('All Services'),
                'count' => Product::whereHas('shop')
                    ->where('source_type', 'service')
                    ->where('status', 1)
                    ->where('language_id', $language_id)
                    ->count(),
            ],
            'best_seller' =>  [
                'title' => __('best_seller'),
                'count' => $bestSellerCount,

            ],
            'featured' => [
                'title' => __('featured'),
                'count' => Product::where('featured', 1)
                    ->whereHas('shop')
                    ->where('source_type', 'service')
                    ->where('status', 1)
                    ->where('language_id', $language_id)
                    ->count(),
            ]
        ];

        $categories = Categories::where(['status' => 1, 'language_id' => $language_id, 'parent_id' => 0, 'source_type' => 'service'])
            ->withCount(['products' => function ($query) {
                $query->where('source_type', 'service')->where('status', 1)
                    ->whereHas('shop');
            }])
            ->get();

        $shops = DB::table('shops as s')
            ->leftJoin('products as p', function ($join) {
                $join->on('p.shop_id', '=', 's.id')
                    ->whereNull('p.deleted_at')
                    ->where('p.status', 1)
                    ->where('p.source_type', 'service');
            })
            ->select(
                's.id',
                's.shop_name',
                DB::raw('COUNT(p.id) as products_count')
            )
            ->where('s.status', 1)
            ->whereNull('s.deleted_at')
            ->where('s.language_id', $language_id)
            ->groupBy('s.id', 's.shop_name', 's.created_at')
            ->get();

        $vendors_locations = Product::query()
            ->whereHas('shop')
            ->where([
                'products.source_type' => 'service',
                'products.language_id' => $language_id
            ])
            ->join('user_details', 'products.user_id', '=', 'user_details.user_id')
            ->join('countries', 'user_details.country', '=', 'countries.id')
            ->select(
                'countries.id',
                'countries.name',
                'user_details.user_id',
                'user_details.first_name',
                'user_details.last_name',
                DB::raw('COUNT(products.id) as product_count')
            )
            ->where('products.status', 1)
            ->whereNull('products.deleted_at')
            ->whereNull('user_details.deleted_at')
            ->groupBy(
                'countries.id',
                'countries.name',
                'user_details.user_id',
                'user_details.first_name',
                'user_details.last_name'
            )
            ->get();

        $authId  = Auth::id();

        return view('frontend.pages.services.index', compact('categories', 'vendors_locations', 'data', 'highlighted', 'shops', 'authId'));
    }

    public function productdetail(Request $request): JsonResponse | View
    {
        session(['link' => url()->current()]);

        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('product_details'),
            'header' => __('product_details'),
        ];

        $product = Product::select('id', 'slug', 'source_name')
            ->where('slug', '=', $request->slug)
            ->firstOrFail();

        $data['banner_visible'] = 0;

        $cartItems = session('cart', []);
        $productIds = array_column($cartItems, 'product_id');
        $cartProducts = Product::whereIn('id', $productIds)
            ->where('status', 1)
            ->get()
            ->keyBy('id');

        $inCart = isset($cartProducts[$product->id]);

        return view('frontend.pages.products.productdetail', compact('data', 'product', 'inCart'));
    }

    public function servicedetail(Request $request): JsonResponse | View
    {
        session(['link' => url()->current()]);

        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('service_details'),
            'header' => __('service_details'),
        ];

        $products = Product::query()
            ->where('slug', '=', $request->slug)
            ->firstOrFail();

        $data['banner_visible'] = 0;
        return view('frontend.pages.services.servicedetail', compact('data', 'products'));
    }

    public function editservice(Request $request): View | JsonResponse
    {
        $products = Product::query()->where('id', '=', $request->id)->firstOrFail();
        if ($products) {
            $includedServices = explode(',', $products->include);
        } else {
            $includedServices = [];
        }

        $products_details = Productmeta::query()->where('product_id', '=', $products->id)->where('source_key', '=', 'product_image')->first();
        $c = "";
        $d = $products->id;
        $products_details1 = Productmeta::where(function ($query) {
            $query->where('source_key', '=', 'Fixed')
                ->orWhere('source_key', '=', 'Hourly')
                ->orWhere('source_key', '=', 'Squre-metter')
                ->orWhere('source_key', '=', 'Minute')
                ->orWhere('source_key', '=', 'Minitue');
        })->where(function ($query) use ($c, $d) {
            $query->where('product_id', '=', $d);
        })->first();

        $products_details2 = Productmeta::query()->where('product_id', '=', $products->id)->where('source_key', '=', 'service_name')->first();
        $products_details3 = Productmeta::query()->where('product_id', '=', $products->id)->where('source_key', '=', 'service_price')->first();
        $products_details4 = Productmeta::query()->where('product_id', '=', $products->id)->where('source_key', '=', 'service_desc')->first();
        $user_details = UserDetail::query()->where('user_id', '=', $products->user_id)->first();

        $product_id = $products->id;
        $productImages = Productmeta::where('product_id', $product_id)
            ->where('source_key', 'product_image')
            ->pluck('source_Values', 'id')
            ->map(function ($imagePath) {
                return 'storage/' . $imagePath;
            });
        $productServices = AdditionalService::where('service_id', $product_id)
            ->get(['name', 'price', 'duration', 'image']);
        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            return response()->json(['code' => "200", 'message' => __('Page details retrieved successfully.'), 'data' => $products], 200);
        }
        $subCategories = Categories::select('id', 'name')->where('id', $products->source_subcategory)->get();
        $addservices = AdditionalService::select('id', 'service_id', 'name', 'price', 'duration', 'image')->where('service_id', $products->id)->get();

        $location = Product::select('country', 'state', 'city')
            ->where('id', '=', $request->id)
            ->first();

        return view('admin.editservice', compact('products', 'subCategories', 'addservices', 'productImages', 'location'));
    }

    public function addComments(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'review' => 'required',
        ], [
            'review.required' => __('Review is required.'),
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'code' => 422,
                'errors' => $validator->messages()->toArray(),
            ], 422);
        }

        $userId = Auth::id() ?? $request->user_id;
        $parentId = $request->parent_id ?? 0;

        $isBooking = isBookingCompleted($request->product_id, $userId);

        if ($parentId == 0) {
            if (!$isBooking) {
                return response()->json([
                    'status' => 'error',
                    'code' => 403,
                    'message' => __('You can only leave a review after the order for this service is completed.')
                ], 403);
            }
        } else {
            $productUserId = getProductUserId($request->product_id);
            if (!$isBooking && ($productUserId != $userId)) {
                return response()->json([
                    'status' => 'error',
                    'code' => 403,
                    'message' => __('You can only leave a reply after the order for this service is completed.'),
                ], 403);
            }
        }

        $data = [
            'review' => $request->review,
            'rating' => $request->rating,
            'parent_id' => $parentId,
            'product_id' => $request->product_id,
            'user_id' => $userId,
            'review_date' => Carbon::now(),
        ];

        try {
            DB::table('ratings')->insert($data);

            return response()->json([
                'code' => 200,
                'message' => __('Review added successfully.'),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'code' => 500,
                'message' => __('Error! while adding review'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function listComments(Request $request): JsonResponse | View
    {
        $product_id = $request->product_id;
        $limit = $request->per_page ?? 10;
        $offset = $request->skip ?? 0;

        try {

            $starRating = DB::table('ratings')
                ->selectRaw('rating, COUNT(*) as count')
                ->where(['product_id' => $product_id, 'parent_id' => 0])
                ->whereNull('deleted_at')
                ->groupBy('rating')
                ->pluck('count', 'rating');

            $starCount = [
                'star5' => $starRating->get(5, 0),
                'star4' => $starRating->get(4, 0),
                'star3' => $starRating->get(3, 0),
                'star2' => $starRating->get(2, 0),
                'star1' => $starRating->get(1, 0),
            ];

            $ratingCount = DB::table('ratings')
                ->selectRaw('COUNT(*) as total_count, ROUND(AVG(rating), 1) as average_rating')
                ->where(['product_id' => $product_id, 'parent_id' => 0])
                ->whereNull('deleted_at')
                ->first();

            $rating_counts = [
                'total_count' => $ratingCount->total_count,
                'avg_rating' => $ratingCount->average_rating,
                'star_count' => $starCount,
            ];

            $ratings = DB::table('ratings')
                ->leftJoin('user_details', 'ratings.user_id', '=', 'user_details.user_id')
                ->select(
                    'ratings.id',
                    'ratings.rating',
                    'ratings.review',
                    'user_details.first_name',
                    'user_details.last_name',
                    'ratings.review_date',
                    'user_details.profile_image',
                )
                ->where('ratings.product_id', $product_id)
                ->where('ratings.parent_id', 0)
                ->whereNull('ratings.deleted_at')
                ->orderBy('ratings.id', 'DESC')
                ->offset($offset)
                ->limit($limit)
                ->get()
                ->map(function ($rating) {
                    $rating->profile_image = $rating->profile_image && file_exists(public_path('storage/profile/' . $rating->profile_image))
                        ? url('storage/profile/' . $rating->profile_image) : url('assets/img/profile-default.png');
                    $rating->review_date = Carbon::parse($rating->review_date)->diffForHumans();
                    $fullName = $rating->first_name . ' ' . $rating->last_name;
                    $rating->name = ucwords($fullName);
                    $rating->replies = $this->fetchReplies($rating->id);
                    return $rating;
                });

            $ratingsData = [
                'rating_counts' => $rating_counts,
                'ratings' => $ratings
            ];

            return response()->json([
                'code' => 200,
                'message' => __('Comments retrieved successfully.'),
                'data' => $ratingsData
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'code' => 500,
                'message' => __('Error! while retrieving comments'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    function fetchReplies($parentId)
    {
        return DB::table('ratings')
            ->leftJoin('user_details', 'ratings.user_id', '=', 'user_details.user_id')
            ->select(
                'ratings.id',
                'ratings.review',
                'user_details.first_name',
                'user_details.last_name',
                'ratings.review_date',
                'user_details.profile_image',
            )
            ->where('ratings.parent_id', $parentId)
            ->whereNull('ratings.deleted_at')
            ->get()
            ->map(function ($reply) {
                $reply->profile_image = $reply->profile_image && file_exists(public_path('storage/profile/' . $reply->profile_image))
                    ? url('storage/profile/' . $reply->profile_image) : url('assets/img/profile-default.png');
                $reply->review_date = Carbon::parse($reply->review_date)->diffForHumans();
                $fullName = $reply->first_name . ' ' . $reply->last_name;
                $reply->name = ucwords($fullName);
                $reply->replies = $this->fetchReplies($reply->id);
                return $reply;
            });
    }

    public function getReviewList(Request $request): JsonResponse
    {
        $userId = $request->user_id ?? '';
        $perPage = $request->get('per_page', 7);
        $page = $request->get('page', 1);

        try {
            $query = Product::query()
                ->where('products.source_type', 'service')
                ->join('products_meta', function ($join) {
                    $join->on('products.id', '=', 'products_meta.product_id')
                        ->where('products_meta.source_key', '=', 'product_image');
                })
                ->join('ratings', function ($join) {
                    $join->on('products.id', '=', 'ratings.product_id')
                        ->where(['ratings.parent_id' => 0, 'ratings.deleted_at' => NULL]);
                })
                ->join('user_details', 'ratings.user_id', '=', 'user_details.user_id')
                ->select(
                    'ratings.id',
                    'ratings.review',
                    'ratings.rating',
                    'ratings.review_date',
                    'products.source_name as service_name',
                    DB::raw('MIN(products_meta.source_values) as service_image'),
                    'user_details.profile_image',
                    DB::raw('CONCAT(user_details.first_name, " ", user_details.last_name) as full_name')
                );

            if ($userId) {
                $query->where('products.user_id', $userId);
            }

            $services = $query->orderBy('ratings.id', 'DESC')
                ->groupBy(
                    'ratings.id',
                    'ratings.review',
                    'ratings.rating',
                    'ratings.review_date',
                    'products.source_name',
                    'user_details.profile_image',
                    'user_details.first_name',
                    'user_details.last_name'
                )
                ->paginate($perPage);

            $dateformatSetting = GlobalSetting::where('key', 'date_format_view')->first();
            $timeformatSetting = GlobalSetting::where('key', 'time_format_view')->first();
            $timezoneSetting = GlobalSetting::where('key', 'timezone_format_view')->first();

            $dateFormat = $dateformatSetting->value ?? 'Y-m-d';
            $timeFormat = $timeformatSetting->value ?? 'H:i:s';
            $timezone = $timezoneSetting->value ?? 'Asia/Kolkata';

            $services->getCollection()->transform(function ($service) use ($dateFormat, $timeFormat, $timezone) {
                $service->service_image = url('storage/' . $service->service_image);
                $service->profile_image = file_exists(public_path('storage/profile/' . $service->profile_image)) ? url('storage/profile/' . $service->profile_image) : url('assets/img/profile-default.png');
                if ($service->review_date) {
                    $service->review_date = formatDateTime($service->review_date, true);
                }
                return $service;
            });

            return response()->json([
                'code' => 200,
                'message' => __('Reviews retrieved successfully.'),
                'data' => $services
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'code' => 500,
                'message' => __('Error! while retrieving reviews'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    protected function isInvalidGoogleMapKey(string $url): bool
    {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $httpCode !== 200 || str_contains($response, 'Maps Platform rejected your request');
    }
}
