<?php

namespace App\Repositories\Eloquent;

use App\Models\Gigs;
use App\Repositories\Contracts\ReviewRepositoryInterface;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\Product\app\Models\Product;
use Modules\Product\app\Models\Rating;

class ReviewRepository implements ReviewRepositoryInterface
{
   /**
     * Get product ratings for the authenticated provider with "load more" pagination.
     */
    public function getProviderProductRatings(Request $request): JsonResponse
    {
        $provider = Auth::user();
        $ratingsPerPage = 4;

        // Get IDs of products owned by the provider
        $productIds = Product::where('user_id', $provider->id)->pluck('id');

        $ratings = Rating::with([
            'user.userDetail:id,user_id,first_name,last_name,profile_image',
            'product:id,source_name',
            'reply.user.userDetail:id,user_id,first_name,last_name,profile_image'
        ])
        ->whereIn('product_id', $productIds)
        ->where(function ($query) {
            $query->whereNull('parent_id')->orWhere('parent_id', 0);
        })
        ->orderBy('created_at', 'desc')
        ->paginate($ratingsPerPage);

        // Format data for a clean frontend response
        $formattedData = $ratings->getCollection()->map(function ($rating) use ($provider) {
            $customer = $rating->user;
            $customerName = $customer->userDetail
                ? trim($customer->userDetail->first_name . ' ' . $customer->userDetail->last_name)
                : $customer->name;

            $customerProfileImage = uploadedAsset(
                $customer->userDetail->profile_image ?? null,
                'profile'
            );

            $firstReply = $rating->reply->where('user_id', $provider->id)->first();
            $replyData = $firstReply ? [
                'id' => $firstReply->id,
                'review_date' => Carbon::parse($firstReply->created_at)->diffForHumans(),
                'profile_image' => uploadedAsset($firstReply->user->userDetail->profile_image ?? null, 'profile'),
                'review' => $firstReply->review ?? '',
            ] : null;

            return [
                'id' => $rating->id,
                'rating' => (float)$rating->rating,
                'review' => $rating->review,
                'review_date' => Carbon::parse($rating->created_at)->diffForHumans(),
                'customer' => [
                    'name' => $customerName,
                    'profile_image' => $customerProfileImage,
                ],
                'product' => [
                    'id' => $rating->product->id,
                    'title' => $rating->product->source_name,
                ],
                'reply' => $replyData,
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $formattedData,
            'pagination' => [
                'has_more_pages' => $ratings->hasMorePages(),
            ]
        ]);
    }

    /**
     * Add a provider's reply to a product rating.
     */
    public function addProviderProductRatingReply(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'parent_id' => 'required|exists:ratings,id',
            'reply_text' => 'required|string|min:3|max:1000',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'message' => 'Validation failed', 'errors' => $validator->errors()], 422);
        }

        $provider = Auth::user();
        $parentRating = Rating::with('product')->find($request->parent_id);

        // Authorization: Ensure the provider owns the product being reviewed
        if ($parentRating->product->user_id !== $provider->id) {
            return response()->json(['status' => 'error', 'message' => 'You are not authorized to reply to this review.'], 403);
        }

        // Prevent multiple replies
        if (Rating::where('parent_id', $parentRating->id)->exists()) {
            return response()->json(['status' => 'error', 'message' => 'A reply already exists for this review.'], 409);
        }

        Rating::create([
            'user_id' => $provider->id,
            'product_id' => $parentRating->product_id,
            'parent_id' => $parentRating->id,
            'review' => $request->reply_text,
            'rating' => null, // Replies do not have a rating
            'review_date' => now(),
        ]);

        return response()->json(['status' => 'success', 'message' => 'Your reply has been posted successfully.']);
    }

    /**
     * Delete a product rating and its associated reply.
     */
    public function deleteProviderProductRating(int $ratingId): JsonResponse
    {
        $provider = Auth::user();
        $rating = Rating::with('product')->find($ratingId);

        if (!$rating) {
            return response()->json(['status' => 'error', 'message' => 'Review not found.'], 404);
        }

        // Authorization: Ensure the provider owns the product associated with the review
        if ($rating->product->user_id !== $provider->id) {
            return response()->json(['status' => 'error', 'message' => 'You are not authorized to delete this review.'], 403);
        }

        // Use a transaction to ensure both review and its reply are deleted
        DB::transaction(function () use ($rating) {
            Rating::where('parent_id', $rating->id)->delete(); // Delete reply first
            $rating->delete(); // Then delete the main review
        });

        return response()->json(['status' => 'success', 'message' => 'Review has been deleted successfully.']);
    }

    public function storeCustomerReview(Request $request): JsonResponse
    {
        $userId = currentUser()->id ?? $request->user_id;
        $id = $request->id ?? '';
        $validator = Validator::make($request->all(), [
            'review' => ['required', 'min:10']
        ], [
            'review.required' => __('review_required'),
            'review.min' => __('review_minlength'),
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'code' => 422,
                'errors' => $validator->errors()->toArray(),
            ], 422);
        }

        $response = [];
        $successMessage = $id ? __('review_update_success') : __('review_create_success');
        $errorMessage = $id ? __('default_update_error') : __('default_create_error');

        try {
            if ($id) {
                Rating::where('id', $id)->update([
                    'review' => $request->review,
                    'rating' => $request->rating ?? 0,
                    'review_date' => now(),
                ]);
            } else {

                if ($request->type == 'service') {
                    $reviewNotAllowedMessage = __('service_review_not_allowed');
                } else {
                    $reviewNotAllowedMessage = __('product_review_not_allowed');
                }

                if (! isBookingCompleted($request->product_id, $request->type, $userId)) {
                    return response()->json([
                        'status' => 'error',
                        'code' => 403,
                        'message' => $reviewNotAllowedMessage,
                    ], 403);
                } else {
                    Rating::create([
                        'product_id' => $request->product_id,
                        'user_id' => $userId,
                        'type' => $request->type,
                        'rating' => $request->rating ?? 0,
                        'review' => $request->review,
                        'review_date' => now(),
                    ]);
                }
            }

            $response = response()->json([
                'status' => 'success',
                'code' => 200,
                'message' => $successMessage,
            ]);
        } catch (\Exception $e) {
            $response = response()->json([
                'status' => 'error',
                'code' => 500,
                'message' => $errorMessage,
                'error' => $e->getMessage(),
            ], 500);
        }

        return $response;
    }

    public function storeCustomerReply(Request $request): JsonResponse
    {
        $userId = currentUser()->id ?? $request->user_id;
        $response = [];
        $validator = Validator::make($request->all(), [
            'reply_message' => ['required', 'min:3'],
            'review_id' => ['required'],
            'product_id' => ['required'],
            'type' => ['required', 'in:product,service,shop'],
        ], [
            'reply_message.required' => __('reply_message_required'),
            'reply_message.min' => __('reply_message_minlength'),
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'code' => 422,
                'errors' => $validator->errors()->toArray(),
            ], 422);
        }

        if ($request->type == 'service') {
            $replyNotAllowedMessage = __('service_reply_not_allowed');
        } else {
            $replyNotAllowedMessage = __('product_reply_not_allowed');
        }

        $isBooking = isBookingCompleted($request->product_id, $request->type, $userId);
        $productId = getProductUserId($request->product_id);

        if (! $isBooking && $userId !== $productId) {
            return response()->json([
                'status' => 'error',
                'code' => 403,
                'message' => $replyNotAllowedMessage,
            ], 403);
        }

        try {
            Rating::create([
                'parent_id' => $request->review_id,
                'user_id' => $userId,
                'product_id' => $request->product_id,
                'review' => $request->reply_message,
                'type' => $request->type,
                'review_date' => now(),
            ]);

            $response = response()->json([
                'status' => 'success',
                'code' => 200,
                'message' => __('reply_create_success'),
            ]);
        } catch (\Exception $e) {
            $response = response()->json([
                'status' => 'error',
                'code' => 500,
                'message' => __('default_create_error'),
                'error' => $e->getMessage(),
            ], 500);
        }

        return $response;
    }

    public function frontendReviewsList(Request $request): JsonResponse
    {
        try {
            $productId = $request->product_id;
            $shopId = $request->shop_id;
            $perPage = $request->get('per_page', 3);
            $productIds = [];

            if ($shopId) {
                $productIds = Product::where('shop_id', $shopId)->pluck('id')->toArray();
            } else {
                $productIds = $productId ? [$productId] : [];
            }

            $reviewsQuery = $this->buildReviewsQuery($productIds);
            $paginatedReviews = $reviewsQuery->paginate($perPage);

            $reviewsData = collect($paginatedReviews->items())
                ->map(fn (Rating $review) => $this->transformReview($review));

            $reviewsMeta = $this->calculateStarRatings($productIds);

            $finalData = [
                'reviews_meta' => [
                    'avg_ratings' => number_format($reviewsMeta['avg_ratings'], 1, '.', ''),
                    'total_reviews' => $reviewsMeta['total_reviews'],
                    'star_ratings_count' => $reviewsMeta['star_ratings_count'],
                    'star_ratings_percentage' => $reviewsMeta['star_ratings_percentage'],
                ],
                'reviews' => $reviewsData,
                'current_page' => $paginatedReviews->currentPage(),
                'next_page' => $paginatedReviews->hasMorePages() ? $paginatedReviews->currentPage() + 1 : null,
            ];

            return response()->json([
                'status' => 'success',
                'code' => 200,
                'data' => $finalData,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'code' => 500,
                'message' => __('default_retrieve_error'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function cutomerReviewsList(Request $request): JsonResponse
    {
        try {
            $userId = currentUser()->id ?? $request->user_id;
            $perPage = $request->get('per_page', 5);

            $reviewsQuery = $this->buildReviewsQuery([], $userId);
            $paginatedReviews = $reviewsQuery->paginate($perPage);

            $reviewsData = collect($paginatedReviews->items())
                ->map(fn (Rating $review) => $this->transformReview($review));

            $finalData = [
                'reviews' => $reviewsData,
                'current_page' => $paginatedReviews->currentPage(),
                'next_page' => $paginatedReviews->hasMorePages() ? $paginatedReviews->currentPage() + 1 : null,
            ];

            return response()->json([
                'status' => 'success',
                'code' => 200,
                'data' => $finalData,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'code' => 500,
                'message' => __('default_retrieve_error'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    private function buildReviewsQuery(array $productIds = [], ?int $userId = null): Builder
    {
        if (!$userId && empty($productIds)) {
            return Rating::query()->whereRaw('1=0');
        }

        $query = Rating::select('id', 'user_id', 'product_id', 'parent_id', 'rating', 'review', 'created_at', 'review_date', 'type')
            ->with([
                'user:id,name',
                'user.userDetail:id,user_id,first_name,last_name,profile_image',
                'replies' => function ($query) {
                    $query->select('id', 'user_id', 'product_id', 'parent_id', 'review', 'created_at', 'review_date')
                        ->with([
                            'user:id,name',
                            'user.userDetail:id,user_id,first_name,last_name,profile_image',
                        ]);
                },
                'product:id,source_name,user_id,shop_id',
                'product.shop:id,shop_name,shop_logo',
            ])
            ->where('parent_id', 0)
            ->whereHas('product')
            ->when($userId, fn($q) => $q->where('user_id', $userId))
            ->when($productIds, fn($q) => $q->whereIn('product_id', $productIds))
            ->orderBy('created_at', 'desc');

        return $query;
    }

    private function transformReview(Rating $review): Rating
    {
        $review->review_date = Carbon::parse($review->review_date)->diffForHumans();

        /** @var \App\Models\User|null $user */
        $user = $review->user;

        if ($user && $user->userDetail) {
            /** @var \App\Models\UserDetail $detail */
            $detail = $user->userDetail;

            $user->name = $detail->first_name
                ? ucwords($detail->first_name . ' ' . $detail->last_name)
                : ucwords($user->name);

            $user->profile_image = uploadedAsset('profile/' . $detail->profile_image ?? '', 'profile');
            unset($user->userDetail);
        }

        $review->rating = number_format($review->rating, 1, '.', '');
        $review->product_name = ucfirst($review->product->source_name ?? '');
        $review->type = ucfirst($review->type);

        if (!empty($review->product->shop)) {
            $shop = $review->product->shop;
            $shop->shop_name = ucwords($shop->shop_name ?? '');
            $shop->shop_logo = uploadedAsset($shop->shop_logo ?? '');
        }

        unset($review->created_at);
        $review->replies = $review->replies->map(fn (Rating $reply) => $this->transformReply($reply));

        return $review;
    }

    private function transformReply(Rating $reply): Rating
    {
        $reply->reply_date = Carbon::parse($reply->review_date)->diffForHumans();

        if ($reply->user && $reply->user->userDetail) {
            $rd = $reply->user->userDetail;
            $reply->user->name = $rd->first_name
                ? ucwords($rd->first_name . ' ' . $rd->last_name)
                : ucwords($reply->user->name);

            $reply->user->profile_image = uploadedAsset( 'profile/' . $rd->profile_image ?? '', 'profile');
            unset($reply->user->userDetail);
        }

        unset($reply->created_at);

        return $reply;
    }

    /**
     * @return array{
     *     avg_ratings: float,
     *     total_reviews: int,
     *     star_ratings_count: array<string, int>,
     *     star_ratings_percentage: array<string, string>
     * }
     */
    private function calculateStarRatings(array $productIds): array
    {
        $avgRatings = Rating::whereIn('product_id', $productIds)->avg('rating');
        $totalReviews = Rating::whereIn('product_id', $productIds)->where('parent_id', 0)->count();

        $starRatings = Rating::whereIn('product_id', $productIds)
            ->where('parent_id', 0)
            ->selectRaw('rating, COUNT(*) as count')
            ->groupBy('rating')
            ->pluck('count', 'rating')
            ->toArray();

        $starRatingsCount = [];
        $starRatingsPercentage = [];

        foreach (range(1, 5) as $star) {
            $count = $starRatings[$star] ?? 0;
            $starRatingsCount["{$star}_star"] = $count;

            $percentage = $totalReviews > 0 ? number_format($count / $totalReviews * 100, 0) : 0;
            $starRatingsPercentage["{$star}_star"] = $percentage;
        }

        return [
            'avg_ratings' => $avgRatings ? floatval($avgRatings) : 0,
            'total_reviews' => $totalReviews,
            'star_ratings_count' => $starRatingsCount,
            'star_ratings_percentage' => $starRatingsPercentage,
        ];
    }

    public function deleteReview(Request $request): JsonResponse
    {
        $reviewId = $request->id;

        $review = Rating::where('id', $reviewId)
            ->where('parent_id', 0)
            ->first();

        if (! $review) {
            return response()->json([
                'status' => 'error',
                'code' => 404,
                'message' => __('review_not_found'),
            ], 404);
        }

        try {
            DB::transaction(function () use ($review) {
                Rating::where('parent_id', $review->id)->delete(); // Delete replies first
                $review->delete(); // Then delete the main review
            });

            return response()->json([
                'status' => 'success',
                'code' => 200,
                'message' => __('review_delete_success'),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'code' => 500,
                'message' => __('default_delete_error'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
