<?php

namespace App\Repositories\Eloquent;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\DB;
use Modules\GlobalSetting\app\Models\Currency;
use App\Models\Bookings;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Modules\Product\app\Models\Product;
use Modules\Product\app\Models\Productmeta;
use Modules\Service\app\Models\Service;
use App\Repositories\Contracts\ProviderRepositoryInterface;
use Modules\Categories\app\Models\Categories;
use Modules\Service\app\Models\Servicemeta;
use App\Models\BookingProduct;

class ProviderRepository implements ProviderRepositoryInterface
{
    public function index()
    {
        $currentRouteName = Route::currentRouteName();
        if ($currentRouteName == 'admin.providerslist') {
            $title = 'Providers';
        } else {
            $title = 'Users';
        }
        return [
            'title' => $title,
        ];
    }

    public function getDashboardSummary(Request $request)
    {
        $providerId = $request->input('user_id');
        if (!$providerId) {
            return [
                'success' => false,
                'message' => 'Provider ID not provided.'
            ];
        }

        $currency = Currency::where('is_default', 1)->first()->symbol ?? '$';

        // --- Overall service stats ---
        $overallServiceStats = Bookings::where('type', 'service')
            ->where('provider_id', $providerId)
            ->where('booking_status', 5)
            ->selectRaw('COUNT(id) as count, SUM(total_amount) as total')
            ->first();

        // --- Overall product stats ---
        $overallProductStats = \DB::table('booking_products')
            ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id')
            ->where('booking_products.provider_id', $providerId)
            ->where('booking_products.product_status', 4)
            ->selectRaw('COUNT(DISTINCT bookings.id) as count, SUM(booking_products.gross_total) as total')
            ->first();

        // Safely get counts and totals
        $serviceCount = $overallServiceStats->count ?? 0;
        $productCount = $overallProductStats->count ?? 0;
        $overallCount = $serviceCount + $productCount;

        $serviceTotal = $overallServiceStats->total ?? 0;
        $productTotal = $overallProductStats->total ?? 0;
        $overallTotal = $serviceTotal + $productTotal;

        // --- Function to get stats between dates ---
        $getStatsByDate = function ($startDate, $endDate) use ($providerId) {
            // Service stats
            $serviceStats = Bookings::where('type', 'service')
                ->where('provider_id', $providerId)
                ->whereIn('booking_status', [5, 6])
                ->whereBetween('created_at', [$startDate, $endDate])
                ->selectRaw('COUNT(id) as count, SUM(total_amount) as total')
                ->first();

            // Product stats
            $productStats = \DB::table('booking_products')
                ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id')
                ->where('booking_products.provider_id', $providerId)
                ->whereIn('bookings.booking_status', [5, 6])
                ->whereBetween('bookings.created_at', [$startDate, $endDate])
                ->selectRaw('COUNT(DISTINCT bookings.id) as count, SUM(booking_products.subtotal) as total')
                ->first();

            return [
                'count' => ($serviceStats->count ?? 0) + ($productStats->count ?? 0),
                'total' => ($serviceStats->total ?? 0) + ($productStats->total ?? 0),
            ];
        };

        // --- Percentage calculation ---
        $calculatePercentageChange = fn($current, $previous) =>
            $previous == 0 ? ($current > 0 ? 100.0 : 0.0) : (($current - $previous) / $previous) * 100;

        $currentMonthStats = $getStatsByDate(Carbon::now()->startOfMonth(), Carbon::now()->endOfMonth());
        $lastMonthStats = $getStatsByDate(Carbon::now()->subMonth()->startOfMonth(), Carbon::now()->subMonth()->endOfMonth());

        $currentMonthCount = $currentMonthStats['count'] ?? 0;
        $lastMonthCount = $lastMonthStats['count'] ?? 0;
        $currentMonthTotal = $currentMonthStats['total'] ?? 0;
        $lastMonthTotal = $lastMonthStats['total'] ?? 0;

        return [
            'currency' => $currency,
            'completed_bookings' => [
                'count' => $overallCount,
                'percentage' => $calculatePercentageChange($currentMonthCount, $lastMonthCount)
            ],
            'total_sales' => [
                'count' => $overallCount,
                'percentage' => $calculatePercentageChange($currentMonthCount, $lastMonthCount)
            ],
            'total_earnings' => [
                'amount' => $overallTotal,
                'percentage' => $calculatePercentageChange($currentMonthTotal, $lastMonthTotal)
            ]
        ];
    }

    public function getRecentlySoldProducts(Request $request)
    {
        $providerId = $request->input('user_id');
        $currency = Currency::where('is_default', 1)->first()->symbol ?? '$';

        // Step 1 & 2: Get booking_products of type product with provider filter
        $bookingProducts = BookingProduct::where('product_status', 4)
            ->where('provider_id', $providerId)
            ->whereHas('booking', fn($q) => $q->where('type', 'product'))
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        $products = collect();

        foreach ($bookingProducts as $bp) {
            $product = Product::with(['category'])
                ->select(
                    'products.*',
                    DB::raw('(SELECT source_values
                            FROM products_meta
                            WHERE product_id = products.id
                                AND deleted_at IS NULL
                                AND source_key = "product_image"
                            LIMIT 1) as image')
                )
                ->find($bp->product_id);

            if ($product) {
                $product->formatted_price = $currency . number_format($product->source_price, 2);
                $product->image_url = uploadedAsset($product->image, 'default2');

                $product->category_name = $product->category?->name ?? null;

                $products->push($product);
            }
        }

        // Sort by most recent booking_products created_at
        $products = $products->sortByDesc(fn($p) => $p->created_at)->take(5)->values();

        return $products;
    }


    public function getTotalBookingsChartData(Request $request)
    {
        $providerId = $request->input('user_id');

        $firstBooking = Bookings::where('provider_id', $providerId)->orderBy('created_at', 'asc')->first();
        if (!$firstBooking) return ['labels' => [], 'data' => []];

        $startDate = Carbon::parse($firstBooking->created_at)->startOfMonth();
        $endDate = Carbon::now()->startOfMonth();

        $bookingsData = Bookings::where('provider_id', $providerId)
            ->select(DB::raw('COUNT(id) as count'), DB::raw("DATE_FORMAT(created_at, '%Y-%m') as month"))
            ->groupBy('month')->orderBy('month', 'asc')->get()->keyBy('month');

        $labels = []; $data = [];
        $currentDate = $startDate->copy();

        while ($currentDate->lessThanOrEqualTo($endDate)) {
            $monthKey = $currentDate->format('Y-m');
            $labels[] = $currentDate->format('M Y');
            $data[] = $bookingsData->has($monthKey) ? $bookingsData[$monthKey]->count : 0;
            $currentDate->addMonth();
        }
        return ['labels' => $labels, 'data' => $data];
    }

    public function getRecentlyBookedServices(Request $request)
    {
        $providerId = $request->input('user_id');
        $currency = Currency::where('is_default', 1)->first()->symbol ?? '$';

        $services = Bookings::select(
            'products.source_name as service_name',
            'users.name as customer_name',
            'bookings.created_at',
            'bookings.order_id',
            'bookings.id as booking_id',
            // Subquery to get a single image per product
            DB::raw('(SELECT source_values
                    FROM products_meta
                    WHERE product_id = products.id
                        AND source_key = "product_image"
                    LIMIT 1) as image')
        )
        ->join('products', 'bookings.product_id', '=', 'products.id')
        ->join('users', 'bookings.user_id', '=', 'users.id')
        ->where('bookings.provider_id', $providerId)
        ->where('bookings.type', 'service')
        ->whereIn('bookings.booking_status', [5, 6])
        ->orderByDesc('bookings.created_at')
        ->limit(5)
        ->get();

        $services->each(function($service) {
            $service->image_url = $service->image ? Storage::url($service->image) : asset('/assets/img/default-placeholder-image.png');
            $service->formatted_date = Carbon::parse($service->created_at)->format('d M Y');
            $service->booking_code = $service->order_id;
        });

        return $services;
    }



    public function getHighlyBookedCategories(Request $request)
    {
        $providerId = $request->input('user_id');

        // Subquery to count bookings per category
        $categoryCountsSubquery = Bookings::select(
                'products.source_category',
                DB::raw('COUNT(bookings.id) as booking_count')
            )
            ->join('products', 'bookings.product_id', '=', 'products.id')
            ->where('bookings.provider_id', $providerId)
            ->whereIn('bookings.booking_status', [5, 6])
            ->whereNotNull('products.source_category')
            ->groupBy('products.source_category');

        // Join with categories table to get category names
        $categories = DB::table('categories')->select('categories.name', 'sub.booking_count')
            ->joinSub($categoryCountsSubquery, 'sub', function($join) {
                $join->on('categories.id', '=', 'sub.source_category');
            })
            ->orderByDesc('sub.booking_count')
            ->limit(5)
            ->get();

        // Calculate percentage for each category
        $totalBookings = $categories->sum('booking_count');
        $categories->each(function($cat) use ($totalBookings) {
            $cat->percentage = ($totalBookings > 0) ? round(($cat->booking_count / $totalBookings) * 100) : 0;
        });

        return $categories;
    }

    public function getRecentTransactions(Request $request)
    {
        $currency = Currency::where('is_default', 1)->first()->symbol ?? '$';

        // Define payment type mapping
        $paymentTypes = [
            1  => 'PayPal',
            2  => 'Stripe',
            3  => 'Razorpay',
            4  => 'Bank Transfer',
            5  => 'COD',
            6  => 'Wallet',
            7  => 'Cashfree',
            8  => 'PayU',
            9  => 'Authorize.Net',
            10 => 'Paystack',
        ];

        // Fetch last 5 bookings (products or services)
        $bookings = Bookings::latest('created_at')->limit(5)->get();

        $bookings->each(function ($booking) use ($currency, $paymentTypes) {

            // Customer info
            $booking->customer_name = $booking->user->name ?? 'N/A';

            // Fetch user profile image from user_details table
            $profileImage = \DB::table('user_details')
                ->where('user_id', $booking->user_id)
                ->value('profile_image');
            $booking->customer_image_url = $profileImage
                ? Storage::url('profile/' . $profileImage)
                : asset('/assets/img/profile-default.png');

            // Invoice and formatting
            $booking->invoice_id = $booking->order_id;
            $booking->formatted_amount = $currency . number_format($booking->total_amount, 2);
            $booking->formatted_date = Carbon::parse($booking->created_at)->format('d M Y');

            // Map payment type
            $booking->payment_method = $paymentTypes[$booking->payment_type] ?? 'Unknown';

            $booking->item_type = ucfirst($booking->type);

            // Get description
            if ($booking->type === 'service') {
                $product = Product::find($booking->product_id);
            } elseif ($booking->type === 'product') {
                $bookingProduct = BookingProduct::where('booking_id', $booking->id)->first();
                $product = $bookingProduct ? Product::find($bookingProduct->product_id) : null;
            }

            $booking->description = $product->source_name ?? 'N/A';
        });

        return $bookings;
    }

}
