<?php

namespace App\Repositories\Eloquent;

use App\Models\User;
use App\Repositories\Contracts\ProviderEarningRepositoryInterface;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use App\Models\Bookings;
use App\Models\PayoutHistory;
use App\Models\WalletHistory;
use Modules\Product\app\Models\Product;
use Illuminate\Support\Facades\DB;

class ProviderEarningRepository implements ProviderEarningRepositoryInterface
{
    private const STORAGE_PATH = 'storage/';

    public function sellerEarningList(Request $request): JsonResponse
    {
        try {
            $type = $request->get('type', 'service'); // default to service
            $currentUserId = auth()->id();

            if (!$currentUserId) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'User not authenticated'
                ], 401);
            }

            $earningList = [];

            if ($type === 'service') {
                // Service bookings (single product per booking)
                $bookings = DB::table('bookings')
                    ->select([
                        'id',
                        'order_id as invoice_id',
                        'product_id',
                        'type',
                        'user_id',
                        'first_name',
                        'last_name',
                        'user_email',
                        'user_phone',
                        'total_amount as amount',
                        'payment_type as payment_method',
                        'created_at as payment_date',
                        'payment_status'
                    ])
                    ->where('provider_id', $currentUserId)
                    ->where('type', 'service')
                    ->whereNull('deleted_at')
                    ->orderBy('created_at', 'desc')
                    ->get();

                foreach ($bookings as $booking) {
                    // Fetch product info
                    $product = DB::table('products')->select('source_name')->where('id', $booking->product_id)->first();
                    $productName = $product ? $product->source_name : 'N/A';

                    // Fetch product image
                    $productImage = DB::table('products_meta')
                        ->select('source_values')
                        ->where('product_id', $booking->product_id)
                        ->whereNotNull('source_values')
                        ->where('source_values', '!=', '')
                        ->first();

                    $productImageUrl = '';
                    if ($productImage && $productImage->source_values) {
                        $sourceValues = $productImage->source_values;
                        if (str_starts_with(trim($sourceValues), '[')) {
                            $images = json_decode($sourceValues, true);
                            $productImageUrl = !empty($images) ? $images[0] : $sourceValues;
                        } else {
                            $productImageUrl = $sourceValues;
                        }
                    }

                    // Get user profile from user_details table
                    $userDetails = DB::table('user_details') ->select([ 'first_name', 'last_name', 'profile_image', ]) ->where('user_id', $booking->user_id) ->first(); // Prepare customer name
                    $customerName = '';
                    if ($userDetails) { $customerName = trim($userDetails->first_name . ' ' . $userDetails->last_name); } // If no user details found, use booking data
                    if (empty($customerName)) { $customerName = trim($booking->first_name . ' ' . $booking->last_name); }
                    $paymentMethods = [
                        1 => 'paypal', 2 => 'stripe', 3 => 'razorpay', 4 => 'BankTransfer',
                        5 => 'COD', 6 => 'Wallet', 7 => 'cashfree', 8 => 'payu', 9 => 'authorizenet', 10 => 'paystack'
                    ];

                    $paymentMethodName = $booking->payment_method && isset($paymentMethods[$booking->payment_method])
                        ? ucfirst($paymentMethods[$booking->payment_method])
                        : 'N/A';

                    $earningList[] = [
                        'id' => $booking->id,
                        'invoice_id' => $booking->invoice_id ?? 'N/A',
                        'customer' => $customerName ?: 'N/A',
                        'service' => $productName,
                        'service_image' => $productImageUrl,
                        'amount' => number_format((float)$booking->amount, 2),
                        'raw_amount' => (float)$booking->amount,
                        'payment_method' => $paymentMethodName,
                        'payment_date' => $booking->payment_date ? date('M d, Y', strtotime($booking->payment_date)) : 'N/A',
                        'payment_status' => ucfirst($booking->payment_status ?? 'pending'),
                        'type' => $booking->type,
                        'user_details' => $userDetails ? ['profile_image' => $userDetails->profile_image] : null,
                    ];
                }

            } else if ($type === 'product') {
                // Product bookings (multiple products per booking)
                $bookingProducts = DB::table('booking_products')
                    ->where('provider_id', $currentUserId)
                    ->where('product_status', '<', 5)
                    ->orderBy('created_at', 'desc')
                    ->get();

                foreach ($bookingProducts as $bp) {
                    $product = DB::table('products')->select('source_name')->where('id', $bp->product_id)->first();
                    $productName = $product ? $product->source_name : 'N/A';

                    $productMeta = DB::table('products_meta')
                        ->select('source_values')
                        ->where('product_id', $bp->product_id)
                        ->whereNotNull('source_values')
                        ->where('source_values', '!=', '')
                        ->first();

                    $productImageUrl = '';
                    if ($productMeta && $productMeta->source_values) {
                        $sourceValues = $productMeta->source_values;
                        if (str_starts_with(trim($sourceValues), '[')) {
                            $images = json_decode($sourceValues, true);
                            $productImageUrl = !empty($images) ? $images[0] : $sourceValues;
                        } else {
                            $productImageUrl = $sourceValues;
                        }
                    }

                    $userDetails = DB::table('user_details') ->select([ 'first_name', 'last_name', 'profile_image', ]) ->where('user_id', $bp->user_id) ->first(); // Prepare customer name
                    $customerName = '';
                    if ($userDetails) { $customerName = trim($userDetails->first_name . ' ' . $userDetails->last_name); } // If no user details found, use booking data
                    // If the constructed name is empty.
                    if (empty($customerName)) {
                        $customerName = 'N/A';
                    }
                    $paymentMethodName = 'N/A'; // For products, payment type comes from parent booking
                    $bookingInfo = DB::table('bookings')->select('payment_type', 'payment_status', 'order_id', 'created_at')->where('id', $bp->booking_id)->first();
                    if ($bookingInfo) {
                        $paymentMethods = [
                            1 => 'paypal', 2 => 'stripe', 3 => 'razorpay', 4 => 'BankTransfer',
                            5 => 'COD', 6 => 'Wallet', 7 => 'cashfree', 8 => 'payu', 9 => 'authorizenet', 10 => 'paystack'
                        ];
                        $paymentMethodName = isset($paymentMethods[$bookingInfo->payment_type]) ? ucfirst($paymentMethods[$bookingInfo->payment_type]) : 'N/A';
                    }

                    $earningList[] = [
                        'id' => $bp->id,
                        'invoice_id' => $bookingInfo->order_id ?? 'N/A',
                        'customer' => $customerName,
                        'service' => $productName,
                        'service_image' => $productImageUrl,
                        'amount' => number_format((float)$bp->subtotal, 2),
                        'raw_amount' => (float)$bp->subtotal,
                        'payment_method' => $paymentMethodName,
                        'payment_date' => isset($bookingInfo->created_at) ? date('M d, Y', strtotime($bookingInfo->created_at)) : 'N/A',
                        'payment_status' => isset($bookingInfo->payment_status) ? ucfirst($bookingInfo->payment_status) : 'pending',
                        'type' => 'product',
                        'user_details' => $userDetails ? ['profile_image' => $userDetails->profile_image] : null,
                    ];
                }
            }

            // Totals
            $totalEarnings = array_sum(array_column($earningList, 'raw_amount'));
            $totalBookings = count($earningList);

            return response()->json([
                'status' => 'success',
                'message' => 'Earnings retrieved successfully',
                'data' => [
                    'type' => $type,
                    'earnings' => $earningList,
                    'summary' => [
                        'total_earnings' => number_format($totalEarnings, 2),
                        'raw_total_earnings' => $totalEarnings,
                        'total_bookings' => $totalBookings
                    ]
                ]
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to retrieve earnings: ' . $e->getMessage()
            ], 500);
        }
    }

}
