<?php

namespace App\Repositories\Eloquent;

use App\Models\AdditionalSetting;
use App\Models\UserDetail;
use App\Repositories\Contracts\CustomerRepositoryInterface;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\GlobalSetting\app\Models\GeneralSetting;
use App\Models\BookingProduct;
use App\Models\Bookings;

class CustomerRepository implements CustomerRepositoryInterface
{
    public function getOrderDetails($orderId)
    {
        $orderDetails = BookingProduct::with([
            'booking', // will give customer info: first_name, last_name, etc.
            'product.meta',
            'product.categories',
            'product.shop'
        ])
        ->where('id', $orderId)
        ->first();

        if ($orderDetails) {
            $productTax = $orderDetails->tax_amount ?? 0;
            $couponDiscount = $orderDetails->coupon_discount_amount ?? 0;

            $orderDetails->booking->amount_tax = number_format($orderDetails->booking->amount_tax, 2, '.', '');
            $orderDetails->booking->total_coupon_discount_amount = number_format($orderDetails->booking->total_coupon_discount_amount, 2, '.', '');
            $orderDetails->booking->total_amount = number_format($orderDetails->booking->total_amount, 2, '.', '');

            $orderDetails->booking->product_tax = number_format($productTax, 2, '.', '');
            $payableAmount = ($orderDetails->subtotal - $couponDiscount) + $productTax;
            $orderDetails->booking->payable_amount = number_format($payableAmount, 2, '.', '');
        }

        return $orderDetails;
    }

    public function getBookingDetails($bookingId)
    {
        return Bookings::with([
            'user',               // customer info
            'provider',           // provider info
            'products.product.meta',       // product meta
            'products.product.categories', // product categories
            'products.product.shop',       // product shop
        ])
        ->where('id', $bookingId)
        ->first();
    }

    public function saveAccountSettings(Request $request): array
    {
        try {
            $id = currentUser()->id ?? $request->user_id;
            $setting = AdditionalSetting::where('user_id', $id)->where('key', 'account_settings')->first();
            $existingData = $setting ? json_decode($setting->value, true) : [];

            if ($request->filled(['paypal_id'])) {
                $existingData['paypal'] = $request->only(['paypal_id']);
            }

            if ($request->filled(['stripe_id'])) {
                $existingData['stripe'] = $request->only(['stripe_id']);
            }

            if ($request->filled(['account_holder_name', 'bank_name'])) {
                $existingData['bank_transfer'] = $request->only(['account_holder_name', 'bank_name', 'ifsc_code', 'account_number']);
            }

            AdditionalSetting::updateOrCreate(
                ['user_id' => $id, 'key' => 'account_settings'],
                ['value' => json_encode($existingData, JSON_UNESCAPED_UNICODE)]
            );

            return [
                'status' => 'success',
                'code' => 200,
                'message' => __('account_settings_update_success'),
            ];
        } catch (\Throwable $e) {
            return [
                'status' => 'error',
                'code' => 500,
                'message' => __('default_update_error'),
                'error' => $e->getMessage(),
            ];
        }
    }

    public function accountSettingsDefaultStatus(Request $request): array
    {
        try {
            $id = currentUser()->id ?? $request->user_id;
            $setting = AdditionalSetting::where('user_id', $id)
                ->where('key', 'account_settings')
                ->first();

            $existingData = $setting ? json_decode($setting->value, true) : [];

            if ($request->filled('default_type')) {
                $existingData['default_type'] = $request->input('default_type');
            }

            AdditionalSetting::updateOrCreate(
                ['user_id' => $id, 'key' => 'account_settings'],
                ['value' => json_encode($existingData, JSON_UNESCAPED_UNICODE)]
            );

            return [
                'status' => 'success',
                'code' => 200,
                'message' => __('account_settings_update_success'),
            ];
        } catch (\Throwable $e) {
            return [
                'status' => 'error',
                'code' => 500,
                'message' => __('default_update_error'),
                'error' => $e->getMessage(),
            ];
        }
    }
}
