<?php

namespace App\Repositories\Eloquent;

use App\Models\BillingAddress;
use App\Models\Country;
use App\Repositories\Contracts\BillingAddressRepositoryInterface;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BillingAddressRepository implements BillingAddressRepositoryInterface
{
    public function index(): array
    {
        $authId = Auth::id();
        $countries = Country::get();
        $billingAddress = BillingAddress::with(['country', 'state', 'city'])
            ->where('user_id', $authId)->get();
        return [
            'countries' => $countries,
            'billingAddress' => $billingAddress
        ];
    }

    public function list(Request $request): array
    {
        try {
            $userId = $request->user_id ?? currentUser()->id;
            $data = BillingAddress::with(['country', 'state', 'city'])
                ->where('user_id', $userId)->get()->map(function ($item) {
                    $item->name = ucfirst($item->name);
                    return $item;
                });

            return [
                'data' => $data,
                'billing_address_count' => $data->count(),
                'code' => 200,
                'message' => __('default_retrieve_success'),
            ];
        } catch (\Throwable $e) {
            return [
                'code'    => 500,
                'message' => __('default_retrieve_error'),
            ];
        }
    }

    public function store(Request $request): array
    {
        $id = $request->id ?? '';

        $successMessage = $id ? __('billing_address_update_success') : __('billing_address_create_success');
        $errorMessage = $id ? __('default_update_error') : __('default_create_error');

        try {

            $data = [
                'user_id' => Auth::id() ?? $request->user_id,
                'name' => $request->name,
                'email' => $request->email,
                'phone_number' => $request->phone_number,
                'address_line_one' => $request->address_line_one,
                'address_line_two' => $request->address_line_two,
                'country_id' => $request->country_id,
                'state_id' => $request->state_id,
                'city_id' => $request->city_id,
                'zipcode' => $request->zipcode,
            ];

            if (empty($id)) {
                $data = BillingAddress::create($data);
            } else {
                BillingAddress::where('id', $id)->update($data);
            }

            $id = empty($id) ? $data->id : $id;
            if (isset($request->is_default) && $request->is_default == 1) {
                $this->updateDefault($id);
            }

            return [
                'code' => 200,
                'message' => $successMessage,
            ];

        } catch (\Throwable $e) {
            return [
                'code'    => 500,
                'message' => $errorMessage,
                'error' => $e->getMessage()
            ];
        }
    }

    public function edit(Request $request): array
    {
        $data = BillingAddress::with(['country', 'state', 'city'])
            ->where('id', $request->id)->first();

        return [
            'data' => $data,
            'code' => 200,
            'message' => __('default_retrieve_success'),
        ];
    }

    public function delete(Request $request): array
    {
        try {
            BillingAddress::where('id', $request->id)->delete();
            return [
                'code' => 200,
                'message' => __('billing_address_delete_success'),
            ];
        } catch (\Throwable $e) {
            return [
                'code'    => 500,
                'message' => __('default_delete_error'),
            ];
        }
    }

    private function updateDefault($id)
    {
        BillingAddress::where('user_id', Auth::id())->update(['is_default' => 0]);
        BillingAddress::where('id', $id)->update(['is_default' => 1]);
    }

    public function setDefault(Request $request): array
    {
        try {
            $this->updateDefault($request->id);
            return [
                'code' => 200,
                'message' => __('billing_address_default_status_success'),
            ];
        } catch (\Throwable $e) {
            return [
                'code'    => 500,
                'message' => __('default_update_error'),
            ];
        }
    }
}
