<?php

namespace App\Http\Controllers;

use App\Models\Wishlist;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use DB;

class WishlistController extends Controller
{
    public function toggle(Request $request): JsonResponse
    {
        if (!Auth::check()) {
            return response()->json([
                'code' => 401,
                'message' => __('login_to_continue'),
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'product_id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'code' => 422,
                'errors' => $validator->errors(),
            ], 422);
        }

        $userId = Auth::id();
        $productId = (int) $request->input('product_id');

        $existing = Wishlist::where('user_id', $userId)->where('service_id', $productId)->where('type',$request->input('type'))->first();

        if ($existing) {
            $existing->delete();
            return response()->json([
                'code' => 200,
                'message' => __('wishlist_removed'),
                'in_wishlist' => false,
            ], 200);
        }

        Wishlist::create([
            'user_id' => $userId,
            'service_id' => $productId,
            'type' => $request->input('type'),
        ]);

        return response()->json([
            'code' => 200,
            'message' => __('wishlist_added'),
            'in_wishlist' => true,
        ], 200);
    }

    public function index()
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('wishlist'),
            'header' => __('wishlist'),
        ];

        // initial page load, empty content
        return view('frontend.pages.customer.wishlist',compact('data'));
    }

    public function load(Request $request)
    {

        $type = $request->type ?? 'shops';
        $page = $request->page ?? 1;
        $perPage = 1;
        $user = auth()->user();

        $total_count = Wishlist::where('wishlists.type', $type)->where('user_id',$user->id)->count();
        // echo $total_count;exit;
        $query = Wishlist::where('wishlists.type', $type)->where('user_id',$user->id);

        $sub = DB::table('ratings')
                ->select('product_id', DB::raw('AVG(rating) as avg_rating'),DB::raw('COUNT(*) as reviews_count'))
                ->where('type', $type)
                ->groupBy('product_id');

        // Load relationship dynamically
        if ($type == 'shops') {
            $query->with('shop', 'shop.countryRelation', 'shop.cityRelation', 'shop.services_list', 'shop.products_list')
            ->leftJoinSub($sub, 'rating_avg', function ($join) {
                $join->on('wishlists.service_id', '=', 'rating_avg.product_id');
            })
            ->select(
                'wishlists.*',
                'rating_avg.avg_rating',
                'rating_avg.reviews_count',
            );
        } elseif ($type == 'products') {
            $query = Wishlist::where('wishlists.type', $type)
                ->where('wishlists.user_id', $user->id)
                ->with('product', 'product.meta')
                ->leftJoinSub($sub, 'rating_avg', function ($join) {
                    $join->on('wishlists.service_id', '=', 'rating_avg.product_id');
                })
                ->leftJoin('user_details', 'wishlists.user_id', '=', 'user_details.user_id')
                ->select(
                    'wishlists.*',
                    'rating_avg.avg_rating',
                    'rating_avg.reviews_count',
                    'user_details.user_id as user_id',
                    'user_details.profile_image as user_profile_image'
                );
        } elseif ($type == 'services') {
            $query->with([
                'service',
                'product.imageMeta',
                'service.categories',
                'service.shop',
                'service.shop.countryRelation',
                'service.shop.cityRelation',
            ])
            ->leftJoinSub($sub, 'rating_avg', function ($join) {
                $join->on('wishlists.service_id', '=', 'rating_avg.product_id');
            })
            ->select(
                'wishlists.*',
                'rating_avg.avg_rating',
                'rating_avg.reviews_count',
            );
        }

        $data['type'] = $type;

        $wishlists = $query->orderBy('wishlists.id', 'desc')
            ->skip(($page - 1) * $perPage)
            ->take($perPage)
            ->get();


        $hasMore = ($page * $perPage) < $total_count;

        // Current page service_ids
        $currentIds = $wishlists->pluck('id')->unique()->values()->toArray();

        // Previous ids coming from request
        $previousIds = $request->input('previous_ids')
            ? explode(',', $request->input('previous_ids'))
            : [];

        // Merge and keep unique
        $serviceIds = array_values(array_unique(array_merge($previousIds, $currentIds)));

        if ($wishlists->isEmpty()) {
            return response()->json(['html' => '', 'next_page' => null]);
        }

        $html = view("frontend.pages.customer.wishlist-items", compact('wishlists','data'))->render();

        return response()->json([
            'html' => $html,
            'next_page' => $page + 1,
            'serviceIds'=> $serviceIds ?implode(',',$serviceIds): '',
            'hasMore'=> $hasMore,
        ]);

    }

    public function clear(Request $request)
    {
        $user = auth()->user();

        if (! $user) {
            return response()->json([
                'status' => false,
                'message' => __('unauthorized_access'),
            ], 401);
        }

        // Delete all wishlist items for this user
        Wishlist::where('user_id', $user->id)
            ->where('type', $request->type)
            ->whereIn('id', explode(',', $request->all_ids))
            ->delete();


        return response()->json([
            'status' => true,
            'message' => __('removed_all'),
        ]);
    }
}


