<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\BillingAddress;
use App\Models\Country;
use App\Models\TaxRate;
use App\Models\WalletHistory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Product\app\Models\Productmeta;
use Modules\Service\app\Models\Service;

class CartController extends Controller
{
    /**
     * Add a product to the cart session.
     */
    public function addToCart(Request $request)
    {
        $request->validate([
            'product_id' => 'required|integer',
            'quantity'   => 'nullable',
            'color'      => 'nullable|string',
            'size'       => 'nullable|string',
            'price'      => 'nullable', // CRITICAL: Get the price from the request
        ]);

        $cart = session()->get('cart', []);
        $cartKey = $request->product_id . '-' . $request->color . '-' . $request->size;
        $quantity = $request->quantity ?? 1;

        if (isset($cart[$cartKey])) {
            $cart[$cartKey]['quantity'] += $quantity;
        } else {
            $cart[$cartKey] = [
                "product_id" => $request->product_id,
                "quantity"   => $quantity,
                "color"      => $request->color,
                "size"       => $request->size,
                "price"      => $request->price,
            ];
        }
        session()->put('cart', $cart);

        return response()->json([
            'success'   => true,
            'message'   => 'Product added to cart!',
            'cartCount' => count($cart)
        ]);
    }

    /**
     * Display the cart page with all items from the session.
     */
    public function viewCart()
    {
        $cartItems = session('cart', []);
        $currecy_details = getDefaultCurrencySymbol();
        $countries = Country::get();
        $authId = Auth::id();
        $totalAmount = WalletHistory::where('user_id', $authId)->where('status', 'completed')->where('type', '1')->sum('amount');
        $totalAmountdebit = WalletHistory::where('user_id', $authId)->where('status', 'completed')->where('type', '2')->sum('amount');
        $walletTotalAmount = number_format($totalAmount - $totalAmountdebit, 2, '.', '');

        if (empty($cartItems)) {
            // Pass an empty array to the main view to handle the "empty" message there.
            return view('user.booking.product_booking_two', [
                'cartData' => [],
                'totalPrice' => 0,
                'currecy_details' => $currecy_details,
                'billingAddresses' => collect([]),
                'countries',
                'walletTotalAmount'
            ]);
        }

        $productIds = array_column($cartItems, 'product_id');
        $products = Service::whereIn('id', $productIds)
            ->where('status', 1)
            ->get()
            ->keyBy('id');

        $cartData = [];
        $totalPrice = 0;

        foreach ($cartItems as $key => $item) {
            $product = $products->get($item['product_id']);
            if ($product) {
                // Use the price stored in the session, not the base product price
                $itemPrice = $product->source_price ?? 0;
                $itemColor = null;
                $itemSize = null;
                $itemQuantity = $item['quantity'] ?? 1;

                $variation = getProductVariation($item['product_id']);
                $itemPrice = $variation['price'] ?? $itemPrice;
                $itemColor = $item['color'] ?? $variation['color'];
                $itemSize = $item['size'] ?? $variation['size'];

                $subtotal = $itemPrice * $itemQuantity;
                $totalPrice += $subtotal;

                $cartData[] = [
                    'cart_key'   => $key,
                    'product'    => $product,
                    'quantity'   => $itemQuantity,
                    'color'      => $itemColor,
                    'size'       => $itemSize,
                    'price'      => number_format($itemPrice, 2, '.', ''), // Pass the correct price to the view
                    'subtotal'   => number_format($subtotal, 2, '.', ''),
                    'image'      => $this->getProductImage($product->id),
                    'source_name' => $product->source_name ?? null,
                ];
            }
        }

        $totalPrice = number_format($totalPrice, 2, '.', '');

        // Global payment info
        $paymentInfo = GlobalSetting::where('group_id', 13)
            ->whereIn('key', [
                'stripe_status',
                'paypal_status',
                'cod_status',
                'bank_status',
                'wallet_status',
            ])->pluck('value', 'key')->toArray();

        $bankInfo = [
            'bank_name'      => optional(GlobalSetting::where('key', 'bank_name')->first())->value,
            'account_number' => optional(GlobalSetting::where('key', 'account_number')->first())->value,
            'branch_code'    => optional(GlobalSetting::where('key', 'branch_code')->first())->value,
        ];
        $billingAddresses = BillingAddress::with(['country', 'state', 'city'])
            ->where('user_id', $authId)
            ->get()->map(function ($address) {
                $address->country_name = optional($address->country)->name;
                $address->state_name = optional($address->state)->name;
                $address->city_name = optional($address->city)->name;

                return $address;
            });

        return view('user.booking.product_booking_two', compact(
            'cartData',
            'totalPrice',
            'currecy_details',
            'paymentInfo',
            'bankInfo',
            'billingAddresses',
            'countries',
            'walletTotalAmount'
        ));
    }

    /**
     * Remove an item from the cart via AJAX.
     */
    public function removeFromCart(Request $request)
    {
        $request->validate(['cart_key' => 'required|string']);

        $cart = session()->get('cart', []);

        if (isset($cart[$request->cart_key])) {
            unset($cart[$request->cart_key]);
            session()->put('cart', $cart);
        }

        return response()->json([
            'success' => true,
            'message' => 'Item removed from cart.',
            'cartCount' => count($cart)
        ]);
    }

    /**
     * Update cart quantities via AJAX.
     */
    public function updateCart(Request $request)
    {
        $request->validate(['quantities' => 'required|array']);

        $cart = session()->get('cart', []);

        foreach ($request->quantities as $cartKey => $quantity) {
            if (isset($cart[$cartKey])) {
                $cart[$cartKey]['quantity'] = max(1, intval($quantity)); // Ensure quantity is at least 1
            }
        }

        session()->put('cart', $cart);

        // No need to return totals, JS will recalculate. Can be added for sync.
        return response()->json([
            'success' => true,
            'message' => 'Cart updated successfully!',
            'cartCount' => count($cart)
        ]);
    }

    /**
     * Empty the entire cart via AJAX.
     */
    public function emptyCart()
    {
        session()->forget('cart');

        return response()->json([
            'success' => true,
            'message' => 'Cart has been emptied.',
            'cartCount' => 0
        ]);
    }

    private function getProductImage($productId)
    {
        $productMeta = Productmeta::where('product_id', $productId)->where('source_key', 'product_image')->first();
        return uploadedAsset($productMeta->source_Values ?? '');
    }
}
