<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Modules\Categories\app\Models\Categories;
use Modules\GlobalSetting\app\Models\Placeholders;
use Modules\Product\app\Models\Product;

class ProductController extends Controller
{
    private const CACHE_MINUTES = 10;
    private const PLACEHOLDER_CACHE_MINUTES = 30;

    public function index()
    {
        return view('admin.dashboard'); // Return the view for the dashboard
    }

    public function productlist(Request $request)
    {
        $cacheKey = sprintf('products.list:%s', md5($request->fullUrl()));

        $products = Cache::remember(
            $cacheKey,
            $this->cacheTtl(),
            static fn () => Product::query()
                ->where('source_type', '=', 'product')
                ->with([
                    'imageMeta:id,product_id,source_key,source_value',
                    'categories:id,name',
                ])
                ->orderByDesc('created_at')
                ->get()
        );

        return view('products', compact('products'));
    }

    public function productdetail(Request $request)
    {
        $validated = $request->validate([
            'slug' => ['required', 'string'],
        ]);

        $cacheKey = sprintf('products.detail:%s', $validated['slug']);

        $product = Cache::remember(
            $cacheKey,
            $this->cacheTtl(),
            static fn () => Product::query()
                ->where('slug', '=', $validated['slug'])
                ->with([
                    'meta:id,product_id,source_key,source_value',
                    'categories:id,name',
                    'shop:id,name',
                ])
                ->first()
        );

        abort_unless($product, 404);

        return view('productdetail', [
            'product' => $product,
            'products' => $product,
        ]);
    }

    public function add(Request $request)
    {
        $getplaceholder = Cache::remember(
            'placeholders.active',
            $this->cacheTtl(self::PLACEHOLDER_CACHE_MINUTES),
            static fn () => Placeholders::query()
                ->select('placeholder_name', 'id')
                ->where('status', 1)
                ->whereNull('deleted_at')
                ->orderBy('placeholder_name')
                ->get()
        );

        return view('admin.invoice-template', compact('getplaceholder'));
    }

    public function showFormCategories(Request $request)
    {
        $categoryId = $request->session()->get('category_id');

        abort_unless($categoryId, 404);

        $categoryName = Cache::remember(
            sprintf('categories.name:%s', $categoryId),
            $this->cacheTtl(),
            static fn () => Categories::query()->where('id', $categoryId)->value('name')
        );

        abort_unless($categoryName, 404);

        return view('admin.form-categories', compact('categoryId', 'categoryName'));
    }

    private function cacheTtl(int $minutes = self::CACHE_MINUTES)
    {
        return now()->addMinutes($minutes);
    }
}
