<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\JsonResponse;
use Modules\Shops\app\Models\Shops;
use Modules\Categories\app\Models\Categories;
use Modules\GlobalSetting\app\Models\Currency;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Modules\GlobalSetting\app\Models\Language;
use Modules\Product\app\Models\Product;

class FrontendShopController extends Controller
{

    private function getLanguageId(): int
    {
        return getLanguageId(app()->getLocale());
    }

    /**
     * Display the main shop listing page for the initial load.
     */
    public function index(Request $request): View
    {
        $language_id = $this->getLanguageId();

        // Use the new 'ratings' relationship to calculate the average and 'actualProducts' to count
        $query = Shops::query()
            ->with(['countryRelation', 'cityRelation', 'stateRelation'])
            ->where('status', 1)
            ->where('deleted_at', NULL)
            ->where('language_id', $language_id)
            ->withCount(['services', 'actualProducts'])
            ->withAvg('ratings', 'rating')
            ->latest();

        $shops = $query->paginate(6);

        $locations = DB::table('shops')
            ->join('countries', 'shops.country', '=', 'countries.id')
            ->select('countries.id', 'countries.name', DB::raw('count(shops.id) as total'))
            ->where('shops.status', 1)
            ->where('shops.language_id', $language_id)
            ->whereNotNull('shops.country')
            ->groupBy('countries.id', 'countries.name')
            ->orderBy('countries.name', 'asc')
            ->get();

        $vendors = DB::table('users')
            ->join('shops', 'users.id', '=', 'shops.created_by')
            ->select('users.id', 'users.name', DB::raw('count(shops.id) as total'))
            ->where('users.user_type', 2)
            ->whereNull('users.deleted_at')
            ->where('shops.status', 1)
            ->where('shops.language_id', $language_id)
            ->groupBy('users.id', 'users.name')
            ->orderBy('users.name', 'asc')
            ->get();

        $currency = Cache::remember('currency_details', 86400, function () {
            return Currency::select('symbol')->where('is_default', 1)->first();
        });

        $hasMorePages = $shops->hasMorePages();
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('shops'),
            'header' => __('shops'),
        ];

        return view('frontend.shops.index', compact('data', 'shops', 'locations', 'vendors', 'currency', 'hasMorePages'));
    }

    /**
     * Handle the AJAX request for filtering and loading more shops.
     */
    public function getShops(Request $request): JsonResponse
    {
        try {
            // 1. Start the base query and calculate average rating
            $query = Shops::query()
                ->where('status', 1)
                ->with(['cityRelation', 'countryRelation', 'stateRelation'])
                ->withCount(['actualProducts as actual_products_count', 'services'])
                ->withAvg('ratings', 'rating'); // Creates 'ratings_avg_rating'

            // 2. Apply search filter
            if ($request->filled('search')) {
                $searchTerm = $request->input('search');
                $query->where(function ($q) use ($searchTerm) {
                    $q->where('shop_name', 'like', "%{$searchTerm}%")
                        ->orWhere('owner_name', 'like', "%{$searchTerm}%");
                });
            }

            if ($request->filled('locations') && is_array($request->locations)) {
                $query->whereIn('country', $request->locations);
            }

            // 4. Apply vendor filter (by user ID)
            if ($request->filled('vendors') && is_array($request->vendors)) {
                $query->whereIn('created_by', $request->vendors);
            }

            // 5. Apply reviews filter based on calculated average rating
            if ($request->filled('reviews') && is_array($request->reviews)) {
                $conditions = [];
                $bindings = [];
                foreach ($request->reviews as $rating) {
                    $rating = (int)$rating;
                    if ($rating >= 1 && $rating <= 5) {
                        // Condition for each star rating (e.g., 4.0 to 4.99 for 4-star)
                        $conditions[] = '(ratings_avg_rating >= ? AND ratings_avg_rating < ?)';
                        $bindings[] = $rating;
                        $bindings[] = $rating + 1;
                    }
                }
                if (!empty($conditions)) {
                    $query->havingRaw(implode(' OR ', $conditions), $bindings);
                }
            }

            // 6. Apply sorting
            $sortBy = $request->input('sort_by', 'recent');
            switch ($sortBy) {
                case 'oldest':
                    $query->oldest();
                    break;
                case 'newest':
                    $query->latest();
                    break;
                case 'recent':
                default:
                    $query->orderBy('updated_at', 'desc');
                    break;
            }

            $shops = $query->paginate(6);
            $html = view('frontend.shops.partials.shop_items', compact('shops'))->render();

            return response()->json([
                'success'      => true,
                'html'         => $html,
                'hasMorePages' => $shops->hasMorePages(),
                'total'        => $shops->total(),
                'firstItem'    => $shops->firstItem(),
                'lastItem'     => $shops->lastItem(),
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'An error occurred while fetching shops.'], 500);
        }
    }

    /**
     * Display shop details page.
     */
    public function show(Request $request, string $slug): View
    {
        $shop = Shops::where('slug', $slug)
            ->where('status', 1)
            ->with([
                'countryRelation',
                'cityRelation',
                'stateRelation',
                'owner',
            ])
            ->withCount('reviews')
            ->withAvg('ratings', 'rating')
            ->firstOrFail();

        $actualProducts = Product::query()
            ->where('products.shop_id', $shop->id)
            ->where('products.source_type', 'product')
            ->select('products.*')
            ->selectSub(function ($query) {
                $query->from('products_meta')
                    ->whereColumn('products_meta.product_id', 'products.id')
                    ->where('products_meta.source_key', '=', 'product_image')
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'asc')
                    ->select('source_values')
                    ->take(1);
            }, 'product_image_url')
            ->selectSub(function ($query) {
                $query->from('ratings')
                    ->whereColumn('ratings.product_id', 'products.id')
                    ->whereNull('deleted_at')
                    ->selectRaw('AVG(rating)');
            }, 'avg_rating')
            ->selectSub(function ($query) {
                $query->from('ratings')
                    ->whereColumn('ratings.product_id', 'products.id')
                    ->whereNull('deleted_at')
                    ->selectRaw('COUNT(id)');
            }, 'ratings_count')
            ->get()->map(function ($product) {
                $price = $product->source_price;
                $variationProduct = getProductVariation($product->id);
                $variationPrice = $variationProduct['price'] ?? $price ?? 0;
                $product->source_price = $variationPrice ?? $price ?? 0;

                return $product;
            });

        $services = Product::query()
            ->leftJoin('categories', 'products.source_category', '=', 'categories.id')
            ->where('products.shop_id', $shop->id)
            ->where('products.source_type', 'service')
            ->select('products.*', 'categories.name as category_name')
            ->selectSub(function ($query) {
                $query->from('products_meta')
                    ->whereColumn('products_meta.product_id', 'products.id')
                    ->where('products_meta.source_key', '=', 'product_image')
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'asc')
                    ->select('source_values')
                    ->take(1);
            }, 'product_image_url')
            ->selectSub(function ($query) {
                $query->from('ratings')
                    ->whereColumn('ratings.product_id', 'products.id')
                    ->whereNull('deleted_at')
                    ->selectRaw('AVG(rating)');
            }, 'avg_rating')
            ->selectSub(function ($query) {
                $query->from('ratings')
                    ->whereColumn('ratings.product_id', 'products.id')
                    ->whereNull('deleted_at')
                    ->selectRaw('COUNT(id)');
            }, 'ratings_count')
            ->get();

        $addressParts = [];
        if (!empty($shop->address)) {
            $addressParts[] = $shop->address;
        }
        if ($shop->cityRelation?->name) {
            $addressParts[] = $shop->cityRelation->name;
        }
        if ($shop->stateRelation?->name) {
            $addressParts[] = $shop->stateRelation->name;
        }
        if ($shop->countryRelation?->name) {
            $addressParts[] = $shop->countryRelation->name;
        }

        $full_address = implode(', ', $addressParts);

        $relatedShops = Shops::where('status', 1)->where('id', '!=', $shop->id)
            ->where(function ($query) use ($shop) {
                $query->where('country', $shop->country)
                    ->orWhere('state', $shop->state)
                    ->orWhere('city', $shop->city);
            })->limit(6)->get();

        $social_link = DB::table('social_links')->where('status', 1)->where('deleted_at', NULL)->get();
        $currency_code = getDefaultCurrencySymbol();
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => '<a href="' . route('frontend.shops.index') . '">' . __('shops') . '</a>',
            'header' => $shop->shop_name,
        ];
        return view('frontend.shops.show', compact('shop', 'relatedShops', 'data', 'social_link', 'full_address', 'currency_code',  'actualProducts', 'services',));
    }

    // Other methods like getShopList, getShopById, etc. can remain below.
    public function getShopList(Request $request): JsonResponse
    {
        $request->validate([
            'parent_id' => 'required|integer|exists:users,id',
        ]);

        $parentId = $request->parent_id;

        $shops = Shops::query()
            ->where('created_by', $parentId)
            ->where('status', 1)
            ->get(['id', 'shop_name']);

        return response()->json([
            'success' => true,
            'data' => $shops,
        ]);
    }

    public function getShopById(Request $request): JsonResponse
    {
        $request->validate([
            'shop_id' => 'required|integer|exists:shops,id',
        ]);

        $shop = Shops::where('id', $request->shop_id)
            ->where('status', 1)
            ->first(['id', 'shop_name']);

        return response()->json([
            'success' => true,
            'data' => $shop,
        ]);
    }
}
