<?php

namespace App\Http\Controllers;

use App\Models\Enquiry;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;
use Exception;

class EnquiryController extends Controller
{
    /**
     * Store a new enquiry
     */
    public function store(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'provider_email' => 'required|email',
                'name'           => 'required|string|max:255',
                'email'          => 'required|email',
                'phone_number'   => 'required|string|min:8|max:20',
                'message'        => 'required|string|min:5',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'code'   => 422,
                    'errors' => $validator->errors(),
                ], 422);
            }

            $provider = User::where('email', $request->provider_email)->first();

            $enquiry = Enquiry::create([
                'provider_id'     => $provider?->id,
                'customer_name'   => $request->name,
                'email'           => $request->email,
                'phone'           => $request->phone_number,
                'enquiry_date'    => now(),
                'enquiry_details' => $request->message,
                'status'          => 'pending',
            ]);

            return response()->json([
                'code'    => 200,
                'message' => 'Enquiry submitted successfully.',
                'data'    => $enquiry,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'code'    => 500,
                'message' => 'Something went wrong while storing enquiry.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Show enquiry index page
     */
    public function index(): View
    {
        try {
            return view('provider.enquiry.index');
        } catch (Exception $e) {
            abort(500, 'Error loading enquiries page: ' . $e->getMessage());
        }
    }

    /**
     * List all enquiries
     */
    public function providerEnquiries(Request $request)
    {
        try {
            $providerId = Auth::id();

            $enquiries = Enquiry::where('provider_id', $providerId)
                ->latest()
                ->get();

            return response()->json([
                'code'    => 200,
                'message' => 'Enquiries fetched successfully.',
                'data'    => $enquiries,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'code'    => 500,
                'message' => 'Something went wrong while fetching enquiries.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

     public function delete(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id' => 'required|integer|exists:enquiries,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'code'   => 422,
                    'errors' => $validator->errors(),
                ], 422);
            }

            $enquiry = Enquiry::find($request->id);

            if ($enquiry->provider_id !== Auth::id()) {
                return response()->json([
                    'code'    => 403,
                    'message' => 'Unauthorized action.',
                ], 403);
            }

            $enquiry->delete();

            return response()->json([
                'code'    => 200,
                'message' => 'Enquiry deleted successfully.',
            ]);
        } catch (Exception $e) {
            return response()->json([
                'code'    => 500,
                'message' => 'Something went wrong while deleting enquiry.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update enquiry status (inprogress or complete) with optional comments
     */
    public function updateStatus(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'id'       => 'required|integer|exists:enquiries,id',
                'status'   => 'required|in:inprogress,complete',
                'comments' => 'nullable|string|max:1000',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'code'   => 422,
                    'errors' => $validator->errors(),
                ], 422);
            }

            $enquiry = Enquiry::find($request->id);

            // Ensure the logged-in provider owns the enquiry
            if ($enquiry->provider_id !== Auth::id()) {
                return response()->json([
                    'code'    => 403,
                    'message' => 'Unauthorized action.',
                ], 403);
            }

            // Prevent reverting from complete -> inprogress
            if ($enquiry->status === 'complete' && $request->status === 'inprogress') {
                return response()->json([
                    'code'    => 400,
                    'message' => 'Completed enquiries cannot be reverted to inprogress.',
                ], 400);
            }

            $enquiry->status = $request->status;
            $enquiry->comments = $request->comments ?? null;
            $enquiry->save();

            return response()->json([
                'code'    => 200,
                'message' => 'Enquiry status updated successfully.',
                'data'    => $enquiry,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'code'    => 500,
                'message' => 'Something went wrong while updating enquiry status.',
                'error'   => $e->getMessage(),
            ], 500);
        }
    }
}
