<?php

namespace App\Http\Controllers;

use App\Http\Requests\CustomerRequest;
use App\Models\AdditionalSetting;
use App\Models\BookingHistory;
use Illuminate\Support\Facades\Auth;
use App\Repositories\Contracts\CustomerRepositoryInterface;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;
use App\Models\Bookings;
use App\Models\BookingProduct;
use App\Models\PayoutHistory;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Modules\Communication\app\Http\Controllers\NotificationController;
use Modules\Service\app\Models\Service;
use App\Models\UserDetail;
use Modules\Product\app\Models\Product;
use App\Models\User;
use App\Models\WalletHistory;
use PDF;
use Illuminate\Support\Facades\Cache;
use Modules\Product\app\Models\Rating;
use Modules\Shops\app\Models\Shops;
use Modules\Communication\app\Models\Templates;
use Illuminate\Support\Str;
use Modules\Communication\app\Http\Controllers\EmailController;
use Illuminate\Support\Facades\Log;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Product\app\Models\Productmeta;


class CustomerController extends Controller
{

    protected CustomerRepositoryInterface $customerRepository;
    protected NotificationController $notificationController;

    public function __construct(
        CustomerRepositoryInterface $customerRepository,
        NotificationController $notificationController
    ) {
        $this->customerRepository = $customerRepository;
        $this->notificationController = $notificationController;
    }

    public function index(): View
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('Dashboard'),
            'header' => __('Dashboard'),
        ];

        return view('frontend.pages.customer.dashboard', compact('data'));
    }

    public function getStats(Request $request)
    {
        try {
            $user = Auth::user();
            $user_id = $user->id;
            $from = $request->input('from_date');
            $to   = $request->input('to_date');

            $query = BookingProduct::query()
                ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id')
                ->where('bookings.user_id', $user_id)
                ->whereNull('bookings.deleted_at')
                ->select(
                    'booking_products.*',
                    'bookings.type as booking_type',
                    'bookings.total_amount',
                    'bookings.payment_status',
                    'bookings.booking_status',
                    'bookings.booking_date',
                );

            // Custom date range
            if ($request->filled('selected_date')) {
                // Split the date range
                $dates = explode(' - ', $request->selected_date);
                if (count($dates) == 2) {
                    $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                    $end = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                    $query->whereBetween('bookings.booking_date', [$start, $end]);
                }
            }
            $query->where('bookings.booking_status', '!=', 7);

            // Services booked
            $servicesBooked = DB::table('bookings')
            ->where('type', 'service')
            ->where('user_id', $user_id)
            ->where('booking_status', '!=', 7)
            ->when($request->filled('selected_date'), function ($query) use ($request) {
                $dates = explode(' - ', $request->selected_date);
                if (count($dates) === 2) {
                    $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                    $end = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                    $query->whereBetween('created_at', [$start, $end]);
                }
            })
            ->count();

            // Total transactions - Combined from both bookings and booking_products
            $totalTransactions = DB::table('bookings')
                ->where('user_id', $user_id)
                ->where('payment_status', 2)
                ->where('booking_status', '!=', 7)
                ->when($request->filled('selected_date'), function ($query) use ($request) {
                    $dates = explode(' - ', $request->selected_date);
                    if (count($dates) === 2) {
                        $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                        $end = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                        $query->whereBetween('created_at', [$start, $end]);
                    }
                })
                ->sum('total_amount');

            // Products purchased
            $productsPurchased = (clone $query)
                ->where('bookings.type', 'product')
                ->sum('booking_products.quantity');

            $firstImage = DB::table('products_meta')
                ->select('product_id', DB::raw('MIN(source_values) as product_image'))
                ->where('source_key', 'product_image')
                ->groupBy('product_id');

            $recentBookings = DB::table('bookings')
                ->leftJoin('products', 'bookings.product_id', '=', 'products.id')
                ->leftJoin('categories', 'products.source_subcategory', '=', 'categories.id')
                ->leftJoinSub($firstImage, 'first_image', function ($join) {
                    $join->on('products.id', '=', 'first_image.product_id');
                })
                ->where('bookings.type', 'service')
                ->where('bookings.user_id', $user_id)
                ->where('bookings.booking_status', '!=', 7)
                ->when($request->filled('selected_date'), function ($q) use ($request) {
                    $dates = explode(' - ', $request->selected_date);
                    if (count($dates) === 2) {
                        $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                        $end   = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                        $q->whereBetween('bookings.created_at', [$start, $end]);
                    }
                })
                ->orderBy('bookings.booking_date', 'desc')
                ->limit(5)
                ->get([
                    'bookings.*',
                    'products.source_name',
                    'categories.name as category_name',
                    'first_image.product_image'
                ]);

            $recentPurchases = (clone $query)
                ->with(['booking', 'product', 'product.meta', 'product.categories'])
                ->where('type', 'product')
                ->latest('bookings.created_at')
                ->take(5)
                ->get();

            $statusMap = [
                1 => __('unpaid'),
                2 => __('paid'),
                3 => __('refund'),
            ];

            $serviceBookingTransactions = Bookings::with([
                    'invoice_product',
                ])
                ->select('bookings.*')
                ->where('type', 'service')
                ->whereIn('booking_status', [1, 2, 3, 5, 6])
                ->where('bookings.user_id', $user->id)
                ->when($request->filled('selected_date'), function ($q) use ($request) {
                    $dates = explode(' - ', $request->selected_date);
                    if (count($dates) === 2) {
                        $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                        $end   = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                        $q->whereBetween('bookings.created_at', [$start, $end]);
                    }
                })
                ->orderBy('bookings.booking_date', 'desc')
                ->take(5)->get()->map(function ($bp) use ($statusMap) {
                    $product = $bp->invoice_product;

                    return (object) [
                        'order_id'     => $bp->order_id,
                        'invoice_date' => formatDateTime($bp->created_at),
                        'due_date'     => formatDateTime($bp->booking_date),
                        'user_id'      => $bp->user_id,
                        'amount'       => getDefaultCurrencySymbol() . number_format($bp->total_amount, 2, '.', ''),
                        'payment_status' => $bp->payment_status,
                        'payment_status_label' => $statusMap[$bp->payment_status] ?? '-',
                        'payment_type' => $bp->payment_type,
                        'description' => $product->source_name ?? '-',
                    ];
                });

            // FIXED: Product Order Transactions Section
            // First, get the last 5 booking IDs with the ordering column included
            $lastFiveBookingIds = DB::table('booking_products')
                ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id')
                ->where('booking_products.user_id', $user->id)
                ->whereIn('product_status', [1, 2, 3, 4])
                ->select('booking_products.booking_id', 'bookings.created_at')
                ->when($request->filled('selected_date'), function ($q) use ($request) {
                    $dates = explode(' - ', $request->selected_date);
                    if (count($dates) === 2) {
                        $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                        $end   = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                        $q->whereBetween('bookings.created_at', [$start, $end]);
                    }
                })
                ->distinct()
                ->orderByDesc('bookings.created_at')
                ->limit(5)
                ->pluck('booking_id');

            // Then, fetch the full booking products for those IDs
            $productOrderTransactions = BookingProduct::with([
                    'booking',
                    'product'
                ])
                ->whereIn('booking_products.booking_id', $lastFiveBookingIds)
                ->whereIn('product_status', [1, 2, 3, 4])
                ->where('booking_products.user_id', $user->id)
                ->get();

            // Group by booking_id
            $groupedByBooking = $productOrderTransactions->groupBy('booking_id');

            $productOrderTransactionsData = [];
            foreach ($groupedByBooking as $bookingId => $bpCollection) {
                $booking = $bpCollection->first()->booking;
                $end_date = \Carbon\Carbon::parse($booking->booking_date)->addDays(7);

                $firstProduct = $bpCollection->first();
                $description = $firstProduct ? $firstProduct->product_name : '';

                $bookingData = [
                    'order_id'     => $booking->order_id,
                    'invoice_date' => formatDateTime($booking->booking_date),
                    'due_date'     => formatDateTime($end_date),
                    'user_id'      => $booking->user_id,
                    'amount'       => getDefaultCurrencySymbol() . number_format($bpCollection->sum('gross_total'), 2, '.', ''),
                    'payment_status' => $booking->payment_status,
                    'payment_status_label' => $statusMap[$booking->payment_status] ?? '-',
                    'payment_type' => $booking->payment_type,
                    'description'  => $description,
                    'provider'    => [],
                ];

                // Group products by provider_id
                $groupedByProvider = $bpCollection->groupBy('provider_id');

                foreach ($groupedByProvider as $providerId => $products) {
                    $providerProducts = [];
                    foreach ($products as $bp) {
                        $providerProducts[] = [
                            'product_name' => $bp->product_name,
                            'price'        => $bp->price,
                            'quantity'     => $bp->quantity,
                            'subtotal'     => $bp->subtotal,
                            'gross_total'  => $bp->gross_total,
                        ];
                    }

                    $bookingData['provider'][] = [
                        'provider_id' => $providerId,
                        'products'    => $providerProducts,
                    ];
                }

                $productOrderTransactionsData[] = $bookingData;
            }

            $productOrderTransactionsData = collect($productOrderTransactionsData)
                ->sortByDesc('order_id')
                ->map(function($item) {
                    return (object)$item;   // convert each booking array to object
                });


            $html = view("frontend.pages.customer.dashboard-report", compact('servicesBooked', 'totalTransactions', 'productsPurchased', 'recentBookings', 'recentPurchases', 'serviceBookingTransactions', 'productOrderTransactionsData'))->render();

            return response()->json([
                'html' => $html,
                'status' => 'success',
            ]);
        } catch (\Exception $e) {
            Log::error('Error fetching orders: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => __('order_list_error'),
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function product_orders(Request $request): View
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('products'),
            'header' => __('products'),
        ];
        return view('frontend.pages.customer.products', compact('data'));
    }

    public function getOrders(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();

            $statusMap = [
                1 =>  __('pending'),
                2 =>  __('confirmed'),
                3 =>  __('shipped'),
                4 =>  __('delivery'),
                5 =>  __('vendor_cancel'),
                6 =>  __('customer_cancel'),
                7 => __('refund_initiated'),
                8 => __('refund_completed'),
            ];

            $statusClass = [
                1 => 'badge-soft-warning',
                2 => 'badge-soft-success',
                3 => 'badge-soft-info',
                4 => 'badge-soft-success',
                5 => 'badge-soft-info',
                6 => 'badge-soft-info',
                7 => 'badge-soft-info',
                8 => 'badge-soft-success',
            ];

            $perPage = $request->get('length', 10);   // DataTables sends "length"
            $start   = $request->get('start', 0);     // DataTables sends "start"
            $page    = ($start / $perPage) + 1;       // Convert to Laravel page number

            // Query booking_products as main table
            $query = BookingProduct::with([
                'booking:id,order_id,booking_date,first_name,last_name,user_id,total_amount,booking_status',
                'product.categories',
            ])->select('booking_products.*')
                ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id');

            if ($request->filled('search_value')) {
                $query->whereHas('product', function ($q) use ($request) {
                    $q->where('source_name', 'like', '%' . $request->search_value . '%');
                });
            }

            // Custom date range
            if ($request->filled('date_range')) {
                // Split the date range
                $dates = explode(' - ', $request->date_range);
                if (count($dates) == 2) {
                    $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                    $end = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                    $query->whereBetween('booking_date', [$start, $end]);
                }
            }

            $query->where('bookings.type', 'product');

            // Role-based filter
            if ($user->user_type === 3) {
                $query->whereHas('booking', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                });
            } elseif ($user->user_type === 2) {
                $query->whereHas('product', function ($q) use ($user) {
                    $q->where('user_id', $user->id); // provider = product owner
                });
            } elseif ($request->type === 'admin') {
                // Admin → no filter
            }

            $query->orderBy('bookings.booking_date', 'DESC');

            $bookingProducts = $query->paginate($perPage, ['*'], 'page', $page);
            $data = [];
            foreach ($bookingProducts as $bp) {
                $order   = $bp->booking;
                $product = $bp->product;

                $image = Productmeta::where('product_id', $product->id)
                ->where('source_key', 'product_image')
                ->orderBy('id', 'asc')   // pick the very first row
                ->value('source_values');

                $profileImagePath = DB::table('user_details')
                    ->where('user_id', $order->user_id)
                    ->value('profile_image');

                if ($profileImagePath && Storage::disk('public')->exists($profileImagePath)) {
                    $profileImage = asset('storage/profile/' . $profileImagePath);
                } else {
                    $profileImage = asset('/assets/img/profile-default.png');
                }

                $data[] = [
                    'id'           => $bp->id,
                    'order_id'     => $order->order_id,
                    'booking_date' => formatDateTime($order->booking_date),
                    'customer'     => ucfirst($order->first_name . ' ' . $order->last_name),
                    'user_id'      => $order->user_id,
                    'profile_image'=> $profileImage,
                    'amount'       => '$' . $order->total_amount,
                    'status'       => $statusMap[$bp->product_status] ?? '-',
                    'status_class' => $statusClass[$bp->product_status] ?? 'badge-soft-info',

                    // product-specific
                    'product_id'   => $product->id,
                    'product_name' => $product->source_name,
                    'product_slug' => $product->slug,
                    'image'        => $image ? asset('storage/' . $image) : asset('img/default-image.png'),
                    'categories'   => $product->categories?->name ?? '-',
                    'quantity'     => $bp->quantity,
                    'price'        => '$' . $bp->gross_total,
                ];
            }

            return response()->json([
                'success'        => true,
                'message'        => __('order_list_success'),
                'draw'           => intval($request->get('draw')),
                'recordsTotal'   => $bookingProducts->total(),
                'recordsFiltered' => $bookingProducts->total(),
                'data'           => $data,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error fetching orders: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => __('order_list_error'),
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function getOrderDetails(Request $request)
    {
        $orderId = $request->get('id');

        // Fetch order using repository
        $order = $this->customerRepository->getOrderDetails($orderId);

        if (!$order) {
            return response()->json(['html' => __("no_order_found")]);
        }

        $currencySymbol = getDefaultCurrencySymbol();

        // Render partial blade view for modal body
        $html = view('frontend.pages.customer.order_details', compact('order', 'currencySymbol'))->render();
        return response()->json(['html' => $html, 'order_id' => $order->booking->order_id ?? '-']);
    }

    public function getBookingDetails(Request $request)
    {
        $bookingId = $request->get('id');

        // Fetch booking with relationships
        $bookings = $this->customerRepository->getBookingDetails($bookingId);

        if (!$bookings) {
            return response()->json(['html' => '<p>__("no_order_found")</p>']);
        }

        // --- 1. Get Product Info (similar to customerBookingUpcomingList) ---
        $firstImage = DB::table('products_meta')
            ->select('product_id', DB::raw('MIN(source_Values) as product_image'))
            ->where('source_key', 'product_image')
            ->groupBy('product_id');

        $product = DB::table('products')
            ->leftJoinSub($firstImage, 'first_image', function ($join) {
                $join->on('products.id', '=', 'first_image.product_id');
            })
            ->leftJoin('categories', 'products.source_subcategory', '=', 'categories.id')
            ->where('products.id', $bookings->product_id)
            ->select(
                'products.id',
                'products.source_name',
                'categories.name as subcategory_name',
                'first_image.product_image'
            )
            ->first();

        if ($product) {
            // Format product image
            if ($product->product_image && Storage::disk('public')->exists($product->product_image)) {
                $product->image_url = Storage::url($product->product_image);
            } else {
                $product->image_url = uploadedAsset('', 'default');
            }
        }

        // --- 2. Get Provider Info ---
        $provider = DB::table('users')
            ->leftJoin('user_details', 'users.id', '=', 'user_details.user_id')
            ->select(
                'users.id',
                'users.name',
                'users.email',
                'users.phone_number',
                'users.created_at',
                'user_details.profile_image'
            )
            ->where('users.id', $bookings->provider_id)
            ->first();

        if ($provider) {
            // Format profile image
            if ($provider->profile_image && Storage::disk('public')->exists($provider->profile_image)) {
                $provider->profile_image_url = Storage::url($provider->profile_image);
            } else {
                $provider->profile_image_url = uploadedAsset('', 'profile');
            }
        }


        // --- 3. Map Payment Info ---
        $paymentTypes = [
            1 => 'Paypal',
            2 => 'Stripe',
            3 => 'Razorpay',
            4 => 'BankTransfer',
            5 => 'COD',
            6 => 'Wallet',
            7 => 'Cashfree',
            8 => 'PayU',
            9 => 'AuthorizeNet',
            10 => 'Paystack',
        ];

        $paymentStatuses = [
            1 => 'Unpaid',
            2 => 'Paid',
            3 => 'Refund',
        ];

        $bookingDetails = [
            'booking' => $bookings,
            'product' => $product,
            'provider' => $provider,
            'payment_type' => $paymentTypes[$bookings->payment_type] ?? 'Unknown',
            'payment_status' => $paymentStatuses[$bookings->payment_status] ?? 'Unknown',
        ];

        $html = view('frontend.pages.customer.booking_details', compact('bookingDetails'))->render();

        return response()->json([
            'html' => $html,
            'order_id' => $bookings->order_id ?? '-'
        ]);
    }

    public function showInvoice($orderId)
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('invoice'),
            'header' => __('invoice'),
        ];

        // Fetch order with relations using order_id
        $order = Bookings::with([
            'bookingProducts' => function ($query) {
                $query->whereIn('product_status', [1, 2, 3, 4]);
            },
            'bookingProducts.vendorInfo.userDetail.user_country',
            'bookingProducts.vendorInfo.userDetail.user_state',
            'bookingProducts.vendorInfo.userDetail.user_city',
            'bookingProducts.product.shop',
        ])->where('order_id', $orderId)->firstOrFail();

        $order->total_gross_amount = 0.00;
        $order->total_tax_amount = 0.00;
        $order->coupon_discount_amount = 0.00;

        if ($order) {
            // Calculate total gross amount from booking_products
            $totalSubtotalAmount = $order->bookingProducts->sum('subtotal');

            $totalGrossAmount = $order->bookingProducts->sum('gross_total');

            // Calculate total tax amount (assuming booking_products has a column total_tax_amount)
            $taxAmount = $order->bookingProducts->sum('tax_amount');

            // Calculate total coupon discount amount (assuming booking_products has a column coupon_discount_amount)
            $couponDiscountAmount = $order->bookingProducts->sum('coupon_discount_amount');

            $order->total_gross_amount = number_format($totalGrossAmount, 2, '.', '');
            $order->total_tax_amount = number_format($taxAmount, 2, '.', '');
            $order->total_coupon_discount_amount = number_format($couponDiscountAmount, 2, '.', '');
            $order->total_subtotal_amount = number_format($totalSubtotalAmount, 2, '.', '');
        }

        $serviceBooking = Bookings::with([
                'product',
                'vendorInfo.userDetail',
                'vendorInfo.userDetail.user_country',
                'vendorInfo.userDetail.user_state',
                'vendorInfo.userDetail.user_city',
            ])
            ->where('order_id', $orderId)->firstOrFail();
        $type = $serviceBooking->type ?? 'service';

        $currencySymbol = getDefaultCurrencySymbol();

        $item = [];
        if ($serviceBooking->type == 'service') {
            $item = [
                'product_name' => $serviceBooking->product->source_name ?? '-',
                'quantity'    => $serviceBooking->service_qty ?? 1,
                'price'       => number_format($serviceBooking->service_amount ?? 0, 2, '.', ''),
                'total_amount'    => number_format($serviceBooking->total_amount ?? 0, 2, '.', ''),
                'tax_amount'    => number_format($serviceBooking->amount_tax ?? 0, 2, '.', ''),
                'total_coupon_discount_amount'    => number_format($serviceBooking->total_coupon_discount_amount ?? 0, 2, '.', ''),
            ];
        }

        return view('frontend.pages.customer.order_invoice', compact('order', 'data', 'item', 'type', 'currencySymbol'));
    }

    public function showProviderInvoice(Request $request, $orderId)
    {
        $productId = $request->query('product_id'); // product_id from URL if passed

        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('invoice'),
            'header' => __('invoice'),
        ];

        // Fetch the order and all relations
        $order = Bookings::with([
            'bookingProducts' => function ($query) {
                $query->select('*'); // fetch all fields from booking_products
            },
            'bookingProducts.vendorInfo.userDetail.user_country',
            'bookingProducts.vendorInfo.userDetail.user_state',
            'bookingProducts.vendorInfo.userDetail.user_city',
            'bookingProducts.product.shop',
            'product', // in case it's a service booking
        ])->where('order_id', $orderId)->firstOrFail();

        if ($productId) {
            $filteredProducts = $order->bookingProducts->filter(function ($product) use ($productId) {
                return $product->id == $productId && isset($product->product_status) && $product->product_status < 5;
            });

            // Replace original bookingProducts with filtered ones
            $order->setRelation('bookingProducts', $filteredProducts);
        }

        $currencySymbol = getDefaultCurrencySymbol();

        $items = [];

        // Loop through bookingProducts only if type is 'product'
        if ($order->type === 'product') {
            foreach ($order->bookingProducts as $product) {
                $items[] = [
                    'product_name'     => $product->product_name ?? '-',
                    'quantity'         => $product->quantity ?? 1,
                    'price'            => $product->price ?? 0,
                    'subtotal'         => $product->subtotal ?? 0,
                    'gross_total'      => $product->gross_total ?? 0, // only for products
                    'tax_amount'       => $product->tax_amount ?? 0,
                    'coupon_discount'  => $product->coupon_discount_amount ?? 0,
                ];
            }
        }

        // For services, you can still optionally add items without gross_total
        if ($order->type === 'service' && $order->product) {
            $items[] = [
                'product_name'     => $order->product->source_name ?? $order->product->product_name ?? '-',
                'quantity'         => $order->service_qty ?? 1,
                'price'            => $order->service_amount ?? 0,
                'subtotal'         => $order->total_amount ?? 0,
                'tax_amount'       => $order->amount_tax ?? 0,
                'coupon_discount'  => $order->total_coupon_discount_amount ?? 0,
            ];
        }

        return view('frontend.pages.vendor.order_invoice', compact(
            'order',
            'data',
            'items',
            'currencySymbol'
        ));
    }

    public function downloadInvoice($orderId)
    {
        // Fetch order with relations using order_id
        $order = Bookings::with([
            'bookingProducts' => function ($query) {
                $query->whereIn('product_status', [1, 2, 3, 4]);
            },
            'bookingProducts.vendorInfo.userDetail.user_country',
            'bookingProducts.vendorInfo.userDetail.user_state',
            'bookingProducts.vendorInfo.userDetail.user_city',
            'bookingProducts.product.shop',
        ])->where('order_id', $orderId)->firstOrFail();

        $order->total_gross_amount = 0.00;
        $order->total_tax_amount = 0.00;
        $order->coupon_discount_amount = 0.00;

        if ($order) {
            // Calculate total gross amount from booking_products
            $totalSubtotalAmount = $order->bookingProducts->sum('subtotal');

            $totalGrossAmount = $order->bookingProducts->sum('gross_total');

            // Calculate total tax amount (assuming booking_products has a column total_tax_amount)
            $taxAmount = $order->bookingProducts->sum('tax_amount');

            // Calculate total coupon discount amount (assuming booking_products has a column coupon_discount_amount)
            $couponDiscountAmount = $order->bookingProducts->sum('coupon_discount_amount');

            $order->total_gross_amount = number_format($totalGrossAmount, 2, '.', '');
            $order->total_tax_amount = number_format($taxAmount, 2, '.', '');
            $order->total_coupon_discount_amount = number_format($couponDiscountAmount, 2, '.', '');
            $order->total_subtotal_amount = number_format($totalSubtotalAmount, 2, '.', '');
        }

        $serviceBooking = Bookings::with([
                'product',
                'vendorInfo.userDetail',
                'vendorInfo.userDetail.user_country',
                'vendorInfo.userDetail.user_state',
                'vendorInfo.userDetail.user_city',
            ])
            ->where('order_id', $orderId)->firstOrFail();
        $type = $serviceBooking->type ?? 'service';

        $currencySymbol = getDefaultCurrencySymbol();

        $item = [];
        if ($serviceBooking->type == 'service') {
            $item = [
                'product_name' => $serviceBooking->product->source_name ?? '-',
                'quantity'    => $serviceBooking->service_qty ?? 1,
                'price'       => number_format($serviceBooking->service_amount ?? 0, 2, '.', ''),
                'total_amount'    => number_format($serviceBooking->total_amount ?? 0, 2, '.', ''),
                'tax_amount'    => number_format($serviceBooking->amount_tax ?? 0, 2, '.', ''),
                'total_coupon_discount_amount'    => number_format($serviceBooking->total_coupon_discount_amount ?? 0, 2, '.', ''),
            ];
        }

        $logoPath = Cache::remember('logoPath', 86400, function () {
            return GlobalSetting::where('key', 'site_logo')->value('value');
        });
        $logoPath = $logoPath && file_exists(public_path('storage/' . $logoPath))
            ? public_path('storage/' . $logoPath)
            : public_path('front/img/logo.svg');


        $currencySymbol = getDefaultCurrencySymbol();

        $pdf = PDF::loadView('frontend.pages.customer.order_invoice_download', compact('order', 'item', 'type', 'currencySymbol', 'logoPath'))
            ->setPaper('a4', 'portrait');

        return $pdf->download('invoice-' . $order->order_id . '.pdf');
    }

    public function downloadVendorInvoice($orderId)
    {
        // Fetch order with relations using order_id
        $order = Bookings::with([
            'bookingProducts',
            'bookingProducts.vendorInfo.userDetail.user_country',
            'bookingProducts.vendorInfo.userDetail.user_state',
            'bookingProducts.vendorInfo.userDetail.user_city',
            'bookingProducts.product.shop',
        ])->where('order_id', $orderId)->firstOrFail();

        $get_booking_data = Bookings::with('product')->where('order_id', $orderId)->get()->first();

        $items = [];
        if ($get_booking_data->type == 'service') {
            $items[] = [
                'product_name' => $get_booking_data->product->source_name ?? $get_booking_data->product->product_name ?? '-',
                'quantity'    => $get_booking_data->service_qty ?? 1,
                'price'       => $get_booking_data->service_amount ?? 0,
                'subtotal'    => $get_booking_data->total_amount ?? 0,
            ];
        }

        $logoPath = Cache::remember('logoPath', 86400, function () {
            return GlobalSetting::where('key', 'site_logo')->value('value');
        });
        $logoPath = $logoPath && file_exists(public_path('storage/' . $logoPath))
            ? public_path('storage/' . $logoPath)
            : public_path('front/img/logo.svg');


        $currencySymbol = getDefaultCurrencySymbol();

        $pdf = PDF::loadView('frontend.pages.vendor.order_invoice_download', compact('order', 'items', 'currencySymbol', 'logoPath'))
            ->setPaper('letter', 'portrait');

        return $pdf->download('invoice-' . $order->order_id . '.pdf');
    }

    public function customerBookingUpcoming(Request $request)
    {
        return view('frontend.pages.customer.customer_booking_upcoming');
    }

    public function customerBookingCompleted(Request $request)
    {
        return view('frontend.pages.customer.customer_booking_completed');
    }

    public function customerBookingCancelled(Request $request)
    {
        return view('frontend.pages.customer.customer_booking_cencelled');
    }

    public function customerBookingUpcomingList(Request $request)
    {
        $user = Auth::user();

        // Subquery to get only the first image for each product
        $firstImage = DB::table('products_meta')
            ->select('product_id', DB::raw('MIN(source_Values) as product_image'))
            ->where('source_key', 'product_image')
            ->groupBy('product_id');

        $bookings = DB::table('bookings')
            ->join('products', 'bookings.product_id', '=', 'products.id')
            ->leftJoinSub($firstImage, 'first_image', function ($join) {
                $join->on('products.id', '=', 'first_image.product_id');
            })
            ->leftJoin('categories as subcat', 'products.source_subcategory', '=', 'subcat.id')
            ->where('bookings.user_id', $user->id)
            ->whereIn('bookings.booking_status', [1, 2]) // 1=Open, 2=Accepted
            ->select(
                'bookings.id as booking_id',
                'bookings.order_id',
                'bookings.booking_date',
                'products.source_name',
                'subcat.name as subcategory_name',
                'first_image.product_image'
            )
            ->orderBy('bookings.booking_date', 'desc')
            ->get();

        // Format the date and construct the full image URL
        $bookings->transform(function ($item) {
            $item->booking_date_formatted = Carbon::parse($item->booking_date)->format('d M Y');

            if ($item->product_image && Storage::disk('public')->exists($item->product_image)) {
                $item->image_url = Storage::url($item->product_image);
            } else {
                $item->image_url = asset('frontend/assets/img/service/service-placeholder.jpg');
            }

            return $item;
        });

        return response()->json(['data' => $bookings]);
    }

    public function customerBookingCompletedList(Request $request)
    {
        $user = Auth::user();

        // Subquery to get only the first image for each product
        $firstImage = DB::table('products_meta')
            ->select('product_id', DB::raw('MIN(source_Values) as product_image'))
            ->where('source_key', 'product_image')
            ->groupBy('product_id');

        $bookings = DB::table('bookings')
            ->join('products', 'bookings.product_id', '=', 'products.id')
            ->leftJoinSub($firstImage, 'first_image', function ($join) {
                $join->on('products.id', '=', 'first_image.product_id');
            })
            ->leftJoin('categories', 'products.source_subcategory', '=', 'categories.id')
            ->where('bookings.user_id', $user->id)
            ->whereIn('bookings.booking_status', [5, 6]) // 1=Open, 2=Accepted
            ->select(
                'bookings.id as booking_id',
                'bookings.order_id',
                'bookings.product_id',
                'bookings.booking_date',
                'products.source_name',
                'categories.name as subcategory_name',
                'products.slug',
                'first_image.product_image',
                'bookings.user_id'
            )
            ->orderBy('bookings.created_at', 'desc')
            ->get();

        // Format the date and construct the full image URL
        $bookings->transform(function ($item) {
            $item->booking_date_formatted = Carbon::parse($item->booking_date)->format('d M Y');

            if ($item->product_image && Storage::disk('public')->exists($item->product_image)) {
                $item->image_url = Storage::url($item->product_image); // gives /storage/...
            } else {
                $item->image_url = asset('frontend/assets/img/service/service-placeholder.jpg');
            }

            $item->review_exists = reviewExists($item->product_id, $item->user_id);

            return $item;
        });

        return response()->json(['data' => $bookings]);
    }

    public function customerBookingCancelledList(Request $request)
    {
        $user = Auth::user();

        // Subquery to get only the first image for each product
        $firstImage = DB::table('products_meta')
            ->select('product_id', DB::raw('MIN(source_Values) as product_image'))
            ->where('source_key', 'product_image')
            ->groupBy('product_id');

        $bookings = DB::table('bookings')
            ->join('products', 'bookings.product_id', '=', 'products.id')
            ->leftJoinSub($firstImage, 'first_image', function ($join) {
                $join->on('products.id', '=', 'first_image.product_id');
            })
            ->leftJoin('categories as subcat', 'products.source_subcategory', '=', 'subcat.id')
            ->where('bookings.user_id', $user->id)
            ->whereIn('bookings.booking_status', [3, 8, 4, 7])
            ->select(
                'bookings.id as booking_id',
                'bookings.order_id',
                'bookings.booking_date',
                'bookings.booking_status',
                'products.source_name',
                'subcat.name as source_subcategory',
                'products.slug',
                'first_image.product_image'
            )
            ->orderBy('bookings.created_at', 'desc')
            ->get();

        // Format the date and construct the full image URL
        $bookings->transform(function ($item) use ($user) {
            $item->booking_date_formatted = Carbon::parse($item->booking_date)->format('d M Y');

            if ($item->product_image && Storage::disk('public')->exists($item->product_image)) {
                $item->image_url = Storage::url($item->product_image);
            } else {
                $item->image_url = asset('frontend/assets/img/service/service-placeholder.jpg');
            }

            $refundPaymentProof = PayoutHistory::where('reference_id', $item->booking_id)
                ->where('type', 2)
                ->where('user_id', $user->id)
                ->first();

            if ($refundPaymentProof) {
                $item->refund_payment_proof = uploadedAsset($refundPaymentProof->payment_proof);
            } else {
                $item->refund_payment_proof = null;
            }

            return $item;
        });

        return response()->json(['data' => $bookings]);
    }

    public function customerBookingUpcomingCancel(Request $request)
    {
        $request->validate([
            'booking_id' => 'required|integer',
            'reason' => 'required|string|max:255',
        ]);

        $user = Auth::user();
        $booking = Bookings::where('id', $request->booking_id)
            ->where('user_id', $user->id)
            ->first();

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found or you do not have permission to cancel it.'
            ], 404);
        }

        // Check if the booking can be cancelled (status 1 or 2)
        if (!in_array($booking->booking_status, [1, 2])) {
            return response()->json([
                'success' => false,
                'message' => 'This booking cannot be cancelled.'
            ], 400);
        }

        // --- Update booking status (8 = Customer Cancelled) ---
        $booking->booking_status = 8;
        $booking->save();

        // --- Notification setup ---
        $status = 'Customer Cancelled';
        $source = 'Booking Cancel';
        $sourceAdmin = 'Customer Cancelled email to Admin';
        $sourceProvider = 'Customer Cancelled email to Provider';
        $sourceStaff = 'Customer Cancelled email to Staff';

        // --- Fetch booking details with joins ---
        $bookingdata = Bookings::select(
            'bookings.*',
            DB::raw("CASE
                WHEN bookings.booking_status = 1 THEN 'Open'
                WHEN bookings.booking_status = 2 THEN 'In progress'
                WHEN bookings.booking_status = 3 THEN 'Provider Cancelled'
                WHEN bookings.booking_status = 4 THEN 'Refund Initiated'
                WHEN bookings.booking_status = 5 THEN 'Completed'
                WHEN bookings.booking_status = 6 THEN 'Order Completed'
                WHEN bookings.booking_status = 7 THEN 'Refund Completed'
                WHEN bookings.booking_status = 8 THEN 'Customer Cancelled'
                ELSE 'Unknown'
            END AS booking_status_label"),
            DB::raw("DATE_FORMAT(bookings.created_at, '%d-%m-%Y') AS bookingdate"),
            DB::raw("TIME_FORMAT(bookings.from_time, '%H:%i') AS fromtime"),
            DB::raw("TIME_FORMAT(bookings.to_time, '%H:%i') AS totime"),
            'products.source_name',
            'users.name as user_name',
            'users.email as user_email',
            'provider.id as provider_id',
            'provider.name as provider_name',
            'provider.email as provideremail',
            'staff.name as staff_name',
            'staff.email as staffemail'
        )
            ->leftJoin('products', 'products.id', '=', 'bookings.product_id')
            ->leftJoin('users', 'users.id', '=', 'bookings.user_id')
            ->leftJoin('users as provider', 'provider.id', '=', 'products.created_by')
            ->leftJoin('users as staff', 'staff.id', '=', 'bookings.staff_id')
            ->where('bookings.id', $booking->id)
            ->first();

        $settingData = getCommonSettingData(['company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website']);

        // --- Send emails ---
        $this->sendEmailToRecipient($source, $bookingdata, $settingData, $bookingdata->user_email ?? null);

        if ($sourceAdmin) {
            $admins = User::with('userDetails')->where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');
                $this->sendEmailToRecipient($sourceAdmin, $bookingdata, $settingData, $admin->email, $adminName);
            }
        }

        if ($sourceProvider && $bookingdata->provideremail) {
            $this->sendEmailToRecipient($sourceProvider, $bookingdata, $settingData, $bookingdata->provideremail);
        }

        if ($sourceStaff && $bookingdata->staffemail) {
            $this->sendEmailToRecipient($sourceStaff, $bookingdata, $settingData, $bookingdata->staffemail);
        }

        try {

            // Notification for Provider
            $data = [
                'communication_type' => '3',
                'source' => $source,
                'reference_id' => $bookingdata->id,
                'user_id' => $bookingdata->user_id,        // customer
                'to_user_id' => $bookingdata->provider_id, // provider
                'to_description' => "Booking #{$bookingdata->order_id} has been cancelled by the customer.",
                'from_description' => "You cancelled booking #{$bookingdata->order_id}.",
            ];
            $this->notificationController->Storenotification(new Request($data));

            // Notification for Admins
            $admins = User::where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $dataAdmin = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $bookingdata->user_id,   // customer
                    'to_user_id' => $admin->id,           // admin
                    'to_description' => "Booking #{$bookingdata->order_id} was cancelled by the customer.",
                    'from_description' => "",
                ];
                $this->notificationController->Storenotification(new Request($dataAdmin));
            }
        } catch (\Exception $e) {
            Log::error('Notification send failed (cancel): ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Booking cancelled successfully and notifications sent.'
        ]);
    }

    public function customerBookingStatusRefund(Request $request)
    {
        $user = Auth::user();
        $booking = Bookings::where('id', $request->booking_id)
            ->where('user_id', $user->id)
            ->first();

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found or you do not have permission to refund it.'
            ], 404);
        }

        $booking->booking_status = 4;
        $booking->save();

        $getbookings = Bookings::where('id', $request->booking_id)->first();
        if (isset($getbookings)) {
            $data['user_id'] = $getbookings->user_id;
            $data['reference_id'] = $request->booking_id;
            $data['type'] = 2;
            $data['total_bookings'] = 1;
            $serviceamount = 0;
            if ($getbookings->total_amount != '') {
                $serviceamount = $getbookings->total_amount;
            }
            $data['total_earnings'] = $data['pay_due'] = $serviceamount;
            $data['process_amount'] = $serviceamount;
            $data['remaining_amount'] = $serviceamount;
            $data['created_by'] = $getbookings->user_id;
            $data['created_at'] = Carbon::now();
            PayoutHistory::insertGetId($data);
        }

        $status = 'Refund Initiated';
        $source = 'Refund Initiation';
        $sourceAdmin = 'Refund Initiation email to Admin';

        $bookingdata = Bookings::select(
            'bookings.*',
            DB::raw("CASE
                WHEN bookings.booking_status = 1 THEN 'Open'
                WHEN bookings.booking_status = 2 THEN 'In progress'
                WHEN bookings.booking_status = 3 THEN 'Provider Cancelled'
                WHEN bookings.booking_status = 4 THEN 'Refund Initiated'
                WHEN bookings.booking_status = 5 THEN 'Completed'
                WHEN bookings.booking_status = 6 THEN 'Order Completed'
                WHEN bookings.booking_status = 7 THEN 'Refund Completed'
                WHEN bookings.booking_status = 8 THEN 'Customer Cancelled'
                ELSE 'Unknown'
            END AS booking_status_label"),
            DB::raw("DATE_FORMAT(bookings.created_at, '%d-%m-%Y') AS bookingdate"),
            DB::raw("TIME_FORMAT(bookings.from_time, '%H:%i') AS fromtime"),
            DB::raw("TIME_FORMAT(bookings.to_time, '%H:%i') AS totime"),
            'products.source_name',
            'users.name as user_name',
            'users.email as user_email',
            'provider.id as provider_id',
            'provider.name as provider_name',
            'provider.email as provideremail',
            'staff.name as staff_name',
            'staff.email as staffemail',
            'payout_history.id as refundid',
            DB::raw("DATE_FORMAT(payout_history.created_at, '%d-%m-%Y') AS trxdate"),
            DB::raw("DATE_FORMAT(payout_history.updated_at, '%d-%m-%Y') AS refunddate")
        )
            ->leftJoin('products', 'products.id', '=', 'bookings.product_id')
            ->leftJoin('users', 'users.id', '=', 'bookings.user_id')
            ->leftJoin('users as provider', 'provider.id', '=', 'products.created_by')
            ->leftJoin('users as staff', 'staff.id', '=', 'bookings.staff_id')
            ->leftJoin('payout_history', 'payout_history.reference_id', '=', 'bookings.id')
            ->where('bookings.id', $booking->id)
            ->first();

        $settingData = getCommonSettingData([
            'company_name',
            'site_email',
            'phone_no',
            'site_address',
            'postal_code',
            'website'
        ]);

        $this->sendEmailToRecipient($source, $bookingdata, $settingData, $bookingdata->user_email ?? null);

        if ($sourceAdmin) {
            $admins = User::with('userDetails')->where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');
                $this->sendEmailToRecipient($sourceAdmin, $bookingdata, $settingData, $admin->email, $adminName);
            }
        }

        try {

            // Notification for Provider
            $data = [
                'communication_type' => '3',
                'source' => $source,
                'reference_id' => $bookingdata->id,
                'user_id' => $bookingdata->user_id,        // customer
                'to_user_id' => $bookingdata->provider_id, // provider
                'to_description' => "Booking #{$bookingdata->order_id} refund has been initiated.",
                'from_description' => "You initiated refund for booking #{$bookingdata->order_id}.",
            ];
            $this->notificationController->Storenotification(new Request($data));

            // Notification for Admins
            $admins = User::where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $dataAdmin = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $bookingdata->user_id,   // customer
                    'to_user_id' => $admin->id,           // admin
                    'to_description' => "Refund initiated for booking #{$bookingdata->order_id}.",
                    'from_description' => "",
                ];
                $this->notificationController->Storenotification(new Request($dataAdmin));
            }
        } catch (\Exception $e) {
            Log::error('Notification send failed (refund): ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Booking refund initiated successfully and notifications sent.'
        ]);
    }

    public function delivery_tracker(Request $request): View
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('Delivery Tracking'),
            'header' => __('Delivery Tracking'),
        ];
        return view('frontend.pages.customer.delivery_tracker', compact('data'));
    }

    public function getDeliveryTrack(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();

            $statusMap = [
                1 =>  __('pending'),
                2 =>  __('confirmed'),
                3 =>  __('shipped'),
                4 =>  __('delivery'),
                5 =>  __('vendor_cancel'),
                6 =>  __('customer_cancel'),
                7 => __('refund_initiated'),
                8 => __('refund_completed'),
            ];

            $statusClass = [
                1 => 'badge-soft-warning',
                2 => 'badge-soft-success',
                3 => 'badge-soft-info',
                4 => 'badge-soft-success',
                5 => 'badge-soft-info',
                6 => 'badge-soft-info',
                7 => 'badge-soft-info',
                8 => 'badge-soft-success',
            ];

            $perPage = $request->get('length', 10);   // DataTables sends "length"
            $start   = $request->get('start', 0);     // DataTables sends "start"
            $page    = ($start / $perPage) + 1;       // Convert to Laravel page number

            // Query booking_products as main table
            $query = BookingProduct::with([
                'booking:id,order_id,booking_date,first_name,last_name,user_id,total_amount,booking_status',
                'product.categories',
            ])
                ->select('booking_products.*')
                ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id');

            if ($request->filled('search_value')) {
                $query->whereHas('product', function ($q) use ($request) {
                    $q->where('source_name', 'like', '%' . $request->search_value . '%');
                });
            }

            // Custom date range
            if ($request->filled('date_range')) {
                // Split the date range
                $dates = explode(' - ', $request->date_range);
                if (count($dates) == 2) {
                    $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                    $end = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                    $query->whereBetween('booking_date', [$start, $end]);
                }
            }

            $query->where('bookings.type', 'product');

            // Role-based filter
            if ($user->user_type === 3) {
                $query->whereHas('booking', function ($q) use ($user) {
                    $q->where('user_id', $user->id)->where('type', 'product');;
                });
            } elseif ($user->user_type === 2) {
                $query->whereHas('product', function ($q) use ($user) {
                    $q->where('user_id', $user->id); // provider = product owner
                });
            } elseif ($request->type === 'admin') {
                // Admin → no filter
            }

            $query->orderBy('bookings.booking_date', 'DESC');

            $bookingProducts = $query->paginate($perPage, ['*'], 'page', $page);

            $data = [];
            foreach ($bookingProducts as $bp) {
                $order   = $bp->booking;
                $product = $bp->product;

                $image = Productmeta::where('product_id', $product->id)
                ->where('source_key', 'product_image')
                ->whereNull('deleted_at')
                ->orderBy('id', 'asc')   // pick the very first row
                ->value('source_Values');

                $end_date = \Carbon\Carbon::parse($order->booking_date)->addDays(7);

                $data[] = [
                    'id'           => $bp->id,
                    'order_id'     => $order->order_id,
                    'booking_date' => formatDateTime($order->booking_date),
                    'expected_date' => formatDateTime($end_date),
                    'user_id'      => $order->user_id,
                    'product_status'       => $bp->product_status ?? 0,
                    'status'       => $statusMap[$bp->product_status] ?? '-',
                    'status_class' => $statusClass[$bp->product_status] ?? 'badge-soft-info',

                    // product-specific
                    'product_id'   => $product->id,
                    'product_name' => $product->source_name,
                    'product_slug' => $product->slug,
                    'image'        => uploadedAsset($image),
                    'categories'   => $product->categories?->name ?? '-',
                ];
            }

            return response()->json([
                'success'        => true,
                'message'        => __('order_list_success'),
                'draw'           => intval($request->get('draw')),
                'recordsTotal'   => $bookingProducts->total(),
                'recordsFiltered' => $bookingProducts->total(),
                'data'           => $data,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error fetching orders: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => __('order_list_error'),
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function viewDeliveryTrackDetail(Request $request, $id)
    {
        $user = Auth::user();

        $user_id = $user->id;

        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('Delivery Tracking'),
            'header' => __('Delivery Tracking'),
        ];

        // Query booking_products as main table
        $query = BookingProduct::with([
            'booking:id,order_id,booking_date,first_name,last_name,user_id,total_amount,booking_status',
            'product.categories'
        ])
            ->select('booking_products.*')
            ->join('bookings', 'booking_products.booking_id', '=', 'bookings.id');

        $query->where('bookings.type', 'product')->where('booking_products.id', $id);

        // Role-based filter
        if ($user->user_type === 3) {
            $query->whereHas('booking', function ($q) use ($user, $id) {
                $q->where('user_id', $user->id)->where('type', 'product')->where('booking_products.id', $id);
            });
        } elseif ($user->user_type === 2) {
            $query->whereHas('product', function ($q) use ($user) {
                $q->where('user_id', $user->id); // provider = product owner
            });
        } elseif ($request->type === 'admin') {
            // Admin → no filter
        }

        $bookingProducts = $query->first();

        $productImage = Productmeta::where('product_id', $bookingProducts->product_id ?? 0)
                ->where('source_key', 'product_image')
                ->whereNull('deleted_at')
                ->orderBy('id', 'asc')
                ->value('source_Values');

        $productImage = uploadedAsset($productImage);

        $payoutHistory = PayoutHistory::where('reference_id', $bookingProducts->booking_id)
                ->where('booking_product_id', $bookingProducts->id)
                ->where('type', 2)
                ->where('user_id', $user_id)
                ->first();
        $refundPaymentProof = null;

        if ($payoutHistory) {
            $refundPaymentProof = uploadedAsset($payoutHistory->payment_proof);
        }

        // Location & map
        $user_name = $city_name = 'California, USA';

        // Get provider (owner of the product in this booking)
        $provider = User::join('products as p', 'p.user_id', '=', 'users.id')
            ->join('user_details as ud', 'ud.user_id', '=', 'users.id')
            ->where('p.id', $bookingProducts->product_id)
            ->select(
                'users.id as provider_id',
                'users.name',
                'users.email',
                'users.phone_number',
                'ud.profile_image',
                'ud.address',
                'ud.country',
                'ud.state',
                'ud.city'
            )
            ->first();

        $user_address = $provider;

        if (!empty($user_address)) {
            $city = DB::table('cities')->where('id', $user_address->city)->first();
            if ($city) {
                $user_name = $city->name;
            }
        }
        if (!empty($bookingProducts)) {
            $vendor_address = UserDetail::where('user_id', $bookingProducts->provider_id)->select('address', 'country', 'state', 'city')->first();

            if ($vendor_address && !empty($vendor_address->city)) {
                $city = DB::table('cities')->where('id', $vendor_address->city)->first();
                if ($city) {
                    $city_name = $city->name;
                }
            }
        }
        $customer_googleMapUrl = 'https://www.google.com/maps/embed/v1/place?key='
            . config('services.google_maps.key')
            . '&q=' . urlencode($user_name);

        $vendor_googleMapUrl = 'https://www.google.com/maps/embed/v1/place?key='
            . config('services.google_maps.key')
            . '&q=' . urlencode($city_name);

        $locationSetting = DB::table('general_settings')->where('key', 'location_status')->first();
        $locationEnabled = $locationSetting && $locationSetting->value == 1;
        $mapHasError = !$locationEnabled || $this->isInvalidGoogleMapKey($customer_googleMapUrl) || $this->isInvalidGoogleMapKey($vendor_googleMapUrl);

        $data['customer_googleMapUrl'] = $customer_googleMapUrl;
        $data['vendor_googleMapUrl'] = $vendor_googleMapUrl;

        $tracker_details = BookingHistory::where('booking_product_id', $bookingProducts->id)->get()->toArray();

        $tracker_data = [];
        foreach ($tracker_details as $key => $value) {
            $tracker_data[$value['status']] = $value['datetime'];
        }
        $end_date = \Carbon\Carbon::parse($bookingProducts->created_at)->addDays(7);

        $shippingDelivery = Carbon::parse($bookingProducts->created_at);
        $expectedDelivery = Carbon::parse($end_date);

        // Calculate the difference
        $diff = $shippingDelivery->diff($expectedDelivery);

        // Format the difference
        $days = $diff->days;            // total days
        $hours = $diff->h;              // hours
        $minutes = $diff->i;            // minutes

        $data['shipping_delivery'] = formatDateTime($bookingProducts->created_at);
        $data['expected_delivery'] = formatDateTime($end_date);
        $data['delivery_difference'] = "{$days} days {$hours} hours";

        return view('frontend.pages.customer.delivery_tracker_details', compact('bookingProducts', 'data', 'tracker_data', 'user_address', 'refundPaymentProof', 'productImage'));
    }

    public function vedorBookingList(Request $request)
    {
        $providerId = Auth::id();

        $products = Service::select("source_name", "id")
            ->where('user_id', $providerId)
            ->where('source_type', "product")
            ->where('status', 1)
            ->get();

        return view('frontend.pages.vendor.vendor_booking_list', [
            'products' => $products,
        ]);
    }

    public function fetchVendorBookingsData(Request $request)
    {
        $request->validate(['status' => 'required|string']);

        $statusKey = $request->query('status');
        $bookingProducts = $this->getBookingProductsByStatus($request, $statusKey);
        $bookingProducts->map(function ($bp) {
            $userId = $bp->user_id ?? null;

            $profileImagePath = DB::table('user_details')
                ->where('user_id', $userId)
                ->value('profile_image');

            $bp->user_profile_image = $profileImagePath;
            $bp->payment_type_label = getPaymentTypeLabel($bp->booking->payment_type);

            return $bp;
        });

        return view('frontend.pages.vendor.orders_table_rows', [
            'bookingProducts' => $bookingProducts,
            'currentStatusKey' => $statusKey,
            'currency' => getDefaultCurrencySymbol()
        ]);
    }

    private function getBookingProductsByStatus(Request $request, $statusKey)
    {
        $statusMap = [
            'new'        => 1,
            'confirmed'  => 2,
            'shipped'    => 3,
            'delivered'  => 4,
            'cancelled'  => 5,
        ];

        $statusCode = $statusMap[$statusKey] ?? 1;
        $providerId = Auth::id();

        $query = BookingProduct::with([
                'booking',
                'user',
                'product.image'
            ])
            ->where('provider_id', $providerId)
            ->where('product_status', $statusCode)
            ->whereHas('booking', function ($query) {
                $query->where(function ($q) {
                    $q->where(function ($sub) {
                        $sub->where('payment_type', '!=', 5)
                            ->where('payment_status', 2);
                    })
                    ->orWhere(function ($sub) {
                        $sub->where('payment_type', 5)
                            ->whereIn('payment_status', [1, 2]);
                    });
                });
            });

        if ($request->filled('product_id')) {
            $query->where('product_id', $request->product_id);
        }

        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('product_name', 'LIKE', "%{$searchTerm}%")
                    ->orWhereHas('booking', function ($subQ) use ($searchTerm) {
                        $subQ->where('order_id', 'LIKE', "%{$searchTerm}%");
                    })
                    ->orWhereHas('user', function ($subQ) use ($searchTerm) {
                        $subQ->where('name', 'LIKE', "%{$searchTerm}%");
                    });
            });
        }

        return $query->latest()->paginate(10)->appends($request->query());
    }

    public function updateBookingProductStatus(Request $request, BookingProduct $bookingProduct)
    {
        $validated = $request->validate([
            'status' => ['required', Rule::in([2, 3, 4, 5])],
            // 2=Confirmed, 3=Shipped, 4=Delivered, 5=Vendor Cancelled
            'reason' => ['required_if:status,5', 'string', 'nullable', 'max:500'],
        ]);

        $authId = Auth::id();

        // Update product status
        $bookingProduct->product_status = $validated['status'];

        if ($validated['status'] == 5) {
            $bookingProduct->cancellation_reason = $validated['reason'];
        }

        $isCod = false;
        if ($request->has('payment_status') && !empty($request->payment_status)) {
            $bookingProduct->cod_payment_status = $request->payment_status;
            $isCod = true;
        }

        $bookingProduct->save();

        if ($bookingProduct && $isCod) {
            $bookingProduct->booking->update([
                'payment_status' => $request->payment_status
            ]);
        }

        // Status mapping
        $statusMap = [
            2 => 'Confirmed',
            3 => 'Shipped',
            4 => 'Delivered',
            5 => 'Vendor Cancelled',
        ];

        $statusMessages = [
            2 => 'Order has been confirmed by the seller.',
            3 => 'Carrier has received the shipment details.',
            4 => 'The delivery agent is en route to the customer.',
            5 => 'Order has been cancelled by the vendor.',
        ];

        // Insert booking history
        BookingHistory::create([
            'booking_product_id' => $bookingProduct->id,
            'status' => strtolower($statusMap[$validated['status']]),
            'message' => $statusMessages[$validated['status']],
            'datetime' => now(),
        ]);

        // --- Setup notification sources (like updateBookingStatus) ---
        $source = $sourceAdmin = $sourceProvider = $sourceStaff = '';
        $messages = [];

        switch ($validated['status']) {
            case 2:
                $source = 'Order Confirmed';
                $sourceAdmin = 'Order Confirmed email to Admin';
                $sourceProvider = 'Order Confirmed email to Provider';
                $messages = [
                    'admin' => "Order #{{order_id}} for {{service_name}} has been confirmed by {{provider_name}}.",
                    'customer' => "Your order #{{order_id}} for {{service_name}} has been confirmed by {{provider_name}}.",
                    'vendor' => "You have confirmed order #{{order_id}} for {{service_name}} placed by {{customer_name}}.",
                ];
                break;

            case 3:
                $source = 'Order Shipped';
                $sourceAdmin = 'Order Shipped email to Admin';
                $sourceProvider = 'Order Shipped email to Provider';
                $messages = [
                    'admin' => "Order #{{order_id}} for {{service_name}} has been shipped by {{provider_name}}.",
                    'customer' => "Your order #{{order_id}} for {{service_name}} has been shipped by {{provider_name}}.",
                    'vendor' => "You shipped order #{{order_id}} for {{service_name}} to {{customer_name}}.",
                ];
                break;

            case 4:
                $source = 'Order Delivered';
                $sourceAdmin = 'Order Delivered email to Admin';
                $sourceProvider = 'Order Delivered email to Provider';
                $messages = [
                    'admin' => "Order #{{order_id}} for {{service_name}} has been delivered to {{customer_name}}.",
                    'customer' => "Your order #{{order_id}} for {{service_name}} has been delivered successfully.",
                    'vendor' => "You delivered order #{{order_id}} for {{service_name}} to {{customer_name}}.",
                ];
                break;

            case 5:
                $source = 'Order Cancelled';
                $sourceAdmin = 'Vendor Cancelled email to Admin';
                $sourceProvider = 'Vendor Cancelled email to Provider';
                $messages = [
                    'admin' => "Order #{{order_id}} for {{service_name}} was cancelled by {{provider_name}}.",
                    'customer' => "Your order #{{order_id}} for {{service_name}} was cancelled by {{provider_name}}.",
                    'vendor' => "You cancelled order #{{order_id}} for {{service_name}} placed by {{customer_name}}.",
                ];
                break;
        }

        // --- Fetch booking + relations for email templates ---
        $bookingdata = BookingProduct::with('booking')
            ->select(
                'booking_products.*',
                'products.source_name',
                'users.name as user_name',
                'users.email as user_email',
                'provider.id as provider_id',
                'provider.email as provideremail',
                'user_details.first_name as user_first_name',
                'user_details.last_name as user_last_name',
                'provider_details.first_name as provider_first_name',
                'provider_details.last_name as provider_last_name',
            )
            ->leftJoin('products', 'products.id', '=', 'booking_products.product_id')
            ->leftJoin('users', 'users.id', '=', 'booking_products.user_id')
            ->leftJoin('user_details', 'user_details.user_id', '=', 'users.id')
            ->leftJoin('users as provider', 'provider.id', '=', 'products.created_by')
            ->leftJoin('user_details as provider_details', 'provider_details.user_id', '=', 'provider.id')
            ->where('booking_products.id', $bookingProduct->id)
            ->first();

        $settingData = getCommonSettingData([
            'company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website'
        ]);

        // --- Send emails ---
        // Customer
        $this->sendProductUpdateEmailToRecipient($source, $bookingdata, $settingData, $bookingdata->user_email ?? null);

        // Admin
        if ($sourceAdmin) {
            $admins = User::with('userDetails')->where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');
                $this->sendProductUpdateEmailToRecipient($sourceAdmin, $bookingdata, $settingData, $admin->email, $adminName);
            }
        }

        // Provider
        if ($sourceProvider && $bookingdata->provideremail) {
            $this->sendProductUpdateEmailToRecipient($sourceProvider, $bookingdata, $settingData, $bookingdata->provideremail);
        }

        // --- Send Notification ---
        $this->sendProductUpdateNotificationToRecipient($source, $bookingdata, $messages, $settingData, $authId);

        return response()->json([
            'success' => true,
            'message' => 'Order status updated successfully, logged in history, and notifications sent.'
        ]);
    }

    private function sendProductUpdateNotificationToRecipient($source = '', $bookingdata, $messages, $settingData, $authId)
    {
        if (!$bookingdata || !$bookingdata->booking) {
            return;
        }

        $userName = ucwords($bookingdata->user_first_name . ' ' . $bookingdata->user_last_name);
        $providerName = ucwords($bookingdata->provider_first_name . ' ' . $bookingdata->provider_last_name);
        $booking = $bookingdata->booking;
        $productName = $bookingdata->source_name;

        $tempdata = [
            '{{customer_name}}' => $userName,
            '{{user_name}}' => $userName,
            '{{booking_id}}' => $booking->order_id,
            '{{order_id}}' => $booking->order_id,
            '{{service_name}}' => $productName,
            '{{order_date}}' => formatDateTime($booking->booking_date),
            '{{provider_name}}' => $providerName,
            '{{contact}}' => $bookingdata->provideremail,
            '{{website_link}}' => $settingData['website'] ?? "",
            '{{transaction_date}}' => $bookingdata->trxdate,
            '{{company_name}}' => $settingData['company_name'] ?? "",
            '{{payment_method}}' => $bookingdata->paymenttype,
            '{{service_address}}' => $bookingdata['product']['createdBy']['userDetails']->address ?? $settingData['site_address'] ?? "",
        ];

        $descriptionForCustomer = Str::replace(array_keys($tempdata), array_values($tempdata), $messages['customer']);
        $descriptionForVendor = Str::replace(array_keys($tempdata), array_values($tempdata), $messages['vendor']);

        // --- Send Notifications (Provider -> Customer + Admins) ---
        try {

            // Provider → Customer
            $data = [
                'communication_type' => '3',
                'source' => $source,
                'reference_id' => $bookingdata->id,
                'user_id' => $authId,  // provider
                'to_user_id' => $bookingdata->user_id,    // customer
                'to_description' => $descriptionForCustomer,
                'from_description' => $descriptionForVendor,
            ];
            $this->notificationController->Storenotification(new Request($data));

            // Provider → Admins
            $admins = User::where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');

                $tempdata['{{admin_name}}'] = $adminName;

                $descriptionForAdmin = Str::replace(array_keys($tempdata), array_values($tempdata), $messages['admin']);

                $dataAdmin = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $authId, // provider
                    'to_user_id' => $admin->id,             // admin
                    'to_description' => $descriptionForAdmin,
                    'from_description' => '',
                ];
                $this->notificationController->Storenotification(new Request($dataAdmin));
            }
        } catch (\Exception $e) {
            Log::error('Notification send failed (updateBookingProductStatus): ' . $e->getMessage());
        }
    }

    private function sendProductUpdateEmailToRecipient($source, $bookingdata, $settingData, $recipientEmail, $adminName = '')
    {
        if (!$recipientEmail) {
            return;
        }

        $gettemplate = Templates::select('templates.subject', 'templates.content')
            ->leftjoin('notification_types', 'notification_types.id', '=', 'templates.notification_type')
            ->where('notification_types.type', $source)
            ->where('templates.type', 1)
            ->where('templates.status', 1)
            ->first();

        if (!$gettemplate || !$bookingdata || !$bookingdata->booking) {
            return;
        }

        $productName = $bookingdata->source_name;
        $booking = $bookingdata->booking;

        $userName = ucwords($bookingdata->user_first_name . ' ' . $bookingdata->user_last_name);
        $providerName = ucwords($bookingdata->provider_first_name . ' ' . $bookingdata->provider_last_name);

        $tempdata = [
            '{{customer_name}}' => $userName,
            '{{user_name}}' => $userName,
            '{{admin_name}}' => $adminName,
            '{{booking_id}}' => $booking->order_id,
            '{{order_id}}' => $booking->order_id,
            '{{service_name}}' => $productName,
            '{{order_date}}' => formatDateTime($booking->booking_date),
            '{{provider_name}}' => $providerName,
            '{{contact}}' => $bookingdata->provideremail,
            '{{website_link}}' => $settingData['website'] ?? "",
            '{{transaction_date}}' => $bookingdata->trxdate,
            '{{company_name}}' => $settingData['company_name'] ?? "",
            '{{payment_method}}' => $bookingdata->paymenttype,
            '{{service_address}}' => $bookingdata['product']['createdBy']['userDetails']->address ?? $settingData['site_address'] ?? "",
        ];

        $finalContent = Str::replace(array_keys($tempdata), array_values($tempdata), $gettemplate->content);
        $subject = Str::replace(array_keys($tempdata), array_values($tempdata), $gettemplate->subject);

        $data = [
            'to_email' => $recipientEmail,
            'subject' => $subject,
            'content' => $finalContent
        ];

        try {
            $emailController = app(EmailController::class);
            $emailController->sendEmail(new Request($data));
        } catch (\Exception $e) {
            Log::error('Email sending failed to ' . $recipientEmail . ': ' . $e->getMessage());
        }
    }

    public function indexService(Request $request)
    {
        $providerId = Auth::id();

        $services = Product::select("source_name")
            ->where('user_id', $providerId)
            ->whereHas('bookings', function ($q) {
                $q->where('type', 'service');
            })
            ->distinct()
            ->get();


        return view('frontend.pages.vendor.service_bookings', [
            'services' => $services,
        ]);
    }

    public function fetchData(Request $request)
    {
        $request->validate(['status' => 'required|string']);

        $statusKey = $request->query('status');
        $bookings = $this->getServiceBookingsByStatus($request, $statusKey);
        $bookings->map(function ($bp) {
            $userId = $bp->user_id ?? null;

            $profileImagePath = DB::table('user_details')
                ->where('user_id', $userId)
                ->value('profile_image');

            $bp->user_profile_image = $profileImagePath;
            $bp->payment_type_label = getPaymentTypeLabel($bp->payment_type);

            return $bp;
        });

        return view('frontend.pages.vendor.service_booking_rows', [
            'bookings' => $bookings,
            'currentStatusKey' => $statusKey,
            'currency' => getDefaultCurrencySymbol()
        ]);
    }

    public function updateStatus(Request $request, Bookings $booking)
    {
        // Validate input
        $request->validate([
            'status' => 'required|integer|in:2,3,5', // 2: ongoing, 3: cancelled, 5: completed
            'reason' => 'nullable|string|max:500',
            'updated_by' => 'nullable|string' // could be 'staff' or null
        ]);

        if ($request->has('payment_status') && !empty($request->payment_status)) {
            $booking->payment_status = $request->payment_status;
        }

        // Update booking status
        $booking->booking_status = $request->status;
        $booking->save();

        // Initialize status variables
        $statusdata = [];
        $source = $sourceProvider = $sourceAdmin = $sourceStaff = '';
        $statusLabel = '';

        switch ($request->status) {
            case 2:
                $statusLabel = 'In progress';
                $source = 'Booking Accept';
                $sourceProvider = 'Booking Accept email to Provider';
                $sourceAdmin = 'Booking Accept email to Admin';
                $sourceStaff = 'Booking Accept email to Staff';
                break;

            case 3:
                $statusLabel = 'Cancelled';
                $source = 'Provider Booking Cancel';
                $sourceProvider = 'Provider Booking Cancel email to Provider';
                $sourceAdmin = 'Provider Booking Cancel email to Admin';
                $sourceStaff = 'Provider Booking Cancel email to Staff';
                break;

            case 5:
                $statusLabel = 'Completed';
                $source = 'Booking Completed';
                $sourceProvider = 'Booking Completed email to Provider';
                $sourceAdmin = 'Booking Completed email to Admin';
                $sourceStaff = 'Booking Completed email to Staff';
                break;
        }

        $statusdata['status'] = $statusLabel;
        $statusdata['status_code'] = $request->status;

        // Fetch booking data with all required joins for notifications
        $bookingdata = Bookings::select(
            'bookings.*',
            DB::raw("CASE
                WHEN bookings.booking_status = 1 THEN 'Open'
                WHEN bookings.booking_status = 2 THEN 'In progress'
                WHEN bookings.booking_status = 3 THEN 'Provider Cancelled'
                WHEN bookings.booking_status = 4 THEN 'Refund Initiated'
                WHEN bookings.booking_status = 5 THEN 'Completed'
                WHEN bookings.booking_status = 6 THEN 'Order Completed'
                WHEN bookings.booking_status = 7 THEN 'Refund Completed'
                WHEN bookings.booking_status = 8 THEN 'Customer Cancelled'
                ELSE 'Unknown'
            END AS booking_status_label"),
            DB::raw("DATE_FORMAT(bookings.created_at, '%d-%m-%Y') AS bookingdate"),
            DB::raw("TIME_FORMAT(bookings.from_time, '%H:%i') AS fromtime"),
            DB::raw("TIME_FORMAT(bookings.to_time, '%H:%i') AS totime"),
            'products.source_name',
            'users.name as user_name',
            'users.email as user_email',
            'provider.name as provider_name',
            'provider.email as provideremail',
            'staff.name as staff_name',
            'staff.email as staffemail'
        )
            ->leftJoin('products', 'products.id', '=', 'bookings.product_id')
            ->leftJoin('users', 'users.id', '=', 'bookings.user_id')
            ->leftJoin('users as provider', 'provider.id', '=', 'products.created_by')
            ->leftJoin('users as staff', 'staff.id', '=', 'bookings.staff_id')
            ->where('bookings.id', $booking->id)
            ->first();

        $settingData = getCommonSettingData([
            'company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website'
        ]);

        // Send email notifications
        // Customer
        $this->sendEmailToRecipient($source, $bookingdata, $settingData, $bookingdata->user_email ?? null);

        // Admin
        if ($sourceAdmin) {
            $admins = User::with('userDetails')->where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');
                $this->sendEmailToRecipient($sourceAdmin, $bookingdata, $settingData, $admin->email, $adminName);
            }
        }

        // Provider
        if ($sourceProvider && $bookingdata->provideremail) {
            $this->sendEmailToRecipient($sourceProvider, $bookingdata, $settingData, $bookingdata->provideremail);
        }

        // Staff
        if ($sourceStaff && $bookingdata->staffemail) {
            $this->sendEmailToRecipient($sourceStaff, $bookingdata, $settingData, $bookingdata->staffemail);
        }

        // Send notifications using the same logic as updateBookingStatus
        $this->sendNotification($source, $bookingdata, $settingData, $request->updated_by ?? '');

        return response()->json([
            'success' => true,
            'message' => "Booking {$statusLabel} and notifications sent successfully.",
            'data' => $statusdata
        ]);
    }

    private function sendEmailToRecipient($source, $bookingdata, $settingData, $recipientEmail, $adminName = '', $refundAmount = '')
    {
        if (!$recipientEmail) {
            return;
        }

        $gettemplate = Templates::select('templates.subject', 'templates.content')
            ->leftjoin('notification_types', 'notification_types.id', '=', 'templates.notification_type')
            ->where('notification_types.type', $source)
            ->where('templates.type', 1)
            ->where('templates.status', 1)
            ->first();

        if (!$gettemplate || !$bookingdata) {
            return;
        }

        $service = $bookingdata->source_name;
        $fromtime = $bookingdata->fromtime ?? "";
        $totime = $bookingdata->totime ?? "";

        $userName = ucwords($bookingdata->user_first_name . ' ' . $bookingdata->user_last_name);
        $providerName = ucwords($bookingdata->provider_first_name . ' ' . $bookingdata->provider_last_name);
        $staffName = isset($bookingdata->staff_first_name) ? ucwords($bookingdata->staff_first_name . ' ' . $bookingdata->staff_last_name) : '';

        $tempdata = [];

        if (!empty($refundAmount)) {
            $productName = $bookingdata->source_name;
            $booking = $bookingdata->booking;

            $tempdata = [
                '{{customer_name}}' => $userName,
                '{{user_name}}' => $userName,
                '{{admin_name}}' => $adminName,
                '{{booking_id}}' => $booking->order_id,
                '{{order_id}}' => $booking->order_id,
                '{{service_name}}' => $productName,
                '{{product_name}}' => $productName,
                '{{order_date}}' => formatDateTime($booking->booking_date),
                '{{provider_name}}' => $providerName,
                '{{contact}}' => $bookingdata->provideremail,
                '{{website_link}}' => $settingData['website'] ?? "",
                '{{transaction_date}}' => $bookingdata->trxdate,
                '{{refund_date}}' => $bookingdata->refunddate,
                '{{refund_id}}' => $booking->order_id,
                '{{refund_amount}}' => $refundAmount ?? '',
                '{{company_name}}' => $settingData['company_name'] ?? "",
                '{{payment_method}}' => $bookingdata->paymenttype,
                '{{service_address}}' => $bookingdata['product']['createdBy']['userDetails']->address ?? $settingData['site_address'] ?? "",
            ];
        } else {
            $tempdata = [
                '{{customer_name}}' => $userName,
                '{{user_name}}' => $userName,
                '{{admin_name}}' => $adminName,
                '{{booking_id}}' => $bookingdata->order_id,
                '{{service_name}}' => $bookingdata->source_name,
                '{{product_name}}' => $bookingdata->source_name,
                '{{appointment_date}}' => $bookingdata->bookingdate,
                '{{appointment_time}}' => $fromtime ? $fromtime . '-' . $totime : '',
                '{{provider_name}}' => $providerName,
                '{{staff_name}}' => $staffName,
                '{{contact}}' => $bookingdata->provideremail,
                '{{website_link}}' => $settingData['website'] ?? "",
                '{{refund_id}}' => $bookingdata->refundid,
                '{{refund_amount}}' => $refundAmount ?? '',
                '{{transaction_date}}' => $bookingdata->trxdate,
                '{{company_name}}' => $settingData['company_name'] ?? "",
                '{{refund_date}}' => $bookingdata->refunddate,
                '{{payment_method}}' => $bookingdata->paymenttype,
                '{{service_address}}' => $bookingdata['product']['createdBy']['userDetails']->address ?? $settingData['site_address'] ?? "",
            ];
        }


        $finalContent = Str::replace(array_keys($tempdata), array_values($tempdata), $gettemplate->content);
        $subject = Str::replace(array_keys($tempdata), array_values($tempdata), $gettemplate->subject);

        $data = [
            'to_email' => $recipientEmail,
            'subject' => $subject,
            'content' => $finalContent
        ];

        try {
            $emailController = app(EmailController::class);
            $emailController->sendEmail(new Request($data));
        } catch (\Exception $e) {
            Log::error('Email sending failed to ' . $recipientEmail . ': ' . $e->getMessage());
        }
    }

    private function sendNotification($source, $bookingdata, $settingData, $updatedBy = '')
    {
        $receipientType = 2;
        if (
            ($source == 'Provider Booking Cancel' || $source == 'Booking Accept' || $source == 'Booking Completed')
            && $updatedBy == 'staff'
        ) {
            $receipientType = 4;
        }

        $gettemplate = Templates::select('templates.subject', 'templates.content')
            ->leftjoin('notification_types', 'notification_types.id', '=', 'templates.notification_type')
            ->where('notification_types.type', $source)
            ->where('recipient_type', $receipientType)
            ->where('templates.type', 3)
            ->where('templates.status', 1)
            ->first();

        if ($gettemplate && $bookingdata) {

            $fromUserId = null;
            $toUserId   = null;

            if ($bookingdata->provideremail) {
                $fromUserId = User::where('email', $bookingdata->provideremail)->value('id');
            }

            if ($bookingdata->user_email) {
                $toUserId = User::where('email', $bookingdata->user_email)->value('id');
            }

            // Fallback if IDs not found
            $fromUserId = $fromUserId ?? $bookingdata->created_by;
            $toUserId   = $toUserId ?? $bookingdata->user_id;

            $fromtime = $bookingdata->fromtime ?? "";
            $totime   = $bookingdata->totime ?? "";

            $userName     = ucwords($bookingdata->user_name ?? '');
            $providerName = ucwords($bookingdata->provider_name ?? '');
            $staffName    = ucwords($bookingdata->staff_name ?? '');

            $tempdata = [
                '{{user_name}}'        => $userName,
                '{{customer_name}}'    => $userName,
                '{{booking_id}}'       => $bookingdata->order_id,
                '{{service_name}}'     => $bookingdata->source_name,
                '{{appointment_date}}' => $bookingdata->bookingdate,
                '{{appointment_time}}' => $fromtime ? $fromtime . '-' . $totime : '',
                '{{provider_name}}'    => $providerName,
                '{{staff_name}}'       => $staffName,
                '{{contact}}'          => $bookingdata->provideremail,
                '{{website_link}}'     => $settingData['website'] ?? "",
                '{{refund_id}}'        => $bookingdata->refundid,
                '{{refund_amount}}'    => $bookingdata->total_amount,
                '{{transaction_date}}' => $bookingdata->trxdate,
                '{{company_name}}'     => $settingData['company_name'] ?? "",
                '{{refund_date}}'      => $bookingdata->refunddate,
                '{{payment_method}}'   => $bookingdata->paymenttype,
                '{{service_address}}'  => $bookingdata['product']['createdBy']['userDetails']->address
                                        ?? $settingData['site_address'] ?? "",
            ];

            $todescription = Str::replace(array_keys($tempdata), array_values($tempdata), $gettemplate->content);

            $getfromtemplate = Templates::select('templates.subject', 'templates.content')
                ->leftjoin('notification_types', 'notification_types.id', '=', 'templates.notification_type')
                ->where('notification_types.type', $source)
                ->where('recipient_type', 1)
                ->where('templates.type', 3)
                ->where('templates.status', 1)
                ->first();

            $fromdescription = "";
            if ($getfromtemplate) {
                $fromdescription = Str::replace(array_keys($tempdata), array_values($tempdata), $getfromtemplate->content);
            }

            $data = [
                'communication_type' => '3',
                'source'             => $source,
                'reference_id'       => $bookingdata->id,
                'user_id'            => $fromUserId,
                'to_user_id'         => $toUserId,
                'to_description'     => $todescription,
                'from_description'   => $fromdescription,
            ];

            try {
                $this->notificationController->Storenotification(new Request($data));
            } catch (\Exception $e) {
                Log::error('Notification send failed: ' . $e->getMessage());
            }

            // Staff notification if exists
            if (isset($bookingdata->staff_id) && $bookingdata->staff_id != 0 && $bookingdata->staff_id != '' && $updatedBy != 'staff') {
                $staffData = [
                    'communication_type' => '3',
                    'source'             => $source,
                    'reference_id'       => $bookingdata->id,
                    'user_id'            => $bookingdata->user_id,
                    'to_user_id'         => $bookingdata->staff_id,
                    'from_description'   => '',
                    'to_description'     => $todescription,
                ];

                try {
                    $this->notificationController->Storenotification(new Request($staffData));
                } catch (\Exception $e) {
                    Log::error('Notification send failed for staff: ' . $e->getMessage());
                }
            }
        }

        // Send admin notifications
        $admins = User::where('user_type', 1)->get();

        $getrecipient3template = Templates::select('templates.subject', 'templates.content')
            ->leftjoin('notification_types', 'notification_types.id', '=', 'templates.notification_type')
            ->where('notification_types.type', $source)
            ->where('recipient_type', 3)
            ->where('templates.type', 3)
            ->where('templates.status', 1)
            ->first();

        if ($getrecipient3template) {
            $recipient3description = Str::replace(array_keys($tempdata), array_values($tempdata), $getrecipient3template->content);

            try {
                foreach ($admins as $admin) {
                    $adminData = [
                        'communication_type' => '3',
                        'source'             => $source,
                        'reference_id'       => $bookingdata->id,
                        'user_id'            => $fromUserId,
                        'to_user_id'         => $admin->id,
                        'to_description'     => $recipient3description,
                        'from_description'   => "",
                    ];
                    $this->notificationController->Storenotification(new Request($adminData));
                }
            } catch (\Exception $e) {
                Log::error('Notification send failed for admin: ' . $e->getMessage());
            }
        }
    }

    /**
     * Private helper to build the service booking query.
     */
    private function getServiceBookingsByStatus(Request $request, $statusKey)
    {
        $statusMap = [
            'new'        => 1,
            'ongoing'    => 2,
            'cancelled'  => 3,
            'completed'  => 5,
        ];

        $statusCode = $statusMap[$statusKey] ?? 1;
        $providerId = Auth::id();

        $query = Bookings::with(['user', 'product'])
            ->where('type', 'service')
            ->where('booking_status', $statusCode)
            ->whereHas('product', function ($query) use ($providerId) {
                $query->where('user_id', $providerId);
            })
            ->where(function ($q) {
                $q->where('payment_type', '!=', 5)
                ->where('payment_status', 2)
                ->orWhere(function ($q2) {
                    $q2->where('payment_type', 5)
                    ->whereIn('payment_status', [1, 2]);
                });
            });


        if ($request->filled('service_name')) {
            $query->whereHas('product', function ($q) use ($request) {
                $q->where('source_name', $request->service_name);
            });
        }

        if ($request->filled('search')) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('order_id', 'LIKE', "%{$searchTerm}%")
                    // CORRECTED: Use orWhereHas to search the related product's source_name
                    ->orWhereHas('product', function ($subQ) use ($searchTerm) {
                        $subQ->where('source_name', 'LIKE', "%{$searchTerm}%");
                    })
                    ->orWhereHas('user', function ($subQ) use ($searchTerm) {
                        $subQ->where('name', 'LIKE', "%{$searchTerm}%");
                    });
            });
        }

        return $query->latest()->paginate(10)->appends($request->query());
    }


    public function cancelCustomerOrder(Request $request)
    {
        $validated = $request->validate([
            'booking_id' => 'required|integer',
            'reason'     => 'required|string|max:255',
        ]);

        $user = Auth::user();
        $bookingProduct = BookingProduct::where('id', $validated['booking_id'])
            ->where('user_id', $user->id)
            ->first();

        if (!$bookingProduct) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found or you do not have permission to cancel it.'
            ], 404);
        }

        if (in_array($bookingProduct->product_status, [3, 4])) {
            return response()->json([
                'success' => false,
                'message' => __('cancel_failed')
            ], 400);
        }

        $bookingProduct->cancellation_reason = $validated['reason'];
        $bookingProduct->product_status = 6; // 6 = User Cancelled
        $bookingProduct->save();

        BookingHistory::create([
            'booking_product_id' => $bookingProduct->id,
            'status'   => 'cancelled',
            'message'  => 'Order cancelled by customer. Reason: ' . $validated['reason'],
            'datetime' => now(),
        ]);

        $source        = 'Order Cancelled by Customer';
        $sourceAdmin   = 'Customer Cancelled email to Admin';
        $sourceProvider= 'Customer Cancelled email to Provider';
        $sourceStaff   = 'Customer Cancelled email to Staff';
        $description   = 'Order cancelled by customer.';

        $bookingdata = $bookingProduct->load([
            'booking.user',
            'booking.staff',
            'product.createdBy'
        ])->booking;

        $settingData = getCommonSettingData([
            'company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website'
        ]);

        // --- Send emails ---
        $this->sendEmailToRecipient($source, $bookingdata, $settingData, $bookingdata->user->email ?? null);

        // Admins
        if ($sourceAdmin) {
            $admins = User::with('userDetails')->where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');
                $this->sendEmailToRecipient($sourceAdmin, $bookingdata, $settingData, $admin->email, $adminName);
            }
        }

        // Provider
        if ($sourceProvider && $bookingProduct->product->createdBy?->email) {
            $this->sendEmailToRecipient($sourceProvider, $bookingdata, $settingData, $bookingProduct->product->createdBy->email);
        }

        // Staff
        if ($sourceStaff && $bookingdata->staff?->email) {
            $this->sendEmailToRecipient($sourceStaff, $bookingdata, $settingData, $bookingdata->staff->email);
        }

        // --- Send Notifications ---
        try {

            // Customer → Provider
            if ($bookingProduct->product->createdBy) {
                $data = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $user->id,                                 // customer
                    'to_user_id' => $bookingProduct->product->createdBy->id, // provider
                    'to_description' => $description,
                    'from_description' => $description,
                ];
                $this->notificationController->Storenotification(new Request($data));
            }

            // Customer → Admins
            $admins = User::where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $dataAdmin = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $user->id,      // customer
                    'to_user_id' => $admin->id,  // admin
                    'to_description' => $description,
                    'from_description' => '',
                ];
                $this->notificationController->Storenotification(new Request($dataAdmin));
            }
        } catch (\Exception $e) {
            Log::error('Notification send failed (cancelCustomerOrder): ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => __('cancel_success')
        ]);
    }

    public function refundCancelCustomerOrder(Request $request)
    {
        $validated = $request->validate([
            'id' => 'required|integer',
        ]);

        $authUserId = Auth::id();
        $bookingProductId = $validated['id'];
        $bookingProduct = BookingProduct::where('id', $bookingProductId)
            ->where('user_id', $authUserId)
            ->first();

        if (!$bookingProduct) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found or you do not have permission.'
            ], 404);
        }

        $bookingProduct->product_status = 7;
        $bookingProduct->save();

        $getbookings = Bookings::where('id', '=', $bookingProduct->booking_id)->first();
        if (isset($getbookings)) {
            $data['user_id'] = $authUserId;
            $data['reference_id'] = $getbookings->id;
            $data['booking_product_id'] = $bookingProductId;
            $data['type'] = 2;
            $data['total_bookings'] = 1;
            $data['process_amount'] = $bookingProduct->gross_total ?? 0;
            $data['total_earnings'] = $bookingProduct->gross_total ?? 0;
            $data['created_by'] = $authUserId;
            $data['created_at'] = Carbon::now();
            PayoutHistory::insertGetId($data);
        }
        $productName = $bookingProduct->product_name;
        $refundAmount = $bookingProduct->gross_total ?? 0.00;

        BookingHistory::create([
            'booking_product_id' => $bookingProduct->id,
            'status'   => 'refund_initiated',
            'message'  => 'Refund has been initiated ' . $productName . ' for the cancelled order.',
            'datetime' => now(),
        ]);

        $source        = 'Refund Initiated';
        $sourceAdmin   = 'Refund Initiated email to Admin';
        $sourceProvider= 'Refund Initiated email to Provider';
        $sourceStaff   = 'Refund Initiated email to Staff';
        $description = 'A refund has been initiated for ' . $productName . ' due to the cancelled order.';

        $bookingdata = BookingProduct::with('booking')
            ->select(
                'booking_products.*',
                'products.source_name',
                'users.name as user_name',
                'users.email as user_email',
                'provider.id as provider_id',
                'provider.email as provideremail',
                'user_details.first_name as user_first_name',
                'user_details.last_name as user_last_name',
                'provider_details.first_name as provider_first_name',
                'provider_details.last_name as provider_last_name',
            )
            ->leftJoin('products', 'products.id', '=', 'booking_products.product_id')
            ->leftJoin('users', 'users.id', '=', 'booking_products.user_id')
            ->leftJoin('user_details', 'user_details.user_id', '=', 'users.id')
            ->leftJoin('users as provider', 'provider.id', '=', 'products.created_by')
            ->leftJoin('user_details as provider_details', 'provider_details.user_id', '=', 'provider.id')
            ->where('booking_products.id', $bookingProduct->id)
            ->first();

        $settingData = getCommonSettingData([
            'company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website'
        ]);

        // --- Send emails ---
        $this->sendEmailToRecipient($source, $bookingdata, $settingData, $bookingdata->user_email ?? null, '', $refundAmount);

        // Admins
        if ($sourceAdmin) {
            $admins = User::with('userDetails')->where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $adminName = $admin->userDetails
                    ? trim(ucwords(($admin->userDetails->first_name ?? '') . ' ' . ($admin->userDetails->last_name ?? '')))
                    : ucwords($admin->name ?? 'Admin');
                $this->sendEmailToRecipient($sourceAdmin, $bookingdata, $settingData, $admin->email, $adminName, $refundAmount);
            }
        }

        // Provider
        if ($sourceProvider && $bookingdata->provideremail) {
            $this->sendEmailToRecipient($sourceProvider, $bookingdata, $settingData, $bookingdata->provideremail, '', $refundAmount);
        }

        // --- Send Notifications ---
        try {

            // Customer → Provider
            if ($bookingdata->provider_id) {
                $data = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $authUserId,                                 // customer
                    'to_user_id' => $bookingdata->provider_id, // provider
                    'to_description' => $description,
                    'from_description' => $description,
                ];
                $this->notificationController->Storenotification(new Request($data));
            }

            // Customer → Admins
            $admins = User::where('user_type', 1)->get();
            foreach ($admins as $admin) {
                $dataAdmin = [
                    'communication_type' => '3',
                    'source' => $source,
                    'reference_id' => $bookingdata->id,
                    'user_id' => $authUserId,      // customer
                    'to_user_id' => $admin->id,  // admin
                    'to_description' => $description,
                    'from_description' => '',
                ];
                $this->notificationController->Storenotification(new Request($dataAdmin));
            }
        } catch (\Exception $e) {
            Log::error('Notification send failed (refundCancelCustomerOrder): ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Refund Initiated successfully.'
        ]);
    }

    public function order_invoice_list(Request $request): View
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('Invoices'),
            'header' => __('Invoices'),
        ];
        return view('frontend.pages.customer.order_invoice_list', compact('data'));
    }

    public function invoice_detail_list(Request $request): JsonResponse
    {
        try {
            $user = Auth::user();
            $type = $request->type ?? 'service';

            $statusMap = [
                1 => __('unpaid'),
                2 => __('paid'),
                3 => __('refund'),
            ];

            $perPage = $request->get('length', 10);
            $start   = $request->get('start', 0);
            $page    = ($start / $perPage) + 1;
            $sortBy = $request->sortBy ?? 'newest';
            $totalBookings = 0;
            $data = [];

            if ($type == 'service') {
                $query = Bookings::with([
                    'invoice_product',
                ])
                ->select('bookings.*')
                ->where('type', 'service')
                ->whereIn('booking_status', [1, 2, 3, 5, 6]);

                // Sorting
                if ($sortBy == 'oldest') {
                    $query->orderBy('bookings.created_at', 'asc');
                } else {
                    $query->orderBy('bookings.created_at', 'desc');
                }

                // Date range filter
                if ($request->filled('date_range')) {
                    $dates = explode(' - ', $request->date_range);
                    if (count($dates) == 2) {
                        $start = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                        $end   = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                        $query->whereBetween('bookings.created_at', [$start, $end]);
                    }
                }

                // Role-based filter
                if ($user->user_type == 3) {
                    // customer → their own bookings
                    $query->where('bookings.user_id', $user->id);
                }

                $bookings = $query->paginate($perPage, ['*'], 'page', $page);

                foreach ($bookings as $bp) {
                    $product = $bp->invoice_product;
                    $end_date = \Carbon\Carbon::parse($bp->booking_date)->addDays(7);

                    $data[] = [
                        'order_id'     => $bp->order_id,
                        'invoice_date' => formatDateTime($bp->created_at),
                        'due_date'     => formatDateTime($bp->booking_date),
                        'user_id'      => $bp->user_id,
                        'amount'       => getDefaultCurrencySymbol() . number_format($bp->total_amount, 2, '.', ''),
                        'payment_status' => $bp->payment_status,
                        'payment_status_label' => $statusMap[$bp->payment_status] ?? '-',
                        'description' => $product->source_name ?? '-',
                    ];
                }

                $totalBookings = $bookings->total();
            } else {
                // Start from booking_products
                $query = BookingProduct::with([
                    'booking',        // relation to bookings
                    'product'         // relation to product if needed
                ])
                ->select('booking_products.*')
                ->whereIn('product_status', [1, 2, 3, 4]);

                // Sorting (based on booking date)
                if ($sortBy == 'oldest') {
                    $query->join('bookings', 'booking_products.booking_id', '=', 'bookings.id')
                        ->orderBy('bookings.created_at', 'asc');
                } else {
                    $query->join('bookings', 'booking_products.booking_id', '=', 'bookings.id')
                        ->orderBy('bookings.created_at', 'desc');
                }

                // Date range filter (from bookings table)
                if ($request->filled('date_range')) {
                    $dates = explode(' - ', $request->date_range);
                    if (count($dates) == 2) {
                        $startDate = \Carbon\Carbon::createFromFormat('j M y', trim($dates[0]))->startOfDay();
                        $endDate   = \Carbon\Carbon::createFromFormat('j M y', trim($dates[1]))->endOfDay();
                        $query->whereHas('booking', function ($q) use ($startDate, $endDate) {
                            $q->whereBetween('created_at', [$startDate, $endDate]);
                        });
                    }
                }

                // Role-based filter
                if ($user->user_type === 3) {
                    // customer → their own bookings
                    $query->whereHas('booking', function ($q) use ($user) {
                        $q->where('user_id', $user->id);
                    });
                } elseif ($user->user_type === 2) {
                    // provider → only their products
                    $query->where('provider_id', $user->id);
                }

                $bookingProducts = $query->paginate($perPage, ['*'], 'page', $page);

                // Group by booking_id
                $groupedByBooking = $bookingProducts->groupBy('booking_id');

                foreach ($groupedByBooking as $bookingId => $bpCollection) {
                    $booking = $bpCollection->first()->booking;
                    $end_date = \Carbon\Carbon::parse($booking->booking_date)->addDays(7);

                    $firstProduct = $bpCollection->first();
                    $description = $firstProduct ? $firstProduct->product_name : '';

                    // Prepare booking level data
                    $bookingData = [
                        'order_id'     => $booking->order_id,
                        'invoice_date' => formatDateTime($booking->booking_date),
                        'due_date'     => formatDateTime($end_date),
                        'user_id'      => $booking->user_id,
                        'amount'       => getDefaultCurrencySymbol() . number_format($bpCollection->sum('gross_total'), 2, '.', ''),
                        'payment_status' => $booking->payment_status,
                        'payment_status_label' => $statusMap[$booking->payment_status] ?? '-',
                        'description'  => $description,
                        'provider'    => [],
                    ];

                    // Group products by provider_id
                    $groupedByProvider = $bpCollection->groupBy('provider_id');

                    foreach ($groupedByProvider as $providerId => $products) {
                        $providerProducts = [];
                        foreach ($products as $bp) {
                            $providerProducts[] = [
                                'product_name' => $bp->product_name,
                                'price'        => $bp->price,
                                'quantity'     => $bp->quantity,
                                'subtotal'     => $bp->subtotal,
                                'gross_total'  => $bp->gross_total,
                            ];
                        }

                        $bookingData['provider'][] = [
                            'provider_id' => $providerId,
                            'products'    => $providerProducts,
                        ];
                    }

                    $data[] = $bookingData;
                    $totalBookings = $bookingProducts->total();
                }

            }

            return response()->json([
                'success'         => true,
                'message'         => __('order_list_success'),
                'draw'            => intval($request->get('draw')),
                'recordsTotal'    => $totalBookings,
                'recordsFiltered' => $totalBookings,
                'data'            => $data,
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error fetching orders: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => __('order_list_error'),
                'error'   => $e->getMessage()
            ], 500);
        }
    }

    public function reviews(Request $request): View
    {
        $data = [
            'menu' => '<a href="' . url('/') . '">' . __('Home') . '</a>',
            'submenu' => __('Reviews'),
            'header' => __('Reviews'),
        ];

        return view('frontend.pages.customer.review', compact('data'));
    }

    public function accountSettings(): View
    {
        $userId = currentUser()->id;
        $accountSettingsData = AdditionalSetting::where('user_id', $userId)->where('key', 'account_settings')->first();
        $accountSettings = $accountSettingsData ? json_decode($accountSettingsData->value, true) : [];

        return view('frontend.pages.customer.account_settings', compact('accountSettings'));
    }

    public function saveAccountSettings(Request $request): JsonResponse
    {
        $result = $this->customerRepository->saveAccountSettings($request);

        if ($result['status'] === 'error') {
            return response()->json($result, $result['code']);
        }

        return response()->json($result);
    }

    public function accountSettingsDefaultStatus(Request $request): JsonResponse
    {
        $result = $this->customerRepository->accountSettingsDefaultStatus($request);

        if ($result['status'] === 'error') {
            return response()->json($result, $result['code']);
        }

        return response()->json($result);
    }

    public function deleteAccount(Request $request): View
    {
        $authId = Auth::id();
        $totalAmount = WalletHistory::where('user_id', $authId)->where('status', 'completed')->where('type', '1')->sum('amount');
        $totalAmountdebit = WalletHistory::where('user_id', $authId)->where('status', 'completed')->where('type', '2')->sum('amount');
        $walletTotalAmount = number_format($totalAmount - $totalAmountdebit, 2, '.', '');
        $serviceBookingIncompleteCount = Bookings::where('type', 'service')
            ->where('user_id', $authId)
            ->whereIn('booking_status', [1, 2])
            ->count();

        $serviceBookingRefundCount = Bookings::where('type', 'service')
            ->where('user_id', $authId)
            ->where('booking_status', 4)
            ->count();

        $productOrderIncompleteCount = BookingProduct::where('user_id', $authId)
            ->whereIn('product_status', [1, 2, 3])
            ->count();

        $productOrderRefundCount = BookingProduct::where('user_id', $authId)
            ->where('product_status', 7)
            ->count();

        return view('frontend.pages.customer.delete_account', compact('walletTotalAmount', 'serviceBookingIncompleteCount', 'serviceBookingRefundCount', 'productOrderIncompleteCount', 'productOrderRefundCount'));
    }
}
