<?php

namespace App\Http\Controllers;

use Modules\Categories\app\Models\Categories;
use Modules\Product\app\Models\Product;
use App\Models\Bookings;
use App\Models\TaxRate;
use Modules\Service\app\Models\Service;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Repositories\Contracts\BookRepositoryInterface;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Carbon;

class BookController extends Controller
{
    protected $bookRepository;
    public function __construct(BookRepositoryInterface $bookRepository)
    {
        $this->bookRepository = $bookRepository;
    }

    public function index(Request $request): View
    {
        return view('admin.dashboard'); // Return the view for the dashboard
    }

    public function productlist(Request $request): View
    {
        $products = Product::query()->where('source_type', '=', 'service')->get();
        $products = DB::table('products')
            ->select('products.id', 'products.slug', 'products.source_name', 'products_meta.source_Values')
            ->join('products_meta', 'products.id', '=', 'products_meta.product_id')
            ->where(['products_meta.source_key' => 'product_image'])
            ->get();

        $email = "";
        if (Auth::check()) {
            $email = Auth::user()->email;
        }
        $data = [
            'email' => $email,
        ];
        return view('services', compact('data', 'products'));
    }

    public function productdetail(Request $request): View
    {
        $response = $this->bookRepository->productdetail($request);
        return $response;
    }

    public function showFormCategories(Request $request): View
    {
        $categoryId = $request->session()->get('category_id');

        $categoryName = Categories::where('id', $categoryId)->value('name');

        return view('admin.form-categories', compact('categoryId', 'categoryName'));
    }

    public function getServiceDetails($product_id)
    {
        $service = Service::where('id', $product_id)->first();

        if ($service) {
            return response()->json([
                'success' => true,
                'data' => $service
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Service not found'
            ], 404);
        }
    }

    public function serviceBooking(Request $request)
    {
        $response = $this->bookRepository->serviceBooking($request);
        return $response;
    }

    public function serviceIndexBooking(Request $request)
    {
        $response = $this->bookRepository->serviceIndexBooking($request);
        return $response;
    }

    public function productIndexBooking(Request $request)
    {
        $response = $this->bookRepository->productIndexBooking($request);
        return $response;
    }

    public function getStaffs(Request $request)
    {
        $response = $this->bookRepository->getStaffs($request);
        return $response;
    }

    public function getInfo(Request $request)
    {
        $response = $this->bookRepository->getInfo($request);
        return $response;
    }

    public function getPersonalInfo(Request $request)
    {
        $response = $this->bookRepository->getPersonalInfo($request);
        return $response;
    }

    public function getSlot(Request $request)
    {
        $response = $this->bookRepository->getSlot($request);
        return $response;
    }

    public function getSlots(Request $request)
    {
        $response = $this->bookRepository->getSlots($request);
        return $response;
    }


    public function getSlotInfo(Request $request)
    {
        $response = $this->bookRepository->getSlotInfo($request);
        return $response;
    }

    public function getPayout(Request $request)
    {
        $response = $this->bookRepository->getPayout($request);
        return $response;
    }

    public function getPayoutApi(Request $request)
    {
        $response = $this->bookRepository->getPayoutApi($request);
        return $response;
    }

    public function payment(Request $request)
    {
        $response = $this->bookRepository->payment($request);
        return $response;
    }

    public function productPayment(Request $request)
    {
        $response = $this->bookRepository->productPayment($request);
        return $response;
    }

    public function calculateProductSummary(Request $request)
    {
        $finalTotal = $request->subtotal; // comes from frontend
        $couponData = $request->couponData ?? []; // array of coupon details
        $totalCouponDiscount = 0;
        $couponDetails = [];

        // Apply coupon discount(s)
        if (!empty($couponData)) {
            foreach ($couponData as $coupon) {
                $price = (float) $coupon['product_price'];
                $productId = $coupon['product_id'];

                if ($coupon['coupon_type'] === 'fixed') {
                    $discount = min((float) $coupon['coupon_value'], $price);
                } elseif ($coupon['coupon_type'] === 'percentage') {
                    $discount = ($price * (float) $coupon['coupon_value']) / 100;
                } else {
                    $discount = 0;
                }

                $totalCouponDiscount += $discount;

                $couponDetails[] = [
                    'product_id'    => $productId,
                    'product_price' => $price,
                    'coupon_discount_amount' => $discount,
                    'coupon_id'     => $coupon['coupon_id'] ?? null,
                    'coupon_type'  => $coupon['coupon_type'] ?? null,
                    'coupon_value' => number_format($coupon['coupon_value'] ?? 0, 2, '.', ''),
                    'coupon_code'  => $coupon['coupon_code'] ?? null,
                ];
            }
        }

        // Subtotal after applying coupons
        $discountedSubtotal = max($finalTotal - $totalCouponDiscount, 0);

        // Calculate taxes on discounted subtotal
        $taxes = TaxRate::where('status', 1)
            ->where('source_type', 'product')
            ->get();

        $taxDetails = [];
        $totalTax = 0;

        foreach ($taxes as $tax) {
            $rate = $tax->tax_rate; // e.g. 10.00 for 10%
            $taxAmount = ($discountedSubtotal * $rate) / 100;

            $taxDetails[] = [
                'name'   => $tax->tax_name,
                'rate'   => $rate,
                'amount' => number_format($taxAmount, 2, '.', '')
            ];

            $totalTax += $taxAmount;
        }

        $grandTotal = $discountedSubtotal + $totalTax;

        return response()->json([
            'success' => true,
            'code'    => 200,
            'data'    => [
                'subtotal'                     => number_format($finalTotal, 2, '.', ''),
                'coupon_discounted_subtotal'   => number_format($discountedSubtotal, 2, '.', ''),
                'coupon_details'               => $couponDetails,
                'total_coupon_discount_amount' => number_format($totalCouponDiscount, 2, '.', ''),
                'taxes'                        => $taxDetails,
                'total_tax'                    => number_format($totalTax, 2, '.', ''),
                'grand_total'                  => number_format($grandTotal, 2, '.', ''),
            ],
            'message' => 'Summary calculated successfully',
        ], 200);
    }

    public function checkProductCoupon(Request $request)
    {
        $couponCode = $request->input('coupon_code');
        $products   = $request->input('products', []); // now contains product_id + price
        $currentDate = Carbon::today();
        $userId = currentUser()->id;

        // Find coupon
        $coupon = DB::table('coupons')
            ->select('id', 'code', 'coupon_type', 'coupon_value', 'quantity', 'quantity_value', 'product_type', 'category_id', 'subcategory_id', 'product_id')
            ->where('status', 1)
            ->whereRaw('BINARY `code` = ?', [$couponCode])
            ->whereDate('start_date', '<=', $currentDate)
            ->whereDate('end_date', '>=', $currentDate)
            ->whereNull('deleted_at')
            ->orderBy('id', 'desc')
            ->first();

        if (!$coupon) {
            return response()->json([
                'success' => false,
                'message' => __('coupon_is_not_valid'),
            ], 422);
        }

        $alreadyUsed = DB::table('coupon_logs')
            ->where('user_id', $userId)
            ->where('coupon_id', $coupon->id)
            ->whereNull('deleted_at')
            ->exists();

        if ($alreadyUsed) {
            return response()->json([
                'success' => false,
                'message' => __('coupon_already_used'),
            ], 422);
        }

        $validProducts = [];

        foreach ($products as $product) {
            $pid   = (int) $product['product_id'];
            $price = (float) $product['price'];

            if ($coupon->product_type === 'all'
                || (isset($coupon->product_id) && in_array($pid, explode(',', $coupon->product_id)))
                || (isset($coupon->subcategory_id) && in_array($request->subcategory_id, explode(',', $coupon->subcategory_id)))
                || (isset($coupon->category_id) && in_array($request->category_id, explode(',', $coupon->category_id)))
            ) {
                $validProducts[] = [
                    'product_id'   => $pid,
                    'product_price' => $price,
                    'coupon_id'    => $coupon->id,
                    'coupon_code'  => $coupon->code,
                    'coupon_type'  => $coupon->coupon_type,
                    'coupon_value' => $coupon->coupon_value,
                ];
            }
        }

        if (empty($validProducts)) {
            return response()->json([
                'success' => false,
                'message' => __('coupon_is_not_applicable'),
            ], 422);
        }

        return response()->json([
            'success' => true,
            'message' => __('coupon_applied_success'),
            'data' => $validProducts
        ], 200);
    }

    public function paypalPaymentSuccess(Request $request)
    {
        $response = $this->bookRepository->paypalPaymentSuccess($request);
        return $response;
    }

    public function stripPaymentSuccess(Request $request)
    {
        $response = $this->bookRepository->stripPaymentSuccess($request);
        return $response;
    }

    public function sucesspaymentMollie(Request $request): View
    {
        Bookings::where('tranaction', Session('paymentid'))->update(['payment_status' => 2]);
        return view('user.booking.service_success_two');
    }

    public function successTwo(Request $request): View
    {
        return view("user.booking.service_success_two");
    }

    public function checkProductUser(Request $request)
    {
        $response = $this->bookRepository->checkProductUser($request);
        return $response;
    }

    public function productPaypalSuccess(Request $request)
    {
        $response = $this->bookRepository->productPaypalSuccess($request);
        return $response;
    }

    public function productStripPaymentSuccess(Request $request)
    {
        $response = $this->bookRepository->productStripPaymentSuccess($request);
        return $response;
    }

    public function productSuccess(Request $request)
    {
        $orderId = $request->query('order_id');

        $booking = Bookings::with('products')->where('order_id', $orderId)->where('user_id', Auth::id())->first();

        if (!$booking) {
            return redirect('/')->with('error', 'Booking not found.');
        }

        $currecy_details = getDefaultCurrencySymbol();

        return view('user.booking.product_success', compact('booking', 'currecy_details'));
    }

    public function productSuccessProduct(Request $request)
    {
        $orderId = $request->query('order_id');

        $booking = Bookings::with('products')->where('order_id', $orderId)->where('user_id', Auth::id())->first();

        if (!$booking) {
            return redirect('/')->with('error', 'Booking not found.');
        }

        $currecy_details = getDefaultCurrencySymbol();

        return view('user.booking.product_success', compact('booking', 'currecy_details'));
    }
}
