<?php

namespace Modules\Testimonials\app\Support;

use Closure;
use Illuminate\Support\Facades\Cache;

class TestimonialsCache
{
    private const CACHE_PREFIX = 'testimonials_module';
    private const CACHE_KEYS_KEY = self::CACHE_PREFIX . ':keys';
    private const DEFAULT_TTL = 600; // 10 minutes

    /**
     * Remember the cached value for the given key.
     */
    public static function remember(string $key, Closure $callback, ?int $ttl = null)
    {
        $ttl = $ttl ?? self::DEFAULT_TTL;
        $fullKey = self::CACHE_PREFIX . ':' . $key;

        if (method_exists(Cache::getStore(), 'tags')) {
            return Cache::tags(self::CACHE_PREFIX)->remember($fullKey, $ttl, function () use ($callback, $fullKey) {
                $value = $callback();
                self::trackKey($fullKey);

                return $value;
            });
        }

        if (Cache::has($fullKey)) {
            return Cache::get($fullKey);
        }

        $value = $callback();
        Cache::put($fullKey, $value, $ttl);
        self::trackKey($fullKey);

        return $value;
    }

    /**
     * Flush all cached testimonial keys.
     */
    public static function flush(): void
    {
        if (method_exists(Cache::getStore(), 'tags')) {
            Cache::tags(self::CACHE_PREFIX)->flush();
            Cache::forget(self::CACHE_KEYS_KEY);

            return;
        }

        $keys = Cache::get(self::CACHE_KEYS_KEY, []);
        foreach ($keys as $key) {
            Cache::forget($key);
        }

        Cache::forget(self::CACHE_KEYS_KEY);
    }

    private static function trackKey(string $key): void
    {
        $keys = Cache::get(self::CACHE_KEYS_KEY, []);
        if (! in_array($key, $keys, true)) {
            $keys[] = $key;
            Cache::forever(self::CACHE_KEYS_KEY, $keys);
        }
    }
}
