<?php

namespace Modules\Testimonials\app\Repositories\Eloquent;

use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Testimonials\app\Models\Testimonial;
use Modules\Testimonials\app\Repositories\Contracts\TestimonialRepositoryInterface;
use Modules\Testimonials\app\Support\TestimonialsCache;

class TestimonialRepository implements TestimonialRepositoryInterface
{
    public function getAll(Request $request): Collection
    {
        $orderBy = $request->input('order_by', 'desc');
        $sortBy = $request->input('sort_by', 'order_by');

        return TestimonialsCache::remember($this->cacheKey($sortBy, $orderBy), function () use ($sortBy, $orderBy) {
            return Testimonial::orderBy($sortBy, $orderBy)
                ->get()
                ->map(function (Testimonial $testimonial) {
                    $testimonial->client_image = $testimonial->client_image
                        ? $testimonial->file($testimonial->client_image)
                        : url('assets/img/user-default.jpg');

                    return $testimonial;
                });
        });
    }

    public function store(Request $request): array
    {
        $data = method_exists($request, 'validated') ? $request->validated() : $request->only([
            'client_name',
            'position',
            'description',
            'status',
            'id',
            'method',
        ]);

        $langCode = $request->input('language_code', 'en');
        $isCreate = strtolower((string) $request->input('method', 'add')) === 'add';

        if ($isCreate) {
            $last = Testimonial::latest('order_by')->first();
            $data['order_by'] = $last ? $last->order_by + 1 : 1;

            if ($request->hasFile('client_image')) {
                $data['client_image'] = $this->uploadImage($request->file('client_image'));
            }

            $payload = [
                'client_name' => $data['client_name'],
                'position' => $data['position'],
                'description' => $data['description'],
                'status' => (int) $data['status'],
                'order_by' => $data['order_by'],
            ];

            if (! empty($data['client_image'])) {
                $payload['client_image'] = $data['client_image'];
            }

            Testimonial::create($payload);

            $this->flushCache();

            return ['message' => __('testimonial_create_success', [], $langCode)];
        }

        $testimonial = Testimonial::findOrFail($request->input('id'));

        if ($request->hasFile('client_image')) {
            $this->deleteImage($testimonial->client_image);
            $data['client_image'] = $this->uploadImage($request->file('client_image'));
        }

        $payload = [
            'client_name' => $data['client_name'],
            'position' => $data['position'],
            'description' => $data['description'],
            'status' => (int) $data['status'],
        ];

        if (! empty($data['client_image'])) {
            $payload['client_image'] = $data['client_image'];
        }

        $testimonial->update($payload);

        $this->flushCache();

        return ['message' => __('testimonial_update_success', [], $langCode)];
    }

    public function destroy(Request $request): array
    {
        $langCode = $request->input('language_code', 'en');

        $testimonial = Testimonial::findOrFail($request->input('id'));
        $this->deleteImage($testimonial->client_image);
        $testimonial->delete();

        $this->flushCache();

        return ['message' => __('testimonial_delete_success', [], $langCode)];
    }

    public function statusChange(Request $request): array
    {
        $langCode = $request->input('language_code', 'en');

        $testimonial = Testimonial::findOrFail($request->input('id'));
        $testimonial->update(['status' => (int) $request->input('status')]);

        $this->flushCache();

        return ['message' => __('testimonial_status_success', [], $langCode)];
    }

    protected function uploadImage($file): string
    {
        $filename = Str::uuid() . '_' . time() . '.' . $file->getClientOriginalExtension();
        $file->storeAs('testimonials', $filename, 'public');

        return $filename;
    }

    protected function deleteImage(?string $filename): void
    {
        if ($filename && Storage::disk('public')->exists('testimonials/' . $filename)) {
            Storage::disk('public')->delete('testimonials/' . $filename);
        }
    }

    private function cacheKey(string $sortBy, string $orderBy): string
    {
        return "list:{$sortBy}:{$orderBy}";
    }

    private function flushCache(): void
    {
        TestimonialsCache::flush();
    }
}
