<?php

namespace Modules\Shops\app\Repositories\Eloquent;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Modules\Shops\app\Models\Shops;
use Modules\Shops\app\Repositories\Contracts\ShopRepositoryInterface;

class ShopRepository implements ShopRepositoryInterface
{
    private const CACHE_TTL = 3600;

    public function index(Request $request): array
    {
        $search = $request->input('search');
        $perPage = $request->input('per_page', 9);
        $userId = Auth::id();
        $languageId = getLanguageId(app()->getLocale());

        $baseQuery = Shops::where('created_by', $userId)
            ->where('language_id', $languageId)
            ->with(['countryRelation', 'cityRelation'])
            ->withCount(['actualProducts', 'services']);

        if ($search) {
            $baseQuery->where(function ($query) use ($search) {
                $query->where('shop_name', 'like', "%{$search}%")
                    ->orWhere('owner_name', 'like', "%{$search}%");
            });
        }

        $activeShops = (clone $baseQuery)->where('status', 1)->latest()->paginate($perPage, ['*'], 'active_page');
        $inactiveShops = (clone $baseQuery)->where('status', 0)->latest()->paginate($perPage, ['*'], 'inactive_page');

        return compact('activeShops', 'inactiveShops');
    }

    // START: FIX - Implementation of new methods for public shop filtering

    /**
     * Get filtered shops for the public frontend.
     *
     * @param Request $request
     * @return LengthAwarePaginator
     */
    public function getFilteredShops(Request $request): LengthAwarePaginator
    {
        $query = Shops::query()
            ->where('status', 1)
            ->with(['cityRelation', 'countryRelation'])
            ->withCount(['actualProducts as actual_products_count', 'services']);

        // Search Filter
        if ($request->filled('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('shop_name', 'like', "%{$searchTerm}%")
                    ->orWhere('owner_name', 'like', "%{$searchTerm}%");
            });
        }

        // Location Filter (by city id)
        if ($request->filled('locations') && is_array($request->locations)) {
            $query->whereIn('city', $request->locations);
        }

        // Vendor Filter (by user id)
        if ($request->filled('vendors') && is_array($request->vendors)) {
            $query->whereIn('created_by', $request->vendors);
        }

        // Reviews Filter (by star rating)
        if ($request->filled('reviews') && is_array($request->reviews)) {
            $query->whereIn(DB::raw('FLOOR(rating)'), $request->reviews);
        }

        // Sorting
        $sortBy = $request->input('sort_by', 'recent');
        switch ($sortBy) {
            case 'oldest':
                $query->oldest();
                break;
            case 'newest':
            case 'recent':
            default:
                $query->latest();
                break;
        }

        return $query->paginate(9);
    }

    /**
     * Get data for the filter sidebar.
     *
     * @return array
     */
    public function getShopFilterData(): array
    {
        // Assuming your 'users' table has 'name' and you want to show shop creators as vendors.
        return Cache::remember('shops:filter:data', self::CACHE_TTL, function () {
            $vendors = DB::table('shops')
                ->join('users', 'shops.created_by', '=', 'users.id')
                ->where('shops.status', 1)
                ->select('users.id', 'users.name', DB::raw('count(shops.id) as total'))
                ->groupBy('users.id', 'users.name')
                ->orderBy('users.name')
                ->get();

            $locations = DB::table('shops')
                ->join('cities', 'shops.city', '=', 'cities.id')
                ->where('shops.status', 1)
                ->whereNotNull('shops.city')
                ->select('cities.id', 'cities.name', DB::raw('count(shops.id) as total'))
                ->groupBy('cities.id', 'cities.name')
                ->orderBy('cities.name')
                ->get();

            return compact('vendors', 'locations');
        });
    }

    // END: FIX

    public function create(): array
    {
        $countries = $this->getCountries();
        $states = $this->getAllStates();
        $cities = $this->getAllCities();
        return compact('countries', 'states', 'cities');
    }

    public function store(Request $request): Shops
    {
        return DB::transaction(function () use ($request) {
            $data = $this->prepareRequestData($request);
            $data['created_by'] = Auth::id();
            $data['status'] = 1;
            $data['language_id'] = getLanguageId(app()->getLocale());
            $data['slug'] = $this->generateUniqueSlug($request->shop_name);

            if ($request->hasFile('shop_logo')) {
                $data['shop_logo'] = $this->uploadFile($request->file('shop_logo'), 'shop_logo');
            }

            if ($request->hasFile('gallery')) {
                $data['gallery'] = $this->uploadGallery($request->file('gallery'));
            }

            $shop = Shops::create($data);
            $this->clearShopFilterCache();

            return $shop;
        });
    }

    public function edit(string $slug): array
    {
        $shop = Shops::where('slug', $slug)
            ->where('created_by', Auth::id())
            ->firstOrFail();

        $countries = $this->getCountries();
        $states = $shop->country ? $this->getStatesByCountry($shop->country) : collect();
        $cities = $shop->state ? $this->getCitiesByState($shop->state) : collect();
        $shopWorkingDays = json_decode($shop->working_day, true) ?? [];

        return compact('shop', 'countries', 'states', 'cities', 'shopWorkingDays');
    }

    public function update(Request $request, Shops $shop): Shops
    {
        return DB::transaction(function () use ($request, $shop) {
            $data = $this->prepareRequestData($request);

            if ($request->shop_name !== $shop->shop_name) {
                $data['slug'] = $this->generateUniqueSlug($request->shop_name, $shop->id);
            }

            if ($request->hasFile('shop_logo')) {
                $this->deleteFile($shop->shop_logo);
                $data['shop_logo'] = $this->uploadFile($request->file('shop_logo'), 'shop_logo');
            }

            $currentGallery = $shop->gallery ?? [];
            if ($request->has('removed_gallery_images')) {
                $removedImages = $request->input('removed_gallery_images', []);
                foreach ($removedImages as $imagePath) {
                    $this->deleteFile($imagePath);
                }
                $currentGallery = array_diff($currentGallery, $removedImages);
            }
            if ($request->hasFile('gallery')) {
                $newImages = $this->uploadGallery($request->file('gallery'));
                $currentGallery = array_merge($currentGallery, $newImages);
            }
            $data['gallery'] = array_values($currentGallery);

            $shop->update($data);
            $this->clearShopFilterCache();

            return $shop;
        });
    }

    public function destroy(Shops $shop): bool
    {
        return DB::transaction(function () use ($shop) {
            $this->deleteFile($shop->shop_logo);
            if (is_array($shop->gallery)) {
                foreach ($shop->gallery as $image) {
                    $this->deleteFile($image);
                }
            }
            $deleted = $shop->delete();
            if ($deleted) {
                $this->clearShopFilterCache();
            }

            return $deleted;
        });
    }

    public function updateStatus(Request $request, Shops $shop): bool
    {
        $updated = $shop->update(['status' => $request->status]);
        if ($updated) {
            $this->clearShopFilterCache();
        }

        return $updated;
    }

    public function getStates(Request $request): array
    {
        $states = $this->getStatesByCountry($request->country_id);
        return ['states' => $states];
    }

    public function getCities(Request $request): array
    {
        $cities = $this->getCitiesByState($request->state_id);
        return ['cities' => $cities];
    }

    private function prepareRequestData(Request $request): array
    {
        $data = $request->except(['_token', '_method', 'shop_logo', 'gallery', 'removed_gallery_images', 'availability', 'category_id', 'sub_category_id']);

        $data['description'] = $request->input('description');

        $data['tax_status'] = $request->input('tax_status', 0);
        if ($data['tax_status'] == 0) {
            $data['tax_number'] = null;
        }

        $workingDays = [];
        if ($request->has('availability')) {
            foreach ($request->input('availability') as $day => $details) {
                if (isset($details['status']) && $details['status'] === 'on') {
                    $workingDays[$day] = [
                        'start_time' => $details['start_time'],
                        'end_time' => $details['end_time'],
                    ];
                }
            }
        }
        $data['working_day'] = json_encode($workingDays);

        return $data;
    }

    private function generateUniqueSlug(string $name, int $ignoreId = null): string
    {
        $baseSlug = Str::slug($name);

        $existingSlugsQuery = Shops::where('slug', 'like', $baseSlug . '%');
        if ($ignoreId) {
            $existingSlugsQuery->where('id', '!=', $ignoreId);
        }

        $existingSlugs = $existingSlugsQuery->pluck('slug')->toArray();

        if (!in_array($baseSlug, $existingSlugs, true)) {
            return $baseSlug;
        }

        $suffix = 1;
        do {
            $newSlug = $baseSlug . '-' . $suffix;
            $suffix++;
        } while (in_array($newSlug, $existingSlugs, true));

        return $newSlug;
    }

    private function uploadFile($file, string $path): string
    {
        $fileName = time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
        $file->move(storage_path('app/public/' . $path), $fileName);
        return "{$path}/{$fileName}";
    }

    private function uploadGallery(array $files): array
    {
        $paths = [];
        foreach ($files as $file) {
            $paths[] = $this->uploadFile($file, 'shop_gallery');
        }
        return $paths;
    }

    private function deleteFile(?string $path): void
    {
        if ($path && File::exists(storage_path('app/public/' . $path))) {
            File::delete(storage_path('app/public/' . $path));
        }
    }

    private function getCountries()
    {
        return Cache::remember('shops:countries', self::CACHE_TTL, function () {
            return DB::table('countries')->select('id', 'name')->orderBy('name')->get();
        });
    }

    private function getAllStates()
    {
        return Cache::remember('shops:states:all', self::CACHE_TTL, function () {
            return DB::table('states')->select('id', 'name', 'country_id')->orderBy('name')->get();
        });
    }

    private function getAllCities()
    {
        return Cache::remember('shops:cities:all', self::CACHE_TTL, function () {
            return DB::table('cities')->select('id', 'name', 'state_id')->orderBy('name')->get();
        });
    }

    private function getStatesByCountry(?int $countryId)
    {
        if (!$countryId) {
            return collect();
        }

        return Cache::remember("shops:states:country:{$countryId}", self::CACHE_TTL, function () use ($countryId) {
            return DB::table('states')->where('country_id', $countryId)->orderBy('name')->get();
        });
    }

    private function getCitiesByState(?int $stateId)
    {
        if (!$stateId) {
            return collect();
        }

        return Cache::remember("shops:cities:state:{$stateId}", self::CACHE_TTL, function () use ($stateId) {
            return DB::table('cities')->where('state_id', $stateId)->orderBy('name')->get();
        });
    }

    private function clearShopFilterCache(): void
    {
        Cache::forget('shops:filter:data');
        incrementPageCacheVersion();
    }
}
