<?php

namespace Modules\Shops\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Modules\Shops\app\Http\Requests\ShopRequest;
use Modules\Shops\app\Repositories\Contracts\ShopRepositoryInterface;
use Modules\Shops\app\Models\Shops;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\JsonResponse;

class ShopsController extends Controller
{
    protected ShopRepositoryInterface $shopRepository;

    public function __construct(ShopRepositoryInterface $shopRepository)
    {
        $this->shopRepository = $shopRepository;
    }

    public function index(Request $request): View
    {
        $data = $this->shopRepository->index($request);
        return view('provider.shops.index', $data);
    }

    public function create(): View
    {
        $data = $this->shopRepository->create();
        return view('provider.shops.create', $data);
    }

    public function store(Request $request): RedirectResponse
    {
        $this->shopRepository->store($request);
        return redirect()->route('provider.shops.index')->with('success', 'Shop created successfully.');
    }

    public function edit(Shops $shop): View
    {
        if ($shop->created_by != Auth::id()) {
            abort(403, 'Unauthorized action.');
        }
        $data = $this->shopRepository->edit($shop->slug);
        return view('provider.shops.edit', $data);
    }

    public function update(ShopRequest $request, Shops $shop): RedirectResponse
    {
        if ($shop->created_by !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }
        $this->shopRepository->update($request, $shop);
        return redirect()->route('provider.shops.index')->with('success', 'Shop updated successfully.');
    }

    public function destroy(Request $request, Shops $shop): JsonResponse|RedirectResponse
    {
        if ($shop->created_by !== Auth::id()) {
            if ($request->wantsJson()) {
                return response()->json(['success' => false, 'message' => 'Unauthorized action.'], 403);
            }
            abort(403, 'Unauthorized action.');
        }

        $this->shopRepository->destroy($shop);
        if ($request->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Shop deleted successfully.']);
        }
        return redirect()->route('provider.shops.index')->with('success', 'Shop deleted successfully.');
    }

    public function updateStatus(Request $request, Shops $shop): JsonResponse|RedirectResponse
    {
        if ($shop->created_by !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }
        $this->shopRepository->updateStatus($request, $shop);
        $message = $request->status == 1 ? 'Shop activated successfully.' : 'Shop deactivated successfully.';

        if ($request->wantsJson()) {
            return response()->json(['success' => true, 'message' => $message]);
        }
        return redirect()->route('provider.shops.index')->with('success', $message);
    }

    public function getStates(Request $request): JsonResponse
    {
        $data = $this->shopRepository->getStates($request);
        return response()->json($data);
    }

    public function getCities(Request $request): JsonResponse
    {
        $data = $this->shopRepository->getCities($request);
        return response()->json($data);
    }


    public function checkName(Request $request): JsonResponse
    {
        $shopName = $request->input('shop_name');
        $shopId = $request->input('id'); // For checking on update

        $query = DB::table('shops')->where('shop_name', $shopName);

        if ($shopId) {
            $query->where('id', '!=', $shopId);
        }

        $exists = $query->exists();

        return response()->json(!$exists);
    }
}
