<?php

namespace Modules\Service\app\Repositories\Eloquent;

use Modules\Service\app\Repositories\Contracts\ServiceRepositoryInterface;
use App\Http\Controllers\Controller;
use App\Models\BookingProduct;
use App\Models\Branches;
use App\Models\PackageTrx;
use App\Models\ProductVariation;
use App\Models\ProviderSocialLink;
use App\Models\ServiceBranch;
use App\Models\ServiceStaff;
use App\Models\User;
use App\Models\Wishlist;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Service\app\Models\Productmeta;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Service\app\Models\AdditionalService as ModelsAdditionalService;
use Modules\Service\app\Models\Service;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Modules\Product\app\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;
use Modules\Categories\app\Models\Categories;
use Modules\Communication\app\Http\Controllers\EmailController;
use Modules\GlobalSetting\app\Models\Language;
use Modules\Communication\app\Http\Controllers\NotificationController;
use Modules\GlobalSetting\app\Models\Templates;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Product\app\Models\Category;
use Modules\Product\app\Models\Rating;
use Modules\Shops\app\Models\Shops;
use App\Models\Bookings;
use App\Models\UserDetail;

class ServiceRepository implements ServiceRepositoryInterface
{
    public function setDefault(Request $request): array
    {
        $category = Service::where('id', $request->input('id'))->first();
        if ($category->status == 0) {
            $product_views = Service::find($request->input('id'));
            $product_views->status = 1;
            $product_views->save();
        }
        if ($category->status == 1) {
            $product_views = Service::find($request->input('id'));
            $product_views->status = 0;
            $product_views->save();
        }

        incrementPageCacheVersion();

        return [
            'code' => '200',
            'success' => true,
            'message' => ucfirst($request->input('type')) . ' status updated Sucesfully'
        ];
    }

    public function delete(Request $request): array
    {
        try {
            $category = Service::where('id', $request->input('id'))->first();
            $category->delete();

            incrementPageCacheVersion();

            return [
                'code' => 200,
                'success' => true,
                'message' => 'Category deleted successfully.',
                'data' => $category
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'success' => false,
                'message' => 'Failed to deleted Category status.',
                'error' => $e->getMessage()
            ];
        }
    }

    public function index(Request $request): array
    {
        try {
            $sortBy      = $request->input('sort_by');
            $categoryId  = $request->input('category');
            $fromPrice   = $request->input('from_price');
            $toPrice     = $request->input('to_price');
            $perPage     = (int) $request->input('per_page', 6);
            $shopIds     = $request->input('shops');
            $vendorIds   = $request->input('vendor');
            $highlighted = $request->input('highlighted');
            $reviews     = $request->input('reviews');
            $keywords    = $request->input('keywords');
            $languageId = $request->input('language_id') ?? getLanguageId(app()->getLocale());

            $query = Service::with(['category', 'user', 'meta'])
                ->where('source_type', 'service')
                ->whereHas('shop')
                ->where('language_id', $languageId);

            // Filter by Keywords
            if (!empty($keywords)) {
                $query->where(function ($q) use ($keywords) {
                    $q->where('source_name', 'like', "%{$keywords}%");
                });
            }

            /* ----------------------------------------
         * Filter by Category
         * ---------------------------------------- */
            if (!empty($categoryId)) {
                if (is_array($categoryId)) {
                    $query->whereIn('source_category', $categoryId);
                } else {
                    $query->where('source_category', $categoryId);
                }
            }

            /* ----------------------------------------
         * Filter by Shop
         * ---------------------------------------- */
            if (!empty($shopIds)) {
                if (is_array($shopIds)) {
                    $query->whereIn('shop_id', $shopIds);
                } else {
                    $query->where('shop_id', $shopIds);
                }
            }

            /* ----------------------------------------
         * Filter by Vendor
         * ---------------------------------------- */
            if (!empty($vendorIds)) {
                if (is_array($vendorIds)) {
                    $query->whereIn('user_id', $vendorIds);
                } else {
                    $query->where('user_id', $vendorIds);
                }
            }

            /* ----------------------------------------
         * Filter by Price Range
         * ---------------------------------------- */
            if (!empty($fromPrice) && !empty($toPrice)) {
                $query->whereBetween('source_price', [$fromPrice, $toPrice]);
            } elseif (!empty($fromPrice)) {
                $query->where('source_price', '>=', $fromPrice);
            } elseif (!empty($toPrice)) {
                $query->where('source_price', '<=', $toPrice);
            }

          /* ----------------------------------------
            * Highlighted Filter (Featured or Best Seller)
            * ---------------------------------------- */
            if (!empty($highlighted)) {
                $highlightedArray = is_array($highlighted) ? $highlighted : [$highlighted];

                // Get best seller IDs dynamically
                $bestSellerIds = Bookings::select('product_id')
                    ->whereHas('product', function ($q) use ($languageId) {
                        $q->where('status', 1)
                            ->where('language_id', $languageId)
                            ->whereHas('shop');
                    })
                    ->where('booking_status', 5)
                    ->groupBy('product_id')
                    ->orderByRaw('SUM(service_qty) DESC')
                    ->limit(5)
                    ->pluck('product_id')
                    ->toArray();

                $query->where(function ($q) use ($highlightedArray, $bestSellerIds) {
                    // Featured products
                    if (in_array('featured', $highlightedArray)) {
                        $q->orWhere('featured', 1);
                    }

                    // Best seller products
                    if (in_array('best_seller', $highlightedArray)) {
                        $q->orWhereIn('id', $bestSellerIds);
                    }
                });
            }

            /* ----------------------------------------
         * Filter by Rating
         * ---------------------------------------- */
            if (!empty($reviews)) {
                $reviewArray = is_array($reviews) ? $reviews : [$reviews];

                $query->whereHas('ratings', function ($q) use ($reviewArray) {
                    // Match rounded star values like 4 or 5 stars
                    $q->whereIn(DB::raw('ROUND(rating)'), $reviewArray);
                });
            }

            /* ----------------------------------------
         * Sorting
         * ---------------------------------------- */
            if ($sortBy === 'old') {
                $query->orderBy('id', 'asc');
            } elseif ($sortBy === 'new') {
                $query->orderBy('id', 'desc');
            } elseif ($sortBy === 'price_low') {
                $query->orderBy('source_price', 'asc');
            } elseif ($sortBy === 'price_high') {
                $query->orderBy('source_price', 'desc');
            } else {
                $query->orderBy('id', 'desc'); // Default: newest first
            }

            /* ----------------------------------------
         * Pagination
         * ---------------------------------------- */
            $products = $query->paginate($perPage);

            /* ----------------------------------------
         * Transform Response
         * ---------------------------------------- */
            $data = $products->getCollection()->map(function ($product) use ($request): array {
                $image = $product->meta
                    ->where('source_key', 'product_image')
                    ->pluck('source_Values')
                    ->first();

                $imageUrl = uploadedAsset(Str::replaceFirst('product_images/', 'product_images/fixed/', $image ?? ''), 'default2');

                // --- Ratings & Reviews ---
                $rating = Rating::where('product_id', $product->id)->where('parent_id', 0)->avg('rating') ?? 0;
                $reviewCount = Rating::where('product_id', $product->id)->where('parent_id', 0)->count();

                // --- Wishlist Check ---
                $wishlistExists = Wishlist::where('user_id', $request->user_id)
                    ->where('service_id', $product->id)
                    ->exists();

                // --- Shop Details ---
                $shop = Shops::select('shop_name', 'shop_logo')
                    ->where('id', $product->shop_id)
                    ->first();

                $shopName = $shop->shop_name ?? 'Default Shop';
                $shopImage = uploadedAsset($shop->shop_logo, 'default_shop_logo');

                return [
                    'id'           => $product->id,
                    'provider_id'  => $product->user_id,
                    'title'        => $product->source_name,
                    'slug'         => $product->slug,
                    'code'         => $product->source_code,
                    'image'        => $imageUrl,
                    'category'     => $product->category->name ?? null,
                    'starts_from'  => $product->source_price,
                    'rating'       => number_format($rating, 1, '.', ''),
                    'review_count' => $reviewCount,
                    'status'       => $product->status,
                    'wishlist'     => $wishlistExists,
                    'is_featured'  => (bool) $product->is_featured,
                    'verified'     => (bool) $product->verified_status,
                    'currency'     => getDefaultCurrencySymbol(),
                    'location'     => $product->location ?? 'New York, USA',
                    'shop_name'    => $shopName,
                    'shop_image'   => $shopImage,
                ];
            });

            /* ----------------------------------------
         * Final JSON Response
         * ---------------------------------------- */
            return [
                'code'    => 200,
                'message' => 'Services retrieved successfully.',
                'data'    => $data,
                'pagination' => [
                    'current_page' => $products->currentPage(),
                    'per_page'     => $products->perPage(),
                    'total'        => $products->total(),
                    'last_page'    => $products->lastPage(),
                ],
            ];
        } catch (\Exception $e) {
            return [
                'code'    => 500,
                'message' => 'An error occurred while retrieving services.',
                'error'   => $e->getMessage(),
            ];
        }
    }

    public function productListApi(Request $request): array
    {
        try {
            $sortBy       = $request->input('sort_by', 'new');
            $categories   = (array) $request->input('category', []);
            $fromPrice    = $request->input('from_price');
            $toPrice      = $request->input('to_price');
            $perPage      = (int) $request->input('per_page', 6);
            $highlighted  = (array) $request->input('highlighted', []);
            $shops        = (array) $request->input('shops', []);
            $reviews      = (array) $request->input('reviews', []);
            $color        = $request->input('color');
            $languageId = $request->input('language_id') ?? getLanguageId(app()->getLocale());

            $bestSellerIds = BookingProduct::select('product_id')
                ->whereHas('product', function ($q) use ($languageId) {
                    $q->where('status', 1)
                        ->where('language_id', $languageId)
                        ->whereHas('shop');
                })
                ->where('product_status', 4)
                ->groupBy('product_id')
                ->orderByRaw('SUM(quantity) DESC')
                ->limit(5)
                ->pluck('product_id')
                ->toArray();

            /**
             * 1. Base query
             */
            $query = Service::with(['category', 'user', 'meta', 'user.userDetail:user_id,profile_image'])
                ->whereHas('shop')
                ->where('status', 1)
                ->where('source_type', 'product')
                ->where('language_id', $languageId);

            /**
             * 2. Category filter
             */
            if (!empty($categories)) {
                $query->whereIn('source_category', $categories);
            }

            /**
             * 3. Price filter
             */
            if ($fromPrice !== null || $toPrice !== null) {
                $query->where(function ($q) use ($fromPrice, $toPrice) {
                    // Case 1: Products with variations → filter by first variation price
                    $q->whereHas('variations', function ($q2) use ($fromPrice, $toPrice) {
                        $q2->whereRaw("JSON_LENGTH(data) > 0");

                        if ($fromPrice !== null && $toPrice !== null) {
                            $q2->whereRaw("CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$[0].price')) AS DECIMAL) BETWEEN ? AND ?", [$fromPrice, $toPrice]);
                        } elseif ($fromPrice !== null) {
                            $q2->whereRaw("CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$[0].price')) AS DECIMAL) >= ?", [$fromPrice]);
                        } elseif ($toPrice !== null) {
                            $q2->whereRaw("CAST(JSON_UNQUOTE(JSON_EXTRACT(data, '$[0].price')) AS DECIMAL) <= ?", [$toPrice]);
                        }
                    })

                    // Case 2: Fallback → check source_price directly
                    ->orWhere(function ($q3) use ($fromPrice, $toPrice) {
                        $q3->whereDoesntHave('variations', function ($q4) {
                            $q4->whereRaw("JSON_LENGTH(data) > 0");
                        });

                        if ($fromPrice !== null && $toPrice !== null) {
                            $q3->whereBetween('source_price', [$fromPrice, $toPrice]);
                        } elseif ($fromPrice !== null) {
                            $q3->where('source_price', '>=', $fromPrice);
                        } elseif ($toPrice !== null) {
                            $q3->where('source_price', '<=', $toPrice);
                        }
                    });
                });
            }

            /**
             * 4. Highlighted filter (best_seller / featured)
             */
            if (!empty($highlighted)) {
                $query->where(function ($q) use ($highlighted, $bestSellerIds) {

                    // best seller filter
                    if (in_array('best_seller', $highlighted)) {
                        $q->orWhereIn('id', $bestSellerIds);
                    }

                    // featured filter
                    if (in_array('featured', $highlighted)) {
                        $q->orWhere('featured', true);
                    }
                });
            }

            /**
             * 5. Shop filter
             */
            if (!empty($shops)) {
                $query->whereIn('shop_id', $shops);
            }

            /**
             * 6. Exact Reviews Filter (e.g., only 4-star or 5-star products)
             */
            if (!empty($reviews)) {
                $query->whereHas('ratings', function ($q) use ($reviews) {
                    $q->selectRaw('product_id, AVG(rating) as avg_rating')
                    ->groupBy('product_id')
                    ->havingRaw('FLOOR(AVG(rating)) IN (' . implode(',', $reviews) . ')');
                });
            }

            /**
             * 8. Color filter (via product_variations table)
             */
            if (!empty($color)) {
                $query->whereHas('variations', function ($q) use ($color) {
                    $q->where('color', $color);
                });
            }

            /**
             * 9. Sorting
             */
            $query->orderBy('id', $sortBy == 'old' ? 'asc' : 'desc');

            /**
             * 10. Paginate results
             */
            $products = $query->paginate($perPage);

            /**
             * 11. Transform data for response
             */
            $data = $products->getCollection()->map(function ($product) use ($request, $bestSellerIds) {
                // --- Product Image ---
                $image = $product->meta
                    ->where('source_key', 'product_image')
                    ->pluck('source_Values')
                    ->first();

                $imageUrl = uploadedAsset($image ?? '', 'default2');

                // --- Wishlist Check ---
                $wishlistExists = Wishlist::where('user_id', $request->user_id)
                    ->where('service_id', $product->id)
                    ->exists();

                // --- Ratings & Reviews ---
                $rating = Rating::where('product_id', $product->id)->where('parent_id', 0)->avg('rating') ?? 0;
                $reviewCount = Rating::where('product_id', $product->id)->where('parent_id', 0)->count();

                // --- Shop Details ---
                $shop = Shops::select('shop_name', 'shop_logo')
                    ->where('id', $product->shop_id)
                    ->first();

                // --- Product Price ---
                $price = $product->source_price;
                $variationPrice = getProductVariationPrice($product->id);

                return [
                    'id'          => $product->id,
                    'provider_id' => $product->user_id,
                    'title'       => ucfirst($product->source_name ?? ''),
                    'slug'        => $product->slug,
                    'code'        => $product->source_code,
                    'short_description' => ucfirst(Str::limit($product->shot_description ?? '', 120, '...')),
                    'image'       => $imageUrl,
                    'category'    => ucfirst($product->category->name ?? ''),
                    'starts_from' => $variationPrice ?? $price,
                    'status'      => $product->status,
                    'wishlist'    => $wishlistExists,
                    'rating'       => number_format($rating, 1, '.', ''),
                    'review_count' => $reviewCount,
                    'is_best_seller' => in_array($product->id, $bestSellerIds),
                    'is_featured' => (bool) $product->featured,
                    'verified'    => (bool) $product->verified_status,
                    'currency'    => getDefaultCurrencySymbol(),
                    'location'    => $product->location ?? 'New York, USA',
                    'shop_name'   => ucfirst($shop->shop_name ?? 'Default Shop'),
                    'shop_image'  => uploadedAsset($shop->shop_logo ?? '', 'default_shop_logo'),
                    'profile_image' => uploadedAsset('profile/' . $product->user->userDetail->profile_image ?? '', 'profile'),
                ];
            });

            /**
             * 12. Return response
             */
            return [
                'code'       => 200,
                'message'    => 'Products retrieved successfully.',
                'data'       => $data,
                'pagination' => [
                    'current_page' => $products->currentPage(),
                    'per_page'     => $products->perPage(),
                    'total'        => $products->total(),
                    'last_page'    => $products->lastPage(),
                ],
            ];
        } catch (\Exception $e) {
            return [
                'code'    => 500,
                'message' => 'An error occurred while retrieving products.',
                'error'   => $e->getMessage(),
            ];
        }
    }

    public function shopListApi(Request $request): array
    {
        try {
            $sortBy = $request->input('sort_by', 'new');
            $search = $request->input('search');
            $perPage = (int) $request->input('per_page', 10);
            $languageId = $request->input('language_id') ?? getLanguageId(app()->getLocale());

            $query = Shops::select('id', 'slug', 'shop_logo', 'shop_name', 'owner_name', 'email', 'status', 'created_at')
                ->where('language_id', $languageId);

            if (!empty($search)) {
                $query->where('shop_name', 'like', "%{$search}%")
                      ->orWhere('owner_name', 'like', "%{$search}%");
            }

            if ($sortBy == 'old') {
                $query->orderBy('created_at', 'asc');
            } else {
                $query->orderBy('created_at', 'desc');
            }

            $shops = $query->paginate($perPage);

            $shops->getCollection()->transform(function ($shop) {
                $shop->shop_logo_url = uploadedAsset($shop->shop_logo, 'default_shop_logo');
                return $shop;
            });

            return [
                'code' => 200,
                'message' => 'Shops retrieved successfully.',
                'data' => $shops->items(),
                'pagination' => [
                    'current_page' => $shops->currentPage(),
                    'per_page' => $shops->perPage(),
                    'total' => $shops->total(),
                    'last_page' => $shops->lastPage(),
                ],
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => 'An error occurred while retrieving shops.',
                'error' => $e->getMessage(),
            ];
        }
    }

    public function serviceDetails(Request $request, string $slug): array
    {
        try {
            $userId = Auth::id() ?? $request->user_id;
            // Main product
            $product = DB::table('products')
                ->select(
                    'products.id',
                    'products.slug',
                    'products.status',
                    'products.verified_status',
                    'products.user_id',
                    'categories.name as category_name',
                    'products.source_code',
                    'products.source_price',
                    'products.source_description',
                    'products.shot_description',
                    'products.source_name',
                    'products.include',
                )
                ->join('categories', 'products.source_category', '=', 'categories.id')
                ->where('products.slug', $slug)
                ->where('products.source_type', 'service')
                ->whereNull('products.deleted_at')
                ->firstOrFail();

            $gallery = DB::table('products_meta')
                ->where('product_id', $product->id)
                ->where('source_key', 'product_image')
                ->pluck('source_values')
                ->map(function ($img) {
                    $fixedImage = Str::replaceFirst('product_images/', 'product_images/fixed/', $img ?? '');
                    return uploadedAsset($fixedImage, 'default2');
                })
                ->toArray();

            // Fallback if no images are found
            if (empty($gallery)) {
                $gallery = [asset('img/default-image.png')];
            }

            // Additional services
            $addServices = DB::table('additional_services')
                ->where('service_id', $product->id)
                ->select(
                    'name',
                    'price',
                    'duration',
                    DB::raw("CONCAT('" . url('storage') . "/', image) as image")
                )
                ->get();

            // Provider info
            $provider = DB::table('users')
                ->join('user_details', 'users.id', '=', 'user_details.user_id')
                ->select(
                    'users.id',
                    'users.name',
                    'users.email',
                    'user_details.first_name',
                    'user_details.last_name',
                    'user_details.mobile_number',
                    'user_details.profile_image',
                    'user_details.bio',
                    'user_details.address',
                    'user_details.country',
                    'user_details.state',
                    'user_details.city',
                    'user_details.postal_code',
                    'user_details.company_name',
                    'user_details.company_image',
                    'user_details.company_website',
                    'user_details.created_at'
                )
                ->where('users.id', $product->user_id)
                ->first();

            $providerData = [
                'name'          => $provider->name ?? '',
                'email'         => $provider->email ?? '',
                'first_name'    => $provider->first_name ?? '',
                'last_name'     => $provider->last_name ?? '',
                'mobile_number' => $provider->mobile_number ?? '',
                'address'       => $provider->address ?? '',
                'member_since'  => $provider->created_at
                    ? Carbon::parse($provider->created_at)->format('d, M Y')
                    : asset('/assets/img/default-image.png'),
            ];

            // Location & map
            $city_name = 'California, USA';
            if ($provider && !empty($provider->city)) {
                $city = DB::table('cities')->where('id', $provider->city)->first();
                if ($city) {
                    $city_name = $city->name;
                }
            }

            $googleMapUrl = 'https://www.google.com/maps/embed/v1/place?key='
                . config('services.google_maps.key')
                . '&q=' . urlencode($city_name);

            $locationSetting = DB::table('general_settings')->where('key', 'location_status')->first();
            $locationEnabled = $locationSetting && $locationSetting->value == 1;
            $mapHasError = !$locationEnabled || $this->isInvalidGoogleMapKey($googleMapUrl);

            // Social links
            $socialLinks = ProviderSocialLink::with('socialLink')
                ->where('provider_id', $product->user_id)
                ->where('status', 1)
                ->get()
                ->map(function ($item) {
                    return [
                        'platform' => $item->platform_name ?? $item->socialLink?->platform_name,
                        'url'      => $item->link,
                        'icon'     => $item->socialLink?->icon,
                    ];
                });

            // ✅ Related services
            $relatedServices = Service::with(['category', 'user', 'meta'])
                ->where('source_type', 'service')
                ->where('id', '!=', $product->id) // exclude current service
                ->limit(6) // show some, not all
                ->get()
                ->map(function ($related) {
                    $image = $related->meta
                        ->where('source_key', 'product_image')
                        ->pluck('source_Values')
                        ->first();

                    $imageUrl = uploadedAsset(Str::replaceFirst('product_images/', 'product_images/fixed/', $image), 'default2');

                    // --- Shop Details ---
                    $shop = Shops::select('shop_name', 'shop_logo')
                        ->where('id', $related->shop_id)
                        ->first();

                    return [
                        'id'           => $related->id,
                        'provider_id'  => $related->user_id,
                        'title'        => $related->source_name,
                        'slug'         => $related->slug,
                        'code'         => $related->source_code,
                        'image'        => $imageUrl,
                        'category'     => $related->category->name ?? null,
                        'starts_from'  => $related->source_price,
                        'rating'       => rand(1, 5),
                        'review_count' => rand(1, 100),
                        'status'       => $related->status,
                        'wishlist'     => (bool) rand(0, 1),
                        'is_featured'  => (bool) rand(0, 1),
                        'verified'     => (bool) $related->verified_status,
                        'currency'     => getDefaultCurrencySymbol(),
                        'location'     => "Newyork, USA",
                        'shop_name'   => $shop->shop_name ?? 'Default Shop',
                        'shop_image'  => $shop && $shop->shop_logo
                            ? asset('storage/' . $shop->shop_logo)
                            : asset('assets/img/shop-default.png'),
                    ];
                });

            // --- Wishlist Check ---
            $wishlistExists = Wishlist::where('user_id', $userId)
                ->where('service_id', $product->id)
                ->exists();

            // --- Ratings & Reviews ---
            $rating = Rating::where('product_id', $product->id)->where('parent_id', 0)->avg('rating') ?? 0;
            $reviewCount = Rating::where('product_id', $product->id)->where('parent_id', 0)->count();

            $bookingCount = Bookings::where('product_id', $product->id)
                ->whereNull('deleted_at')
                ->count();

            // ✅ Final structured response
            $data = [
                'id'            => $product->id,
                'slug'          => $product->slug,
                'title'         => $product->source_name,
                'code'          => $product->source_code,
                'category'      => $product->category_name,
                'starts_from'   => $product->source_price,
                'currency'      => getDefaultCurrencySymbol(),
                'status'        => $product->status,
                'rating'       => number_format($rating, 1, '.', ''),
                'wishlist'     => $wishlistExists,
                'review_count' => $reviewCount,
                'is_featured'   => (bool) rand(0, 1),
                'no_of_bookings' => $bookingCount,
                'location'      => "Newyork, USA",
                'map_url'       => $googleMapUrl,
                'include'       => $product->include,
                'overview'      => [
                    'description' => $product->source_description,
                ],
                'gallery'       => $gallery,
                'add_services'  => $addServices,
                'provider'      => $providerData,
                'social_profile' => $socialLinks,
                'related_service' => $relatedServices,
            ];

            return [
                'code'    => 200,
                'message' => 'Service details retrieved successfully.',
                'data'    => $data,
            ];
        } catch (\Exception $e) {
            return [
                'code'    => 500,
                'message' => 'An error occurred while retrieving service details.',
                'error'   => $e->getMessage(),
            ];
        }
    }

    public function productDetails(Request $request, string $slug): array
    {
        try {
            $userId = Auth::id() ?? $request->user_id;

            // Main product
            $product = Product::query()
                ->select(
                    'products.id',
                    'products.slug',
                    'products.status',
                    'products.verified_status',
                    'products.user_id',
                    'categories.name as category_name',
                    'products.source_code',
                    'products.source_price',
                    'products.source_description',
                    'products.source_name',
                    'products.shot_description',
                    'products.source_stock',
                    'products.domestic_shipping',
                    'products.international_shipping',
                    'products.include',
                )
                ->join('categories', 'products.source_category', '=', 'categories.id')
                ->where('products.slug', $slug)
                ->where('products.source_type', 'product')
                ->whereNull('products.deleted_at')
                ->first();

            if (!$product) {
                return [
                    'code'    => 404,
                    'message' => 'Product not found',
                    'data'    => null
                ];
            }

            // ✅ START: NEW CODE BLOCK FOR PRODUCT VARIATIONS
            $productVariations = DB::table('product_variations')
                ->where('product_id', $product->id)
                ->get()
                ->map(function ($variation) {
                    // Decode the images JSON and create full URLs
                    $imagePaths = json_decode($variation->images, true) ?? [];
                    $images = array_map(function ($path) {
                        return uploadedAsset($path ?? '', 'default');
                    }, $imagePaths);

                    // Decode the data JSON for sizes, prices, and stock
                    $attributes = json_decode($variation->data, true) ?? [];

                    return [
                        'id'         => $variation->id,
                        'color'      => $variation->color,
                        'images'     => $images,
                        'attributes' => $attributes,
                    ];
                });
            // ✅ END: NEW CODE BLOCK FOR PRODUCT VARIATIONS

            $gallery = DB::table('products_meta')
                ->where('product_id', $product->id)
                ->where('source_key', 'product_image')
                ->whereNull('deleted_at')
                ->pluck('source_values')
                ->map(function ($img) {
                    return uploadedAsset($img ?? '', 'default2');
                })
                ->toArray();

            // If gallery is completely empty, return one default image
            if (empty($gallery)) {
                $gallery = [uploadedAsset('', 'default2')];
            }

            // Provider info
            $provider = DB::table('users')
                ->join('user_details', 'users.id', '=', 'user_details.user_id')
                ->select(
                    'users.id',
                    'users.name',
                    'users.email',
                    'user_details.first_name',
                    'user_details.last_name',
                    'user_details.mobile_number',
                    'user_details.profile_image',
                    'user_details.bio',
                    'user_details.address',
                    'user_details.country',
                    'user_details.state',
                    'user_details.city',
                    'user_details.postal_code',
                    'user_details.company_name',
                    'user_details.company_image',
                    'user_details.company_website',
                    'user_details.created_at'
                )
                ->where('users.id', $product->user_id)
                ->first();

            $providerData = [
                'name'          => $provider->name ?? '',
                'email'         => $provider->email ?? '',
                'first_name'    => $provider->first_name ?? '',
                'last_name'     => $provider->last_name ?? '',
                'mobile_number' => $provider->mobile_number ?? '',
                'address'       => $provider->address ?? '',
                'member_since'  => $provider->created_at
                    ? Carbon::parse($provider->created_at)->format('d, M Y')
                    : null,
            ];

            // Location & map
            $city_name = 'California, USA';
            if ($provider && !empty($provider->city)) {
                $city = DB::table('cities')->where('id', $provider->city)->first();
                if ($city) {
                    $city_name = $city->name;
                }
            }

            $googleMapUrl = 'https://maps.google.com/maps/api/staticmap?center='
                . urlencode($city_name)
                . '&zoom=14&size=600x400&key='
                . config('services.google_maps.key');


            $locationSetting = DB::table('general_settings')->where('key', 'location_status')->first();
            $locationEnabled = $locationSetting && $locationSetting->value == 1;
            $mapHasError = !$locationEnabled; // Simplified check

            // Social links
            $socialLinks = ProviderSocialLink::with('socialLink')
                ->where('provider_id', $product->user_id)
                ->where('status', 1)
                ->get()
                ->map(function ($item) {
                    return [
                        'platform' => $item->platform_name ?? $item->socialLink?->platform_name,
                        'url'      => $item->link,
                        'icon'     => $item->socialLink?->icon,
                    ];
                });

            // Related services
            $relatedProducts = Service::with(['category', 'user', 'meta', 'user.userDetail:user_id,profile_image'])
                ->where('source_type', 'product')
                ->where('id', '!=', $product->id) // exclude current service
                ->where('status', 1)
                ->whereHas('shop')
                ->limit(6) // show some, not all
                ->get()
                ->map(function ($related) use ($userId) {
                    $image = $related->meta
                        ->where('source_key', 'product_image')
                        ->pluck('source_Values')
                        ->first();

                    $imageUrl = uploadedAsset($image ?? '', 'default2');

                    // --- Shop Details ---
                    $shop = Shops::select('shop_name', 'shop_logo')
                        ->where('id', $related->shop_id)
                        ->first();

                    // --- Wishlist Check ---
                    $wishlistExists = Wishlist::where('user_id', $userId)
                        ->where('service_id', $related->id)
                        ->exists();

                    // --- Ratings & Reviews ---
                    $rating = Rating::where('product_id', $related->id)->where('parent_id', 0)->avg('rating') ?? 0;
                    $reviewCount = Rating::where('product_id', $related->id)->where('parent_id', 0)->count();

                    $price = $related->source_price;
                    $variationProduct = getProductVariation($related->id);

                    $variationPrice = $variationProduct['price'] ?? $price ?? 0;

                    return [
                        'id'           => $related->id,
                        'provider_id'  => $related->user_id,
                        'title'        => $related->source_name,
                        'slug'         => $related->slug,
                        'code'         => $related->source_code,
                        'image'        => $imageUrl,
                        'category'     => $related->category->name ?? null,
                        'starts_from'  => $variationPrice ?? $related->source_price ?? 0,
                        'rating'       => number_format($rating, 1, '.', ''),
                        'review_count' => $reviewCount,
                        'status'       => $related->status,
                        'wishlist'     => $wishlistExists,
                        'is_featured'  => (bool) rand(0, 1),
                        'verified'     => (bool) $related->verified_status,
                        'location'     => "Newyork, USA",
                        'shop_name'   => $shop->shop_name ?? 'Default Shop',
                        'shop_image'  => uploadedAsset($shop->shop_logo ?? ''),
                        'profile_image' => uploadedAsset('profile/' . $related->user->userDetail->profile_image ?? '', 'profile'),
                    ];
                });

            // Fetch specification meta
            $specificationMeta = Productmeta::where('product_id', $product->id)
                ->where('source_key', 'specification')
                ->first();

            $specification = [];
            if ($specificationMeta && !empty($specificationMeta->source_Values)) {
                $specification = json_decode($specificationMeta->source_Values, true);
            }

            // --- Wishlist Check ---
            $wishlistExists = Wishlist::where('user_id', $userId)
                ->where('service_id', $product->id)
                ->exists();

            // --- Ratings & Reviews ---
            $rating = Rating::where('product_id', $product->id)->where('parent_id', 0)->avg('rating') ?? 0;
            $reviewCount = Rating::where('product_id', $product->id)->where('parent_id', 0)->count();

            // Final structured response
            $data = [
                'id'             => $product->id,
                'slug'           => $product->slug,
                'title'          => $product->source_name,
                'shot_description' => $product->shot_description,
                'source_stock' => $product->source_stock,
                'code'           => $product->source_code,
                'category'       => $product->category_name,
                'starts_from'    => $product->source_price,
                'currency'       => getDefaultCurrencySymbol(),
                'status'         => $product->status,
                'wishlist'     => $wishlistExists,
                'is_featured'    => (bool) rand(0, 1),
                'rating' => number_format($rating, 1, '.', ''),
                'review_count' => $reviewCount,
                'no_of_bookings' => rand(1, 100),
                'location'       => "Newyork, USA",
                'map_url'        => $googleMapUrl,
                'include'        => $product->include,
                'overview'       => [
                    'description' => $product->source_description,
                ],
                'gallery'         => $gallery,
                'variations'      => $productVariations, // <-- VARIATIONS ADDED HERE
                'shipping_details'       => [
                    'international_shipping'        => $product->international_shipping,
                    'domestic_shipping'        => $product->domestic_shipping,
                ],
                'specification' => $specification,
                'provider'        => $providerData,
                'social_profile'  => $socialLinks,
                'related_products' => $relatedProducts,
            ];

            return [
                'code'    => 200,
                'message' => 'Product details retrieved successfully.',
                'data'    => $data,
            ];
        } catch (\Exception $e) {
            return [
                'code'    => 500,
                'message' => 'An error occurred while retrieving product details.',
                'error'   => $e->getMessage(),
            ];
        }
    }
    private function getFixedImagePath(string $originalPath): string
    {
        // Example input: "products/variations/abc123.jpg"
        // Desired output: "products/variations/fixed/abc123.jpg"
        $parts = explode('/', $originalPath);
        $fileName = end($parts); // Get "abc123.jpg"
        return 'products/variations/fixed/' . $fileName;
    }
    private function getFixedProductImagePath(string $originalPath): string
    {
        $parts = explode('/', $originalPath);
        $fileName = end($parts); // Get "abc123.jpg"
        return 'product_images/fixed/' . $fileName;
    }

    public function galleryandvariatonApi(Request $request): array
    {
        $productId = $request->input('product_id');
        $color     = $request->input('color');

        $gallery    = [];
        $variations = [];

        if ($productId && $color) {
            // ✅ Specific variation by product + color
            $variation = DB::table('product_variations')
                ->where('product_id', $productId)
                ->where('color', $color)
                ->first();

            if ($variation) {
                // Decode images
                $imagePaths = json_decode($variation->images, true) ?? [];
                $gallery = array_map(fn($path) => asset('storage/' . $path), $imagePaths);

                // Decode attributes (sizes, prices, stock, etc.)
                $attributes = json_decode($variation->data, true) ?? [];

                $variations[] = [
                    'id'        => $variation->id,
                    'color'     => $variation->color,
                    'attributes' => $attributes,
                ];
            }
        }

        if ($productId && empty($gallery)) {
            // ✅ If no color → show product meta gallery
            $gallery = DB::table('products_meta')
                ->where('product_id', $productId)
                ->where('source_key', 'product_image')
                ->pluck('source_values')
                ->map(fn($img) => $img ? asset('storage/' . $img) : asset('img/default-image.png'))
                ->toArray();

            // ✅ Get all variations
            $variations = DB::table('product_variations')
                ->where('product_id', $productId)
                ->get()
                ->map(function ($variation) {
                    $imagePaths = json_decode($variation->images, true) ?? [];
                    $images = array_map(fn($path) => asset('storage/' . $path), $imagePaths);

                    $attributes = json_decode($variation->data, true) ?? [];

                    return [
                        'id'        => $variation->id,
                        'color'     => $variation->color,
                        'images'    => $images,
                        'attributes' => $attributes,
                    ];
                })->toArray();
        }

        if (empty($gallery)) {
            $gallery = [asset('img/default-image.png')];
        }

        return [
            'gallery'   => $gallery,
            'variation' => $variations,
        ];
    }
    protected function isInvalidGoogleMapKey(string $url): bool
    {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return $httpCode !== 200 || str_contains($response, 'Maps Platform rejected your request');
    }
    public function store(Request $request): JsonResponse | RedirectResponse
    {
        DB::beginTransaction();

        try {
            $validator = $request->validate([
                'source_name' => 'required|unique:products|max:255',
                'source_code' => 'required',
                'category_fied' => 'required',
                'Subcategory_fied' => 'required',
                'source_desc' => 'required',
            ]);

            $slug_text = $request->source_name;
            $divider = "-";
            $slug_text = preg_replace('~[^\pL\d]+~u', $divider, $slug_text);

            // transliterate
            $slug_text = iconv('utf-8', 'us-ascii//TRANSLIT', $slug_text);

            // remove unwanted characters
            $slug_text = preg_replace('~[^-\w]+~', '', $slug_text);

            // trim
            $slug_text = trim($slug_text, $divider);

            // remove duplicate divider
            $slug_text = preg_replace('~-+~', $divider, $slug_text);

            // lowercase
            $slug_text = strtolower($slug_text);

            $userId = Auth::id();

            $data = [
                'source_name' => $request->source_name,
                'source_type' => 'service',
                'slug' => $slug_text,
                'user_id' => $userId,
                'source_category' => $request->category_fied,
                'source_subcategory' => $request->Subcategory_fied,
                'price_type' => $request->price_type,
                'source_price' => $request->fixed_price,
                'source_brand' => $request->source_brand,
                'source_stock' => $request->source_stock,
                'seo_title' => $request->seo_title,
                'seo_description' => $request->content,
                'include' => $request->include,
                'tags' => $request->tags,
                'source_description' => $request->source_desc,
                'source_code' => $request->source_code,
                'created_by' => $userId,
                'country' => $request->country,
                'state' => $request->state,
                'city' => $request->city,
            ];

            $currency = Service::create($data);

            if ($request->has('service_name')) {
                foreach ($request->service_name as $index => $name) {
                    $imagePath = null;

                    if ($request->hasFile("service_image.{$index}")) {
                        $image = $request->file("service_image.{$index}");
                        $imagePath = $image->store('additional_service_images', 'public');
                    }

                    ModelsAdditionalService::create([
                        'provider_id' => 0,
                        'service_id'  => $currency->id,
                        'name'        => $name,
                        'price'       => $request->service_price[$index],
                        'duration'    => $request->service_desc[$index],
                        'image'       => $imagePath,
                    ]);
                }
            }
            $message = 'Product created successfully.';
            $statusCode = 200;
            if ($request->hasFile('logo')) {

                foreach ($request->file('logo') as $photo) {
                    $logoPath = $photo->store('service_images', 'public');
                    $data = [
                        'product_id' => $currency->id,
                        'source_key' => 'product_image',
                        'source_Values' => $logoPath
                    ];
                    $product_meta = Productmeta::create($data);
                }
            }

            if ($request->service_name != "") {
                $service_name = serialize($request->service_name);
                $service_price = serialize($request->service_price);
                $service_desc = serialize($request->service_desc);

                $data = [
                    'product_id' => $currency->id,
                    'source_key' => "service_name",
                    'source_Values' => $service_name
                ];
                $product_meta = Productmeta::create($data);

                $data = [
                    'product_id' => $currency->id,
                    'source_key' => "service_price",
                    'source_Values' => $service_price
                ];
                $product_meta = Productmeta::create($data);
                $data = [
                    'product_id' => $currency->id,
                    'source_key' => "service_desc",
                    'source_Values' => $service_desc
                ];
                $product_meta = Productmeta::create($data);
            }
            if ($request->price_type != "") {

                $data = [
                    'product_id' => $currency->id,
                    'source_key' => $request->price_type,
                    'source_Values' => $request->fixed_price
                ];

                $product_meta = Productmeta::create($data);
            }
            DB::commit();

            incrementPageCacheVersion();
            return redirect('admin/services');

            return response()->json([
                'code' => $statusCode,
                'success' => true,
                'message' => $message,
                'data' => $currency
            ], $statusCode);
        } catch (\Exception $e) {
            DB::rollback();
            return redirect('admin/addservice')->withErrors($e->getMessage())->withInput();
        }
    }

    public function update(Request $request): RedirectResponse
    {
        $userId = Auth::id();

        $data = [
            'source_name' => $request->source_name,
            'source_type' => 'service',
            'source_category' => $request->category,
            'source_subcategory' => $request->Subcategory_fied,
            'price_type' => $request->price_type,
            'source_price' => $request->fixed_price,
            'source_brand' => $request->source_brand,
            'source_stock' => $request->source_stock,
            'seo_title' => $request->seo_title,
            'seo_description' => $request->content,
            'include' => $request->include,
            'tags' => $request->tags,
            'source_description' => $request->source_desc,
            'source_code' => $request->source_code,
            'updated_by' => $userId,
            'country' => $request->country,
            'state' => $request->state,
            'city' => $request->city,
        ];
        Service::where('id', $request->source_id)->update($data);
        if ($request->hasFile('logo')) {
            foreach ($request->file('logo') as $photo) {
                $logoPath = $photo->store('service_images', 'public');
                $data = [
                    'product_id' => $request->source_id,
                    'source_key' => 'product_image',
                    'source_Values' => $logoPath
                ];
                $product_meta = Productmeta::create($data);
            }
        }

        if ($request->has('service_name')) {
            foreach ($request->service_name as $index => $name) {
                $imagePath = null;

                if ($request->hasFile('service_image') && isset($request->file('service_image')[$index])) {
                    $image = $request->file('service_image')[$index];
                    $imagePath = $image->store('additional_service_images', 'public');
                }

                $existingService = ModelsAdditionalService::where('service_id', $request->source_id)
                    ->first();

                if ($existingService) {
                    $existingService->update([
                        'name'    => $request->service_name[$index],
                        'price'    => $request->service_price[$index],
                        'duration' => $request->service_desc[$index],
                        'image'    => $imagePath ?? $existingService->image,
                    ]);
                } else {
                    ModelsAdditionalService::create([
                        'provider_id' => 0,
                        'service_id'  => $request->source_id,
                        'name'        => $name,
                        'price'       => $request->service_price[$index],
                        'duration'    => $request->service_desc[$index],
                        'image'       => $imagePath,
                    ]);
                }
            }
        }

        $removedImages = str_replace('storage/', '', explode(',', $request->removed_images));
        if (!empty($removedImages)) {
            foreach ($removedImages as $removedImage) {
                if (!empty($removedImage)) {
                    if (Storage::exists($removedImage)) {
                        Storage::delete($removedImage);
                    }
                    Productmeta::where(['product_id' => $request->source_id, 'source_Values' => $removedImage])->delete();
                }
            }
        }

        incrementPageCacheVersion();

        return redirect('admin/services');
    }

    public function providerServiceIndex(Request $request): array
    {
        $orderBy = $request->input('order_by', 'desc');
        $sortBy = $request->input('sort_by', 'id');
        $authId = $request->input('auth_id');

        // Fetch user language
        $user = User::select('user_language_id')->where('id', $authId)->first();

        if (!$user) {
            return [
                'code'    => '404',
                'message' => __('User not found.'),
                'data'    => []
            ];
        }

        // Convert status text to numeric
        $status = null;
        if ($request->filled('status')) {
            $status = strtolower($request->status) === 'active' ? 1 : 0;
        }

        // Build base query
        $query = Service::query()
            ->whereHas('shop');

        // If mobile, filter by provider_id and request language
        if ($request->has('is_mobile') && $request->get('is_mobile') === "yes") {
            $query->where('user_id', $request->provider_id)
                ->where('language_id', $request->language_id);
        } else {
            $query->where('user_id', $authId)
                ->where('source_type', $request->type)
                ->where('language_id', $user->user_language_id);
        }

        // Filter by shop_id if provided
        if ($request->filled('shop_id')) {
            $query->where('shop_id', $request->shop_id);
        }

        // Filter by status (active/inactive)
        if (!is_null($status)) {
            $query->where('status', $status);
        }

        // Apply sorting
        $services = $query->orderBy($sortBy, $orderBy)->get();

        // Load location data from JSON files
        $countries = json_decode(file_get_contents(public_path('countries.json')), true);
        $states = json_decode(file_get_contents(public_path('states.json')), true);
        $cities = json_decode(file_get_contents(public_path('cities.json')), true);

        $countryMap = collect($countries['countries'])->pluck('name', 'id')->all();
        $stateMap = collect($states['states'])->pluck('name', 'id')->all();
        $cityMap = collect($cities['cities'])->pluck('name', 'id')->all();

        $data = $services->map(function ($service) use ($countryMap, $stateMap, $cityMap, $request) {
            $baseUrl = url('/storage');

            // Fetch product images
            $productMeta = ProductMeta::where('product_id', $service->id)
                ->where('source_key', 'product_image')
                ->pluck('source_values')
                ->map(fn($image) => uploadedAsset($image, 'default2'));

            // Additional services
            $additionalServices = ModelsAdditionalService::where('service_id', $service->id)->get(['name', 'price', 'duration', 'image']);
            $additionalServices->transform(function ($additional) use ($baseUrl) {
                if ($additional->image) {
                    $additional->image = $baseUrl . '/' . $additional->image;
                }
                return $additional;
            });

            // Location mappings
            $cityNames = collect(explode(',', $service->city))->map(fn($cityId) => $cityMap[$cityId] ?? $cityId)->unique()->implode(', ');
            $stateName = $stateMap[$service->state] ?? $service->state;
            $countryName = $countryMap[$service->country] ?? $service->country;

            // Category details
            $category = Category::select('name')->where('id', $service->source_category)->first();
            $subCategory = Category::select('name')->where('id', $service->source_subcategory)->first();

            $price = $service->source_price;
            $quntity = $service->source_stock;
            $variationProduct = getProductVariation($service->id);

            return [
                'id'                 => $service->id,
                'user_id'            => $service->user_id,
                'shop_id'            => $service->shop_id,
                'source_name'        => $service->source_name,
                'slug'               => $service->slug,
                'source_code'        => $service->source_code,
                'source_type'        => $service->source_type,
                'source_tag'         => $service->source_tag,
                'source_description' => $service->source_description,
                'source_category'    => $category->name ?? null,
                'source_subcategory' => $request->is_mobile === "yes" ? $subCategory->name ?? null : $service->source_subcategory,
                'source_price'       => $variationProduct['price'] ?? $price,
                'plan'               => $service->plan,
                'price_description'  => $service->price_description,
                'source_brand'       => $service->source_brand,
                'source_stock'       => $variationProduct['stock'] ?? $quntity,
                'seo_title'          => $service->seo_title,
                'tags'               => $service->tags,
                'featured'           => $service->featured,
                'popular'            => $service->popular,
                'seo_description'    => $service->seo_description,
                'price_type'         => $service->price_type,
                'duration'           => $service->duration,
                'country'            => $countryName,
                'state'              => $stateName,
                'city'               => $cityNames,
                'address'            => $service->address,
                'pincode'            => $service->pincode,
                'include'            => $service->include,
                'status'             => $service->status,
                'created_by'         => $service->created_by,
                'product_image'      => $productMeta,
                'additional_services' => $additionalServices,
                'verified_status'    => $service->verified_status,
            ];
        });

        return [
            'code'    => '200',
            'message' => __('Service details retrieved successfully.'),
            'data'    => $data
        ];
    }


    public function getDetails(Request $request, string $slug): array
    {
        $product = Service::where('slug', $slug)->where('language_id', $request->language_id)->first();

        if (!$product) {
            $product = Service::where('language_id', $request->language_id)
                ->where('parent_id', $request->service_id)
                ->orWhere('id', $request->parent_id)->first();
        }

        if ($product) {
            $productMeta = ProductMeta::where('product_id', $product->id)->get();

            $baseUrl = url('/storage');

            $productMeta->transform(function ($meta) use ($baseUrl) {
                if ($meta->source_key === 'product_image') {
                    $meta->source_Values = $baseUrl . '/' . $meta->source_Values;
                }
                return $meta;
            });

            $productMeta->transform(function ($meta) use ($baseUrl) {
                if ($meta->source_key === 'product_video') {
                    $meta->source_Values = $baseUrl . '/' . $meta->source_Values;
                }
                return $meta;
            });

            $additionalServices = ModelsAdditionalService::where('service_id', $product->id)->get();

            $additionalServices->transform(function ($additional) use ($baseUrl) {
                if ($additional->image) {
                    $additional->image = $baseUrl . '/' . $additional->image;
                }
                unset($additional->source_Values);
                return $additional;
            });

            $serviceBranches = ServiceBranch::where('service_id', $product->id)->get();

            $branchIds = $serviceBranches->pluck('branch_id');

            $serviceBranchDetails = Branches::whereIn('id', $branchIds)
                ->select('id', 'branch_name', 'branch_email', 'branch_mobile', 'branch_image', 'branch_address', 'branch_country', 'branch_state', 'branch_city', 'branch_zip')
                ->get();

            $serviceBranchDetails = $serviceBranchDetails->map(function ($branch) use ($serviceBranches) {
                $serviceBranch = $serviceBranches->firstWhere('branch_id', $branch->id);

                if ($serviceBranch) {
                    $staffIds = ServiceStaff::where('service_branch_id', $serviceBranch->id)->pluck('staff_id');

                    $staffDetails = User::whereIn('id', $staffIds)
                        ->select('id', 'name', 'phone_number')
                        ->get();

                    $branch->staff_details = $staffDetails;
                } else {
                    $branch->staff_details = [];
                }

                return $branch;
            });

            return [
                'code' => 200,
                'data' => [
                    'product' => $product,
                    'meta' => $productMeta,
                    'additional_services' => $additionalServices,
                    'service_branch' => $serviceBranchDetails,
                ]
            ];
        } else {
            return [
                'code' => 422,
                'message' => 'Product not found',
                'data' => [
                    'status' => true,
                ],
            ];
        }
    }

    public function providerServiceStore(Request $request): JsonResponse
    {
        $rules = [
            'service_name'    => 'required|string|max:255',
            'product_code'    => 'required|string|max:100',
            'category'        => 'required|integer',
            'sub_category'    => 'required|integer',
            'description'     => 'required|string|min:10',
            'seo_title'       => 'required|string|max:255',
            // 'seo_description' => 'required|string|max:500|min:20',
            'address'         => 'nullable|string|max:255|min:5',
            'pincode'         => 'nullable',
            'state'           => 'nullable|string|max:100',
            'city' => 'nullable|string',
            'country'         => 'nullable|string|max:100',
        ];

        $messages = [];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $slug = Str::slug($request->service_name);

        $slugCount = Service::where('slug', $slug)->count();

        if ($slugCount > 0) {
            $slug = $slug . '-' . ($slugCount + 1);
        }


        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            $include = !empty($request->include) ? implode(',', $request->include) : null;
        } else {
            $include = $request->include;
        }

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            $seoTag = !empty($request->seo_tag) ? implode(',', $request->seo_tag) : null;
        } else {
            $seoTag = $request->seo_tag;
        }

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            $userId = $request->user_id;
        } else {
            $userId = Auth::id();
        }

        if ($request->free_price === 'on') {
            $priceType = 'free';
            $servicePrice = 0;
        } else {
            $priceType = $request->price_type;
            $servicePrice = $request->service_price;
        }

        $serviceApprovalStatus = serviceApprovalStatus();
        $verifiedStatus = $serviceApprovalStatus == 1 ? 0 : 1;

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            $languageId = $request->language_id;
            $verifiedStatus = 1;
        } else {
            $languageId = $request->userLangId;
        }

        $data = [
            'user_id'            => $userId,
            'source_name'        => $request->service_name,
            'shop_id'        => $request->shop_id,
            'slug'               => $slug,
            'source_code'        => $request->product_code,
            'source_type'        => 'service',
            'source_category'    => $request->category,
            'source_subcategory' => $request->sub_category,
            'source_description' => $request->description,
            'seo_title'          => $request->seo_title,
            'tags'               => $seoTag,
            'seo_description'    => $request->seo_description,
            'price_type'         => $priceType,
            'source_price'       => $servicePrice,
            'duration'           => $priceType == 'hourly' ? 1 : ($request->duration_hours ?? $request->duration_minute),
            'price_description'  => $request->price_description,
            'featured'           => 1,
            'popular'            => 1,
            'country'            => $request->country,
            'state'              => $request->state,
            'city'               => $request->city,
            'address'            => $request->address,
            'pincode'            => $request->pincode,
            'language_id'        => $languageId,
            'include'            => $include,
            'parent_id'          => 0,
            'created_by'         => $userId,
            'verified_status'    => $verifiedStatus,
        ];

        $save = Service::create($data);

        if (!request()->has('is_mobile')) {
            if ($request->has('branch_staff_payload')) {
                $branchStaffPayload = json_decode($request->input('branch_staff_payload'), true);

                foreach ($branchStaffPayload as $branchData) {
                    $branchId = $branchData['branch_id'];
                    $staffIds = $branchData['staff_ids'];

                    $serviceBranch = ServiceBranch::create([
                        'service_id' => $save->id,
                        'branch_id' => $branchId,
                    ]);

                    foreach ($staffIds as $staffId) {
                        ServiceStaff::create([
                            'service_branch_id' => $serviceBranch->id,
                            'staff_id' => $staffId,
                        ]);
                    }
                }
            }
        }
        if ($request->has('is_mobile') && $request->get('is_mobile') === "yes") {
            $branchStaffPayload = $request->input('branch_staff_payload');

            if (!empty($branchStaffPayload) && is_array($branchStaffPayload)) {
                foreach ($branchStaffPayload as $branchData) {
                    $branchId = $branchData['branch_id'];
                    $staffIds = $branchData['staff_ids'];

                    $serviceBranch = ServiceBranch::create([
                        'service_id' => $save->id,
                        'branch_id' => $branchId,
                    ]);

                    foreach ($staffIds as $staffId) {
                        ServiceStaff::create([
                            'service_branch_id' => $serviceBranch->id,
                            'staff_id' => $staffId,
                        ]);
                    }
                }
            }
        }

        if ($request->has('day_checkbox') && $request->has('start_time') && $request->has('end_time')) {
            $dayCheckbox = json_decode($request->day_checkbox, true);
            $startTimeData = json_decode($request->start_time, true);
            $endTimeData = json_decode($request->end_time, true);

            foreach ($dayCheckbox as $day) {
                // Keep the key format exactly as in the JSON (no strtolower)
                $dayKey = $day;

                // Check if this day exists in both start and end time arrays
                if (!isset($startTimeData[$dayKey]) || !isset($endTimeData[$dayKey])) {
                    continue; // Skip this day if missing
                }

                $startTimes = $startTimeData[$dayKey];
                $endTimes = $endTimeData[$dayKey];

                foreach ($startTimes as $index => $startTime) {
                    if (!empty($startTime) && isset($endTimes[$index]) && !empty($endTimes[$index])) {
                        $slotCounter = $index + 1;
                        $endTime = $endTimes[$index];

                        // Store in DB similar to product_image
                        Productmeta::create([
                            'product_id'    => $save->id,
                            'source_key'    => strtolower($dayKey) . '_slot_' . $slotCounter,
                            'source_Values' => $startTime . ' - ' . $endTime,
                        ]);
                    }
                }
            }
        }

        if ($request->has('is_mobile') && $request->get('is_mobile') === "yes") {
            if ($request->has('day_checkbox') && $request->has('start_time') && $request->has('end_time')) {
                $dayCheckbox = $request->input('day_checkbox'); // Already an array
                $startTimeData = $request->input('start_time'); // Already an array
                $endTimeData = $request->input('end_time'); // Already an array

                foreach ($dayCheckbox as $day) {
                    $dayKey = strtolower($day);

                    if (isset($startTimeData[$dayKey]) && isset($endTimeData[$dayKey])) {
                        $startTimes = $startTimeData[$dayKey];
                        $endTimes = $endTimeData[$dayKey];

                        foreach ($startTimes as $index => $startTime) {
                            if (isset($endTimes[$index])) {
                                $slotCounter = $index + 1;
                                $endTime = $endTimes[$index];

                                $metaData = [
                                    'product_id'    => $save->id,
                                    'source_key'    => $dayKey . '_slot_' . $slotCounter,
                                    'source_Values' => $startTime . ' - ' . $endTime,
                                ];

                                Productmeta::create($metaData);
                            }
                        }
                    }
                }
            }
        }

        if (strtolower($request->basic) !== 'basic') {
            $metaprice = [
                'source_key'   => ucfirst($priceType),
                'source_Values' => $servicePrice,
                'product_id'   => $save->id,
            ];

            $saveData = Productmeta::create($metaprice);
        }

        $metaMapping = [
            'add_name'     => 'service_name',
            'add_price'    => 'service_price',
            'add_duration' => 'service_desc',
        ];

        foreach ($metaMapping as $requestKey => $metaKey) {
            if ($request->has($requestKey)) {
                $serializedValue = serialize($request->get($requestKey));

                Productmeta::create([
                    'product_id'    => $save->id,
                    'source_key'    => $metaKey,
                    'source_Values' => $serializedValue,
                ]);
            }
        }

        $plans = ['basic', 'premium', 'pro'];
        foreach ($plans as $plan) {
            $priceField = "{$plan}_service_price";
            $descriptionField = "{$plan}_price_description";

            if ($request->has($priceField) && $request->filled($priceField)) {
                Productmeta::create([
                    'product_id'    => $save->id,
                    'source_key'    => "{$plan}_service_price",
                    'source_Values' => $request->$priceField,
                ]);
            }

            if ($request->has($descriptionField) && $request->filled($descriptionField)) {
                Productmeta::create([
                    'product_id'    => $save->id,
                    'source_key'    => "{$plan}_price_description",
                    'source_Values' => $request->$descriptionField,
                ]);
            }
        }

        if ($request->has('add_name')) {
            foreach ($request->add_name as $index => $name) {
                $imagePath = null;

                if ($request->hasFile("add_image.{$index}")) {
                    $image = $request->file("add_image.{$index}");

                    if ($image instanceof \Illuminate\Http\UploadedFile) {
                        $imagePath = $image->store('additional_service_images', 'public');
                    } elseif (is_array($image)) {
                        foreach ($image as $img) {
                            if ($img instanceof \Illuminate\Http\UploadedFile) {
                                $imagePath = $img->store('additional_service_images', 'public');
                            }
                        }
                    }
                }

                ModelsAdditionalService::create([
                    'provider_id' => 0,
                    'service_id'  => $save->id,
                    'name'        => $name,
                    'price'       => $request->add_price[$index],
                    'duration'    => $request->add_duration[$index],
                    'image'       => $imagePath,
                ]);
            }
        }

        if ($request->hasFile('client_image')) {
            $file = $request->file('client_image');
            if ($file instanceof \Illuminate\Http\UploadedFile) {
                $filename = Str::uuid() . '_' . time() . '.' . $file->getClientOriginalExtension();
                $file->storeAs('testimonials', $filename, 'public');
                $data['client_image'] = $filename;
            }
        }

        $upload_method = GlobalSetting::where('key', 'aws_status')->first();



        if ($request->hasFile('service_images') && $request->file('service_images')) {
            $images = $request->file('service_images');

            // Handle multiple images
            if (!$images instanceof \Illuminate\Http\UploadedFile) {
                foreach ($images as $image) {
                    if ($image instanceof \Illuminate\Http\UploadedFile) {

                        // Generate a unique filename
                        $filename = Str::random(40) . '.' . $image->getClientOriginalExtension();

                        // Define storage paths
                        $originalPath = 'product_images/' . $filename;
                        $fixedPath    = 'product_images/fixed/' . $filename;

                        /**
                         * 1. Save the ORIGINAL image
                         */
                        Storage::disk('public')->putFileAs('product_images', $image, $filename);

                        /**
                         * 2. Create and save the FIXED version (840x491) using GD
                         */
                        $this->resizeAndSaveImage(
                            Storage::disk('public')->path($originalPath),
                            $fixedPath,
                            840,
                            491
                        );

                        /**
                         * 3. Save only the original image path in DB
                         */
                        Productmeta::create([
                            'product_id'    => $save->id,
                            'source_key'    => 'product_image',
                            'source_Values' => $originalPath, // ONLY original image path
                        ]);
                    }
                }
            }
        }

        if ($request->service_video) {
            $videoData = [
                'product_id' => $save->id,
                'source_key' => 'video_link',
                'source_Values' => $request->service_video, // Corrected here
            ];

            Productmeta::create($videoData);
        }
        $sourcenotify = "New Service";
        $notificationType = 24;
        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->first();
        $admins = User::where('user_type', 1)->get();
        $todescriptionContent = __('New service is created');
        $fromdescriptionadmin = __('New service is created');

        try {
            foreach ($admins as $admin) {
                $data = [
                    'communication_type' => '3',
                    'source' => $sourcenotify,
                    'reference_id' => $save->id,
                    'user_id' => $userId,
                    'to_user_id' => $admin->id,
                    'from_description' => $fromdescriptionadmin ?? null,
                    'to_description' => $todescriptionContent  ?? null,
                ];

                $notificationRequest = new Request($data);
                $notification = new NotificationController();
                $notification->Storenotification($notificationRequest);
            }
        } catch (\Exception $e) {
            Log::error('Error storing notifications: ' . $e->getMessage());
        }

        if (!$save) {
            return response()->json(['message' => 'Something went wrong while saving the service!'], 500);
        }

        $redirectUrl = route('provider.service');

        // ________________________________
        //Cache Invalidation______
        $indexKey = 'services_list_index';
        $trackedKeys = Cache::get($indexKey, []);

        foreach ($trackedKeys as $key) {
            Cache::forget($key);
        }

        Cache::forget($indexKey);
        incrementPageCacheVersion();
        //Cache Invalidation______
        // ________________________________

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            return response()->json([
                'code' => 200,
                'message' => __('service_create_success'),
                'verify_status' => $verifiedStatus,
            ], 200);
        }

        return response()->json([
            'code'    => 200,
            'message' => __('service_create_success'),
            'redirect_url' => $redirectUrl,
            'service_approval_status' => $serviceApprovalStatus,
            'data'    => [],
        ], 200);
    }

    public function verifyService(Request $request): array
    {
        $languageCode = $request->language_code ?? app()->getLocale();
        try {

            Service::where('id', $request->id)
                ->update(['verified_status' => 1]);

            $service = Service::where('id', $request->id)->first();

            $data = User::with('userDetails')
                ->where('id', $service->user_id)
                ->first();
            $serviceName = $service->source_name ?? '';

            $this->sendServiceVerificationEmail($data, 33, $serviceName);

            return [
                'code' => 200,
                'message' => __('service_verification_status_update_success', [], $languageCode)
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => __('service_verification_status_update_error', [], $languageCode),
                'error' => $e->getMessage()
            ];
        }
    }

    private function sendServiceVerificationEmail($data, $notificationType, $serviceName): void
    {
        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', $notificationType)
            ->first();

        if ($template) {
            $settings = GlobalSetting::whereIn('key', ['company_name', 'website', 'phone_no', 'site_email'])->pluck('value', 'key');
            $companyName = $settings['company_name'] ?? '';
            $companyWebsite = $settings['website'] ?? '';
            $companyPhone = $settings['phone_no'] ?? '';
            $companyEmail = $settings['site_email'] ?? '';
            $contact = $companyEmail . ' | ' . $companyPhone;
            $customerName = $data['userDetails']['first_name'] . ' ' . $data['userDetails']['last_name'];

            // Prepare email data
            $subject = str_replace(
                ['{{service_name}}', '{{provider_name}}', '{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', '{{website_link}}', '{{contact}}'],
                [$serviceName, $customerName, $customerName, $data['userDetails']['first_name'], $data['userDetails']['last_name'], $customerName, $data['phone_number'], $data['email'], $companyName, $companyWebsite, $contact],
                $template->subject
            );

            $content = str_replace(
                ['{{service_name}}', '{{provider_name}}', '{{user_name}}', '{{first_name}}', '{{last_name}}', '{{customer_name}}', '{{phone_number}}', '{{email_id}}', '{{company_name}}', '{{website_link}}', '{{contact}}'],
                [$serviceName, $customerName, $customerName, $data['userDetails']['first_name'], $data['userDetails']['last_name'], $customerName, $data['phone_number'], $data['email'], $companyName, $companyWebsite, $contact],
                $template->content
            );

            $emailData = [
                'to_email' => $data['email'],
                'subject' => $subject,
                'content' => $content
            ];

            try {
                $emailRequest = new Request($emailData);
                $emailController = new EmailController();
                $emailController->sendEmail($emailRequest);
            } catch (\Exception $e) {
                Log::error('Failed to send registration email: ' . $e->getMessage());
            }
        }
    }

    public function providerServiceUpdate(Request $request): JsonResponse
    {
        $rules = [
            'service_name'    => 'required|string|max:255',
            'product_code'    => 'required|string|max:100',
            'category'        => 'required',
            'sub_category'    => 'nullable',
            'description'     => 'required|string',
            'seo_title'       => 'required|string|max:255',
            'seo_description' => 'required|string|max:500',
            'price_type'      => 'required',
        ];

        $messages = [];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $product = $request->id;

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            $userId = $request->user_id;
        } else {
            $userId = Auth::id();
        }

        if ($request->has('is_mobile') && $request->get('is_mobile') === "yes") {
            $includeArray = json_decode($request->include, true);
            $include = implode(',', $includeArray);
        } else {
            $include = $request->include;
        }

        if ($request->has('is_mobile') && $request->get('is_mobile') === "yes") {
            $SeoArray = json_decode($request->seo_tag, true);
            $seoTag = implode(',', $SeoArray);
        } else {
            $seoTag = $request->seo_tag;
        }

        $slug = Str::slug($request->service_name);
        $slugCount = Service::where('slug', $slug)->count();

        $data = [
            'user_id'            => $userId,
            'source_name'        => $request->service_name,
            'shop_id'        => $request->shop_id,
            'slug'               => $slug,
            'source_code'        => $request->product_code,
            'source_type'        => 'service',
            'source_category'    => $request->category,
            'source_subcategory' => $request->sub_category,
            'source_description' => $request->description,
            'seo_title'          => $request->seo_title,
            'tags'               => $seoTag,
            'seo_description'    => $request->seo_description,
            'price_type'         => $request->price_type,
            'source_price'       => $request->service_price,
            'price_description'  => $request->price_description,
            'duration'           => $request->price_type == 'hourly' ? 1 : ($request->duration_hours ?? $request->duration_minute),
            'featured'           => 1,
            'popular'            => 1,
            'country'            => $request->country,
            'state'              => $request->state,
            'city'               => $request->city,
            'address'            => $request->address,
            'pincode'            => $request->pincode,
            'include'            => $include,
            'language_id'        => $request->language_id ?? 1,
            'created_by'         => $userId,
        ];

        $existingService = Service::where('id', $product)
            ->where('language_id', $request->language_id ?? 1)
            ->first();

        $existingLangService = Service::where('parent_id', $product)
            ->where('language_id', $request->language_id  ?? 1)
            ->first();

        if ($existingService) {
            if ($slugCount > 0 && $slug != $existingService->slug) {
                $data['slug'] = $slug . '-' . ($slugCount + 1);
            }
            $update = Service::where('id', $product)
                ->where('language_id', $request->language_id  ?? 1)
                ->update($data);

            $updatedService = Service::find($product);
        } else if ($existingLangService) {
            if ($slugCount > 0 && $slug != $existingLangService->slug) {
                $data['slug'] = $slug . '-' . ($slugCount + 1);
            }
            $update = Service::where('parent_id', $product)
                ->where('language_id', $request->language_id  ?? 1)
                ->update($data);

            $updatedService = Service::where('parent_id', $product)
                ->where('language_id', $request->language_id  ?? 1)
                ->first();
        } else {
            if ($slugCount > 0) {
                $data['slug'] = $slug . '-' . ($slugCount + 1);
            }
            $data['parent_id'] = $request->serviceId ?? 0;
            $updatedService = Service::create($data);
        }

        if ($updatedService) {
            $metaData = [
                'source_key'    => ucfirst($request->price_type),
                'source_Values' => $request->service_price,
                'product_id'    => $updatedService->id,
            ];

            Productmeta::updateOrCreate(
                ['product_id' => $updatedService->id],
                $metaData
            );
        }

        if ($request->has('branch_staff_payload')) {
            $branchStaffPayload = json_decode($request->input('branch_staff_payload'), true);

            $existingServiceBranches = ServiceBranch::where('service_id', $updatedService->id)->get();
            $existingBranchIds = $existingServiceBranches->pluck('branch_id')->toArray();

            foreach ($branchStaffPayload as $branchData) {
                $branchId = $branchData['branch_id'];
                $staffIds = $branchData['staff_ids'];

                $serviceBranch = $existingServiceBranches->firstWhere('branch_id', $branchId);

                if ($serviceBranch) {
                    $existingStaffIds = ServiceStaff::where('service_branch_id', $serviceBranch->id)->pluck('staff_id')->toArray();

                    foreach ($staffIds as $staffId) {
                        if (!in_array($staffId, $existingStaffIds)) {
                            ServiceStaff::create([
                                'service_branch_id' => $serviceBranch->id,
                                'staff_id' => $staffId,
                            ]);
                        }
                    }

                    ServiceStaff::where('service_branch_id', $serviceBranch->id)
                        ->whereNotIn('staff_id', $staffIds)
                        ->delete();
                } else {
                    $serviceBranch = ServiceBranch::create([
                        'service_id' => $updatedService->id,
                        'branch_id' => $branchId,
                    ]);

                    foreach ($staffIds as $staffId) {
                        ServiceStaff::create([
                            'service_branch_id' => $serviceBranch->id,
                            'staff_id' => $staffId,
                        ]);
                    }
                }
            }

            $newBranchIds = array_column($branchStaffPayload, 'branch_id');
            ServiceBranch::where('service_id', $updatedService->id)
                ->whereNotIn('branch_id', $newBranchIds)
                ->each(function ($serviceBranch) {
                    ServiceStaff::where('service_branch_id', $serviceBranch->id)->delete();
                    $serviceBranch->delete();
                });
        }

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            if ($request->has('day_checkbox') && $request->has('start_time') && $request->has('end_time')) {
                // Decode the input JSON fields
                $dayCheckbox = json_decode($request->day_checkbox, true);
                $startTimeData = json_decode($request->start_time, true);
                $endTimeData = json_decode($request->end_time, true);

                // Remove all existing meta data for the product
                Productmeta::where('product_id', $updatedService->id)->delete();

                // Insert updated meta data
                foreach ($dayCheckbox as $day) {
                    $dayKey = strtolower($day);

                    // Ensure start_time and end_time data exist for the day
                    if (isset($startTimeData[$dayKey]) && isset($endTimeData[$dayKey])) {
                        $startTimes = $startTimeData[$dayKey];
                        $endTimes = $endTimeData[$dayKey];

                        foreach ($startTimes as $index => $startTime) {
                            if (isset($endTimes[$index])) {
                                $slotCounter = $index + 1;
                                $endTime = $endTimes[$index];

                                $metaData = [
                                    'product_id'    => $updatedService->id,
                                    'source_key'    => $dayKey . '_slot_' . $slotCounter,
                                    'source_Values' => $startTime . ' - ' . $endTime,
                                ];

                                Productmeta::create($metaData);
                            }
                        }
                    }
                }
            }
        }

        $metaMapping = [
            'add_name'     => 'service_name',
            'add_price'    => 'service_price',
            'add_duration' => 'service_desc',
        ];


        foreach ($metaMapping as $requestKey => $metaKey) {
            if ($request->has($requestKey)) {
                $serializedValue = serialize($request->$requestKey);
                Productmeta::updateOrCreate(
                    [
                        'product_id' => $updatedService->id,
                        'source_key' => $metaKey,
                    ],
                    [
                        'source_Values' => $serializedValue,
                    ]
                );
            }
        }

        $plans = ['basic', 'premium', 'pro'];

        foreach ($plans as $plan) {
            $priceField = "{$plan}_service_price";
            $descriptionField = "{$plan}_price_description";

            if ($request->has($priceField) && $request->filled($priceField)) {
                Productmeta::updateOrCreate(
                    [
                        'product_id' => $updatedService->id,
                        'source_key' => "{$plan}_service_price",
                    ],
                    [
                        'source_Values' => $request->$priceField,
                    ]
                );
            }

            if ($request->has($descriptionField) && $request->filled($descriptionField)) {
                Productmeta::updateOrCreate(
                    [
                        'product_id' => $updatedService->id,
                        'source_key' => "{$plan}_price_description",
                    ],
                    [
                        'source_Values' => $request->$descriptionField,
                    ]
                );
            }
        }

        if ($request->has('services')) {
            // Initialize arrays for storing field values
            $serviceNames = [];
            $servicePrices = [];
            $serviceDescs = [];

            // Loop through the services to extract and populate values
            foreach ($request->input('services') as $service) {
                $serviceNames[] = $service['name'];
                $servicePrices[] = $service['price'];
                $serviceDescs[] = $service['duration'];
            }

            // Serialize the data for each key
            $serializedServiceNames = serialize($serviceNames);
            $serializedServicePrices = serialize($servicePrices);
            $serializedServiceDescs = serialize($serviceDescs);

            // Define fields and their serialized values
            $fields = [
                'service_name' => $serializedServiceNames,
                'service_price' => $serializedServicePrices,
                'service_desc' => $serializedServiceDescs,
            ];

            // Loop through fields and update or create Productmeta records
            foreach ($fields as $sourceKey => $serializedValue) {
                Productmeta::updateOrCreate(
                    [
                        'product_id' => $updatedService->id,
                        'source_key' => $sourceKey,
                    ],
                    [
                        'source_Values' => $serializedValue,
                    ]
                );
            }
        }

        if ($request->has('branch_select')) {
            $currentBranchIds = ServiceBranch::where('service_id', $updatedService->id)
                ->pluck('branch_id')
                ->toArray();

            $newBranchIds = $request->branch_select;

            ServiceBranch::where('service_id', $updatedService->id)
                ->whereNotIn('branch_id', $newBranchIds)
                ->delete();

            foreach ($newBranchIds as $branchId) {
                ServiceBranch::updateOrCreate([
                    'service_id' => $updatedService->id,
                    'branch_id' => $branchId,
                ]);
            }
        } else {
            ServiceBranch::where('service_id', $updatedService->id)->delete();
        }

        if ($request->has('services')) {
            foreach ($request->input('services') as $service) {
                $name = $service['name'];
                $price = $service['price'];
                $duration = $service['duration'];

                // Find the existing service based on `name` or any other unique identifier
                $existingService = ModelsAdditionalService::where('service_id', $updatedService->id)
                    ->where('name', $name) // Match by name or any unique field
                    ->first();

                if ($existingService) {
                    $existingService->update([
                        'name'    => $name,
                        'price'   => $price,
                        'duration' => $duration,
                    ]);
                } else {
                    ModelsAdditionalService::create([
                        'provider_id' => 0,
                        'service_id'  => $updatedService->id,
                        'name'        => $name,
                        'price'       => $price,
                        'duration'    => $duration,
                    ]);
                }
            }
        }

        if ($request->has('day_checkbox') && $request->has('start_time') && $request->has('end_time')) {
            $dayCheckbox = json_decode($request->day_checkbox, true);
            $startTimeData = json_decode($request->start_time, true);
            $endTimeData = json_decode($request->end_time, true);

            foreach ($dayCheckbox as $day) {
                // Use the exact day key from JSON
                $dayKey = $day;

                // Validate that both start and end time data exist for this day
                if (!isset($startTimeData[$dayKey]) || !isset($endTimeData[$dayKey])) {
                    continue; // Skip if missing
                }

                $startTimes = $startTimeData[$dayKey];
                $endTimes = $endTimeData[$dayKey];

                foreach ($startTimes as $index => $startTime) {
                    if (!empty($startTime) && isset($endTimes[$index]) && !empty($endTimes[$index])) {
                        $slotCounter = $index + 1;
                        $sourceKey = strtolower($dayKey) . '_slot_' . $slotCounter; // Store in lowercase like in create
                        $sourceValues = $startTime . ' - ' . $endTimes[$index];

                        // Check if this slot already exists
                        $existingMeta = Productmeta::where('product_id', $updatedService->id)
                            ->where('source_key', $sourceKey)
                            ->first();

                        if ($existingMeta) {
                            // Update existing record
                            $existingMeta->update([
                                'source_values' => $sourceValues,
                            ]);
                        } else {
                            // Create a new record
                            Productmeta::create([
                                'product_id'    => $updatedService->id,
                                'source_key'    => $sourceKey,
                                'source_values' => $sourceValues,
                            ]);
                        }
                    }
                }
            }
        }

        if ($request->has('add_name')) {
            foreach ($request->add_name as $index => $name) {
                $imagePath = null;

                // Check if the 'add_image' has a file for this index
                if ($request->hasFile('add_image') && isset($request->file('add_image')[$index])) {
                    $image = $request->file('add_image')[$index];

                    // If it's a single file, process it
                    if ($image instanceof \Illuminate\Http\UploadedFile) {
                        $imagePath = $image->store('additional_service_images', 'public');
                    }
                }

                // Check if the service already exists
                $existingService = ModelsAdditionalService::where('service_id', $updatedService->id)
                    ->where('name', $name)
                    ->first();

                // If the service exists, update it, otherwise create a new one
                if ($existingService) {
                    $existingService->update([
                        'price'    => $request->add_price[$index],
                        'duration' => $request->add_duration[$index],
                        'image'    => $imagePath ?? $existingService->image, // Preserve existing image if no new image is provided
                    ]);
                } else {
                    ModelsAdditionalService::create([
                        'provider_id' => 0,
                        'service_id'  => $updatedService->id,
                        'name'        => $name,
                        'price'       => $request->add_price[$index],
                        'duration'    => $request->add_duration[$index],
                        'image'       => $imagePath, // Set the image path if a new image is uploaded
                    ]);
                }
            }
        }


        if ($request->hasFile('service_images') && $request->file('service_images')) {
            // Delete existing records for this service
            Productmeta::where('product_id', $updatedService->id)
                ->where('source_key', 'product_image')
                ->delete();

            $images = $request->file('service_images');

            // Handle multiple images
            if (!$images instanceof \Illuminate\Http\UploadedFile) {
                foreach ($images as $image) {
                    if ($image instanceof \Illuminate\Http\UploadedFile) {
                        // Generate unique filename
                        $filename = Str::random(40) . '.' . $image->getClientOriginalExtension();

                        // Define paths
                        $originalPath = 'product_images/' . $filename;
                        $fixedPath    = 'product_images/fixed/' . $filename;

                        /**
                         * 1. Save original image
                         */
                        Storage::disk('public')->putFileAs('product_images', $image, $filename);

                        /**
                         * 2. Create fixed resized image (840×491)
                         */
                        $this->resizeAndSaveImage(
                            Storage::disk('public')->path($originalPath),
                            $fixedPath,
                            840,
                            491
                        );

                        /**
                         * 3. Save only original path in DB
                         */
                        Productmeta::create([
                            'product_id'    => $updatedService->id,
                            'source_key'    => 'product_image',
                            'source_Values' => $originalPath,
                        ]);
                    }
                }
            } else {
                // Single file upload
                $filename = Str::random(40) . '.' . $images->getClientOriginalExtension();

                $originalPath = 'product_images/' . $filename;
                $fixedPath    = 'product_images/fixed/' . $filename;

                // Save original image
                Storage::disk('public')->putFileAs('product_images', $images, $filename);

                // Create fixed resized image
                $this->resizeAndSaveImage(
                    Storage::disk('public')->path($originalPath),
                    $fixedPath,
                    840,
                    491
                );

                // Save to DB
                Productmeta::create([
                    'product_id'    => $updatedService->id,
                    'source_key'    => 'product_image',
                    'source_Values' => $originalPath,
                ]);
            }
        }

        if ($request->service_video) {
            $videoData = [
                'product_id' => $request->id,
                'source_key' => 'video_link',
            ];

            Productmeta::updateOrCreate(
                ['product_id' => $request->id, 'source_key' => 'video_link'],
                ['source_Values' => $request->service_video]
            );
        }

        $redirectUrl = route('provider.service');

        // ________________________________
        //Cache Invalidation______
        $indexKey = 'services_list_index';
        $trackedKeys = Cache::get($indexKey, []);

        foreach ($trackedKeys as $key) {
            Cache::forget($key);
        }

        Cache::forget($indexKey);

        incrementPageCacheVersion();
        //Cache Invalidation______
        // ________________________________

        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            return response()->json([
                'code'    => 200,
                'message' => __('service_update_success'),
                'data'    => [],
            ], 200);
        } else {
            return response()->json([
                'code'    => 200,
                'message' => __('service_update_success'),
                'redirect_url' => $redirectUrl,
                'data'    => [],
            ], 200);
        }
    }

    public function deleteServiceImage(string $id): array
    {
        try {
            $productMeta = Productmeta::where('id', $id)->first();

            if (!$productMeta) {
                return [
                    'code' => 200,
                    'message' => 'Image not found.',
                ];
            }

            $productMeta->delete();

            // ________________________________
            //Cache Invalidation______
            $indexKey = 'services_list_index';
            $trackedKeys = Cache::get($indexKey, []);

            foreach ($trackedKeys as $key) {
                Cache::forget($key);
            }

            Cache::forget($indexKey);

            incrementPageCacheVersion();
            //Cache Invalidation______
            // ________________________________

            return [
                'code' => 200,
                'message' => 'Image deleted successfully.',
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => 'An error occurred while deleting the image.',
            ];
        }
    }

    public function deleteSlot(string $id): array
    {
        try {
            $slot = Productmeta::where('id', $id)->first();

            if (!$slot) {
                return [
                    'code' => 200,
                    'message' => 'Time slot not found.',
                ];
            }

            $slot->delete();

            // ________________________________
            //Cache Invalidation______
            $indexKey = 'services_list_index';
            $trackedKeys = Cache::get($indexKey, []);

            foreach ($trackedKeys as $key) {
                Cache::forget($key);
            }

            Cache::forget($indexKey);

            incrementPageCacheVersion();
            //Cache Invalidation______
            // ________________________________

            return [
                'code' => 200,
                'message' => 'Time slot deleted successfully.',
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => 'An error occurred while deleting the time slot.',
            ];
        }
    }

    public function deleteAdditionalServices(string $id): array
    {
        try {
            $service = ModelsAdditionalService::where('id', $id)->first();

            if (!$service) {
                return [
                    'code' => 200,
                    'message' => 'Time slot not found.',
                ];
            }

            $service->delete();

            // ________________________________
            //Cache Invalidation______
            $indexKey = 'services_list_index';
            $trackedKeys = Cache::get($indexKey, []);

            foreach ($trackedKeys as $key) {
                Cache::forget($key);
            }

            Cache::forget($indexKey);
            incrementPageCacheVersion();
            //Cache Invalidation______
            // ________________________________

            return [
                'code' => 200,
                'message' => 'Time slot deleted successfully.',
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => 'An error occurred while deleting the time slot.',
            ];
        }
    }

    public function deleteServices(Request $request): array
    {
        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
            $id = $request->service_id;
        } else {
            $id = $request->input('id');
        }

        $delete = Service::where('id', $id)->first();

        $delete->deleted_at = Carbon::now();
        $delete->save();

        Productmeta::where('product_id', $id)->delete();

        ModelsAdditionalService::where('service_id', $id)->delete();

        // ________________________________
        //Cache Invalidation______
        $indexKey = 'services_list_index';
        $trackedKeys = Cache::get($indexKey, []);

        foreach ($trackedKeys as $key) {
            Cache::forget($key);
        }

        Cache::forget($indexKey);
        incrementPageCacheVersion();
        //Cache Invalidation______
        // ________________________________

        return [
            'code' => '200',
            'success' => true,
            'message' => __('service_delete_success'),
        ];
    }

    public function status(Request $request): array
    {
        $request->validate([
            'id' => 'required',
        ]);

        $id = $request->input('id');

        $service = Service::select('status')->where('id', $id)->first();

        if ($service) {
            $newStatus = $service->status == 0 ? 1 : 0;

            DB::table('products')
                ->where('id', $id)
                ->update(['status' => $newStatus]);

            // ________________________________
            //Cache Invalidation______
            $indexKey = 'services_list_index';
            $trackedKeys = Cache::get($indexKey, []);

            foreach ($trackedKeys as $key) {
                Cache::forget($key);
            }

            Cache::forget($indexKey);
            incrementPageCacheVersion();
            //Cache Invalidation______
            // ________________________________

            return [
                'code' => '200',
                'success' => true,
                'message' => __('Status updated successfully.'),
            ];
        }

        return [
            'code' => '200',
            'success' => false,
            'message' => 'Service not found.',
        ];
    }

    public function checkUnique(Request $request): bool
    {
        $serviceName = $request->input('service_name');
        $languageId = $request->input('language_id');

        $exists = DB::table('products')
            ->where('source_name', $serviceName)
            ->whereNull('deleted_at')
            ->when($languageId, function ($query, $languageId) {
                return $query->where('language_id', $languageId);
            })
            ->exists();

        return !$exists;
    }

    public function checkEditUnique(Request $request): bool
    {
        $serviceName = $request->input('edit_service_name');
        $id = $request->input('id');
        $languageId = $request->input('language_id');

        $exists = DB::table('products')
            ->where('source_name', $serviceName)
            ->whereNull('deleted_at')
            ->when($languageId, function ($query, $languageId) {
                return $query->where('language_id', $languageId);
            })
            ->when($id, function ($query, $id) {
                // Exclude the current record with the given ID
                return $query->where('id', '!=', $id);
            })
            ->exists();

        return !$exists;
    }

    public function providerSub(Request $request): array
    {
        $id = Auth::id();

        $packageTrx = PackageTrx::where('provider_id', $id)
            ->where('status', 1)
            ->where('payment_status', 2)
            ->where('subscription_type', 'regular')
            ->select(
                'id',
                'provider_id',
                'number_of_service',
                'number_of_product',
                'number_of_staff',
                'package_id',
                'trx_date',
                'end_date'
            )
            ->orderByDesc('id')
            ->first();

        $topup = PackageTrx::where('provider_id', $id)
            ->where('status', 1)
            ->where('payment_status', 2)
            ->where('subscription_type', 'topup')
            ->select(
                'id',
                'provider_id',
                'number_of_service',
                'number_of_product',
                'number_of_staff',
                'package_id',
                'trx_date',
                'end_date'
            )
            ->orderByDesc('id')
            ->first();

        if (!$packageTrx && !$topup) {
            return [
                'code' => 200,
                'success' => false,
                'no_package' => true,
                'message' => 'No Subscription and Topup Found.'
            ];
        }

        $currentDate = now();

        $packageEndDateCount = $packageTrx ? $currentDate->diffInDays(Carbon::parse($packageTrx->end_date), false) : -999;
        $topupEndDateCount   = $topup ? $currentDate->diffInDays(Carbon::parse($topup->end_date), false) : -999;

        $regularCount = 0;
        $topupCount   = 0;
        $totalCount = 0;
        $countExist = 0;
        $redirectUrl = '';

        if ($request->type == 'product') {
            $regularCount = $packageTrx ? $packageTrx->number_of_product : 0;
            $topupCount   = $topup ? $topup->number_of_product : 0;
            $countExist = Product::where('user_id', $id)
                ->where('source_type', 'product')->count();
            $redirectUrl = route('provider.product.create');
        } else if ($request->type == 'service') {
            $regularCount = $packageTrx ? $packageTrx->number_of_service : 0;
            $topupCount   = $topup ? $topup->number_of_service : 0;
            $countExist = Product::where('user_id', $id)
                ->where('source_type', 'service')->count();
            $redirectUrl = route('provider.add.service');
        } else {
            $regularCount = $packageTrx ? $packageTrx->number_of_staff : 0;
            $topupCount   = $topup ? $topup->number_of_staff : 0;
            $countExist = UserDetail::where('parent_id', $id)->count();
            $redirectUrl = route('provider.add_staffs');
        }

        if ($packageEndDateCount > -1) {
            $totalCount += $regularCount;
        }

        if ($topupEndDateCount > -1) {
            $totalCount += $topupCount;
        }

        if ($packageEndDateCount <= -1 && $topupEndDateCount <= -1) {
            return [
                'code' => 200,
                'success' => true,
                'sub_end' => true,
                'message' => 'Subscription or Topup Ended.',
            ];
        }

        if ($countExist >= $totalCount) {
            return [
                'code' => 200,
                'success' => true,
                'sub_count_end' => true,
                'message' => 'Subscription or Topup limit Ended.',
            ];
        }

        $language = Language::select('id', 'code')->where('is_default', 1)->where('status', 1)->first();

        return [
            'code' => 200,
            'success' => true,
            'redirect_url' => $redirectUrl,
            'language' => $language,
            'message' => 'Successfully.',
        ];
    }

    public function providerSubApi(Request $request): array
    {
        $id = $request->provider_id;

        $packageTrx = PackageTrx::where('provider_id', $id)
            ->where('status', 1)
            ->where('payment_status', 2)
            ->where('subscription_type', 'regular')
            ->select(
                'id',
                'provider_id',
                'number_of_service',
                'number_of_product',
                'number_of_staff',
                'package_id',
                'trx_date',
                'end_date'
            )
            ->orderByDesc('id')
            ->first();

        $topup = PackageTrx::where('provider_id', $id)
            ->where('status', 1)
            ->where('payment_status', 2)
            ->where('subscription_type', 'topup')
            ->select(
                'id',
                'provider_id',
                'number_of_service',
                'number_of_product',
                'number_of_staff',
                'package_id',
                'trx_date',
                'end_date'
            )
            ->orderByDesc('id')
            ->first();

        if (!$packageTrx && !$topup) {
            return [
                'code' => 200,
                'success' => false,
                'no_package' => true,
                'message' => 'No Subscription and Topup Found.'
            ];
        }

        $currentDate = now();

        $packageEndDateCount = $packageTrx ? $currentDate->diffInDays(Carbon::parse($packageTrx->end_date), false) : -999;
        $topupEndDateCount   = $topup ? $currentDate->diffInDays(Carbon::parse($topup->end_date), false) : -999;

        $regularCount = 0;
        $topupCount   = 0;
        $totalCount = 0;
        $countExist = 0;

        if ($request->type == 'product') {
            $regularCount = $packageTrx ? $packageTrx->number_of_product : 0;
            $topupCount   = $topup ? $topup->number_of_product : 0;
            $countExist = Product::where('user_id', $id)
                ->where('source_type', 'product')->count();
        } else if ($request->type == 'service') {
            $regularCount = $packageTrx ? $packageTrx->number_of_service : 0;
            $topupCount   = $topup ? $topup->number_of_service : 0;
            $countExist = Product::where('user_id', $id)
                ->where('source_type', 'service')->count();
        } else {
            $regularCount = $packageTrx ? $packageTrx->number_of_staff : 0;
            $topupCount   = $topup ? $topup->number_of_staff : 0;
            $countExist = UserDetail::where('parent_id', $id)->count();
        }

        if ($packageEndDateCount > -1) {
            $totalCount += $regularCount;
        }

        if ($topupEndDateCount > -1) {
            $totalCount += $topupCount;
        }

        if ($packageEndDateCount <= -1 && $topupEndDateCount <= -1) {
            return [
                'code' => 422,
                'success' => true,
                'sub_end' => true,
                'message' => 'Subscription or Topup Ended.',
            ];
        }

        if ($countExist >= $totalCount) {
            return [
                'code' => 422,
                'success' => true,
                'sub_count_end' => true,
                'message' => 'Subscription or Topup limit Ended.',
            ];
        }

        return [
            'code' => 200,
            'success' => true,
            'message' => 'Successfully.',
        ];
    }

    public function translate(Request $request): array
    {
        try {
            $langCode = App::getLocale();
            if (request()->has('language_code') && !empty($request->language_code)) {
                $langCode = $request->language_code;
            }
            $languageId = getLanguageId($langCode);

            if (request()->has('language_id') && !empty($request->language_id) && $request->language_id != $languageId) {
                $lang = Language::select('code')->where('id', $request->language_id)->first();
                $langCode = $lang->code;
            }

            $translatedValues = [];

            $path = resource_path("lang/{$langCode}.json");
            if (file_exists($path)) {
                $translatedValues = json_decode(file_get_contents($path), true);
            }

            return [
                'code' => 200,
                'success' => true,
                'message' => __('Translated Successfully.'),
                'translated_values' => $translatedValues,
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'code' => 500,
                'message' => 'Translation failed: ' . $e->getMessage(),
            ];
        }
    }

    public function deleteImage(Request $request): array
    {
        $request->validate([
            'id' => 'required',
        ]);

        $deleteImage = Productmeta::find($request->id);

        incrementPageCacheVersion();

        if ($deleteImage) {

            $deleteImage->delete();

            return ['success' => true, 'message' => 'Image deleted successfully.'];
        }

        return ['success' => false, 'message' => 'Image not found.'];
    }

    public function checkCoupon(Request $request): array
    {
        $request->validate([
            'coupon_code'   => 'required|string',
            'service_id'    => 'required|integer',
            'category_id'   => 'required|integer',
            'subcategory_id' => 'required|integer',
        ]);

        $coupon = DB::table('coupons')
            ->whereRaw('BINARY `code` = ?', [$request->coupon_code])
            ->where('status', 1)
            ->first();

        $serviceId = $request->service_id;
        $categoryId = $request->category_id;
        $subcategoryId = $request->subcategory_id;

        $productType = $coupon->product_type ?? '';

        // Explode and convert to arrays
        $productIds = explode(',', $coupon->product_id ?? '');
        $categoryIds = explode(',', $coupon->category_id ?? '');
        $subcategoryIds = explode(',', $coupon->subcategory_id ?? '');

        // Check if any match
        $matched = in_array($serviceId, $productIds) ||
            in_array($categoryId, $categoryIds) ||
            in_array($subcategoryId, $subcategoryIds) ||
            $productType == 'all';

        if (!$matched) {
            return [
                'success' => false,
                'message' => 'This coupon is not valid.',
            ];
        }

        return [
            'success' => true,
            'message' => 'Coupon applied successfully!',
            'coupon' => [
                'coupon_id'   => $coupon->id,
                'coupon_code' => $coupon->code,
                'coupon_type' => $coupon->coupon_type,
                'coupon_value' => $coupon->coupon_value ?? 0,
            ]
        ];
    }

    /**
     * Resize an image and save it to a fixed folder using native GD.
     */
    private function resizeAndSaveImage(string $sourcePath, string $savePath, int $targetWidth, int $targetHeight)
    {
        // Make sure the 'fixed' directory exists
        Storage::disk('public')->makeDirectory('product_images/fixed');

        // Get original image info
        [$width, $height, $type] = getimagesize($sourcePath);

        // Create source image resource
        switch ($type) {
            case IMAGETYPE_JPEG:
                $srcImage = imagecreatefromjpeg($sourcePath);
                break;
            case IMAGETYPE_PNG:
                $srcImage = imagecreatefrompng($sourcePath);
                break;
            case IMAGETYPE_WEBP:
                $srcImage = imagecreatefromwebp($sourcePath);
                break;
            default:
                throw new \Exception('Unsupported image type.');
        }

        // Create new blank image for resizing
        $dstImage = imagecreatetruecolor($targetWidth, $targetHeight);

        // Preserve transparency for PNG and WEBP
        if (in_array($type, [IMAGETYPE_PNG, IMAGETYPE_WEBP])) {
            imagealphablending($dstImage, false);
            imagesavealpha($dstImage, true);
            $transparent = imagecolorallocatealpha($dstImage, 0, 0, 0, 127);
            imagefilledrectangle($dstImage, 0, 0, $targetWidth, $targetHeight, $transparent);
        }

        // Resize
        imagecopyresampled($dstImage, $srcImage, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);

        // Save the fixed image
        $absoluteSavePath = Storage::disk('public')->path($savePath);

        switch ($type) {
            case IMAGETYPE_JPEG:
                imagejpeg($dstImage, $absoluteSavePath, 90);
                break;
            case IMAGETYPE_PNG:
                imagepng($dstImage, $absoluteSavePath, 9);
                break;
            case IMAGETYPE_WEBP:
                imagewebp($dstImage, $absoluteSavePath, 90);
                break;
        }

        // Free memory
        imagedestroy($srcImage);
        imagedestroy($dstImage);
    }
}
