<?php

namespace Modules\Service\app\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Branches;
use App\Models\PackageTrx;
use App\Models\ProductVariation;
use App\Models\ServiceBranch;
use App\Models\ServiceStaff;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Service\app\Models\Productmeta;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Modules\Service\app\Models\AdditionalService as ModelsAdditionalService;
use Modules\Service\app\Models\Service;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Modules\Product\app\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;
use Modules\Categories\app\Models\Categories;
use Modules\Communication\app\Http\Controllers\EmailController;
use Modules\GlobalSetting\app\Models\Language;
use Modules\Communication\app\Http\Controllers\NotificationController;
use Modules\GlobalSetting\app\Models\Templates;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\Product\app\Models\Category;
use Modules\Service\app\Repositories\Contracts\ServiceRepositoryInterface;
use Illuminate\Http\UploadedFile;
use Modules\Shops\app\Models\Shops;

class ServiceController extends Controller
{
    protected ServiceRepositoryInterface $serviceRepository;

    public function __construct(ServiceRepositoryInterface $serviceRepository)
    {
        $this->serviceRepository = $serviceRepository;
    }

    /**
     * Display a listing of the resource.
     */
    public function setdefault(Request $request)
    {
        $response = $this->serviceRepository->setDefault($request);
        return response()->json($response, $response['code']);
    }

    public function delete(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->delete($request);
        return response()->json($response, $response['code']);
    }

    public function index(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->index($request);
        return response()->json($response, $response['code']);
    }

    public function productListApi(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->productListApi($request);
        return response()->json($response, $response['code']);
    }

    public function shopListApi(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->shopListApi($request);
        return response()->json($response, $response['code']);
    }

    public function serviceDetails(Request $request, string $slug): JsonResponse
    {
        $response = $this->serviceRepository->serviceDetails($request, $slug);
        return response()->json($response, $response['code']);
    }

    public function productDetails(Request $request, string $slug): JsonResponse
    {
        $response = $this->serviceRepository->productDetails($request, $slug);
        return response()->json($response, $response['code']);
    }

    public function galleryandvariatonApi(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->galleryandvariatonApi($request);

        return response()->json(
            $response,
            $response['code'] ?? 200 // ✅ Default to 200 if not set
        );
    }

    public function store(Request $request)
    {
        return $this->serviceRepository->store($request);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request)
    {
        return $this->serviceRepository->update($request);
    }

    /**+
     *
     * Remove the specified resource from storage.
     */
    public function providerServiceIndex(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->providerServiceIndex($request);
        return response()->json($response, $response['code']);
    }

    public function providerService(): View
    {
        $userId = Auth::id();
        return view("provider.providerService", compact('userId'));
    }

    public function providerProduct(): View
    {
        $userId = Auth::id();
        return view("provider.product.index", compact('userId'));
    }

    public function providerAddServiceIndex(): View
    {
        $serviceSlot = DB::table('general_settings')->where('key', 'service_slot')->value('value');
        $servicePackage = DB::table('general_settings')->where('key', 'service_package')->value('value');

        $userId = Auth::id();
        $userLangId = User::where('id', $userId)->value('user_language_id');

        $chatstatus = GlobalSetting::where('group_id', 4)
            ->where('key', 'chatgpt_status')
            ->pluck('value')
            ->first();
        $shopList = Shops::where("created_by", Auth::id())->get();


        return view("provider.providerAddService", [
            'show_slot' => $serviceSlot == 1,
            'show_package' => $servicePackage == 1,
            'userLangId' => $userLangId,
            'chat_status' => $chatstatus,
            'shopList' => $shopList
        ]);
    }

    public function providerEditService($id): View
    {
        // 1. Find the service and its relationships
        $service = Service::with(['productImages', 'variations'])->findOrFail($id);

        // Security check
        if ($service->user_id !== Auth::id()) {
            abort(403, 'Unauthorized action.');
        }

        // 2. Fetch data for dropdowns
        $userLangId = Auth::user()->user_language_id;
        $categoriesLang = Categories::select('id', 'name')->where('status', 1)->where('language_id', $userLangId)->where('parent_id', 0)->get();
        $subcategories = Categories::select('id', 'name')->where('status', 1)->where('language_id', $userLangId)->where('parent_id', $service->source_category)->get();
        $shopList = Shops::where('created_by', Auth::id())->where('status', 1)->get();

        // 3. Fetch and process availability from the product_meta table
        $slotMetas = Productmeta::where('product_id', $id)->where('source_key', 'like', '%_slot_%')->get();
        $availability = [];
        $daysOfWeek = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];

        foreach ($slotMetas as $meta) {
            $keyParts = explode('_', $meta->source_key);
            $day = $keyParts[0] ?? null;
            if ($day && in_array($day, $daysOfWeek)) {
                $timeParts = explode(' - ', $meta->source_Values);
                $startTime = trim($timeParts[0] ?? '');
                $endTime = trim($timeParts[1] ?? '');

                if ($startTime && $endTime) {
                    $slot = new \stdClass();
                    $slot->start_time = $startTime;
                    $slot->end_time = $endTime;

                    $dayCapitalized = ucfirst($day);
                    if (!isset($availability[$dayCapitalized])) {
                        $availability[$dayCapitalized] = [];
                    }
                    $availability[$dayCapitalized][] = $slot;
                }
            }
        }

        // 4. Fetch the video link
        $videoMeta = Productmeta::where('product_id', $id)->where('source_key', 'service_video')->first();

        // 5. Pass all data to the view
        return view("provider.providerEditService", [
            'service' => $service,
            'userLangId' => $userLangId,
            'categoriesLang' => $categoriesLang,
            'subcategories' => $subcategories,
            'shopList' => $shopList,
            'availability' => $availability,
            'video_link' => $videoMeta->source_Values ?? '',
        ]);
    }

    public function getDetails(Request $request, string $slug): JsonResponse
    {
        $response = $this->serviceRepository->getDetails($request, $slug);
        return response()->json($response, $response['code']);
    }

    public function providerServiceStore(Request $request): JsonResponse
    {
        return $this->serviceRepository->providerServiceStore($request);
    }

    public function verifyService(Request $request)
    {
        $response = $this->serviceRepository->verifyService($request);
        return response()->json($response, $response['code']);
    }

    public function providerServiceUpdate(Request $request): JsonResponse
    {
        return $this->serviceRepository->providerServiceUpdate($request);
    }

    public function deleteServiceImage(string $id): JsonResponse
    {
        $response = $this->serviceRepository->deleteServiceImage($id);
        return response()->json($response, $response['code']);
    }

    public function deleteSlot(string $id): JsonResponse
    {
        $response = $this->serviceRepository->deleteSlot($id);
        return response()->json($response, $response['code']);
    }

    public function deleteAdditionalServices(string $id): JsonResponse
    {
        $response = $this->serviceRepository->deleteAdditionalServices($id);
        return response()->json($response, $response['code']);
    }

    public function deleteServices(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->deleteServices($request);
        return response()->json($response, $response['code']);
    }

    public function status(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->status($request);
        return response()->json($response, $response['code']);
    }

    public function checkUnique(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->checkUnique($request);
        return response()->json($response);
    }

    public function checkEditUnique(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->checkEditUnique($request);
        return response()->json($response);
    }

    public function providerSub(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->providerSub($request);
        return response()->json($response, $response['code']);
    }

    public function providerSubApi(Request $request): JsonResponse
    {
        $response = $this->serviceRepository->providerSubApi($request);
        return response()->json($response, $response['code']);
    }

    public function translate(Request $request)
    {
        $response = $this->serviceRepository->translate($request);
        return response()->json($response, $response['code']);
    }

    public function deleteImage(Request $request)
    {
        $response = $this->serviceRepository->deleteImage($request);
        return response()->json($response);
    }

    public function checkCoupon(Request $request)
    {
        $response = $this->serviceRepository->checkCoupon($request);
        return response()->json($response);
    }

    public function productIndex()
    {
        $serviceSlot = DB::table('general_settings')->where('key', 'service_slot')->value('value');
        $servicePackage = DB::table('general_settings')->where('key', 'service_package')->value('value');

        $userId = Auth::id();
        $userLangId = User::where('id', $userId)->value('user_language_id');

        $chatstatus = GlobalSetting::where('group_id', 4)
            ->where('key', 'chatgpt_status')
            ->pluck('value')
            ->first();

        return view("provider.product.createNew", [
            'show_slot' => $serviceSlot == 1,
            'show_package' => $servicePackage == 1,
            'userLangId' => $userLangId,
            'chat_status' => $chatstatus
        ]);
    }

    public function productStore(Request $request)
    {

        $userId = Auth::id();

        $data = [
            'user_id'            => $userId,
            'shop_id'         => $request->shop_id,
            'source_name'        => $request->product_name,
            'slug'               => $request->slug,
            'source_code'        => $request->product_code,
            'source_type'        => 'product',
            'source_category'    => $request->category_id,
            'source_subcategory' => $request->sub_category_id,
            'source_description' => $request->description,
            'shot_description' => $request->shot_description,
            'international_shipping' => $request->international_shipping,
            'domestic_shipping' => $request->domestic_shipping,
            'seo_title'          => $request->meta_title,
            'seo_description'    => $request->meta_description,
            'source_price'         => $request->price,
            'source_stock'         => $request->stock,
            'featured'           => $request->feature ?? 1,
            'popular'            => 1,
            'language_id'        => 1,
            'parent_id'          => 0,
            'created_by'         => $userId,
        ];

        $product = Service::create($data);

        $variations = $request->input('variations', []);

        if ($request->hasFile('product_images') && $request->file('product_images')) {
            $images = $request->file('product_images');
            if (!$images instanceof UploadedFile) {
                foreach ($images as $image) {
                    if ($image instanceof UploadedFile) {
                        // Store original image
                        $imagePath = $image->store('product_images', 'public');

                        // ✅ Also store fixed version (208x429)
                        $this->storeFixedImage($image, 'product_images/fixed', 260, 170);
                        $this->storeFixedImage($image, 'product_images/fixed_md', 266, 293);
                        $this->storeFixedImage($image, 'product_images/fixed_xl', 355, 232);

                        // Store original path in DB
                        $imageData = [
                            'product_id' => $product->id,
                            'source_key' => 'product_image',
                            'source_Values' => $imagePath
                        ];

                        Productmeta::create($imageData);
                    }
                }
            }
        }

        if (!empty($variations)) {
            foreach ($variations as $index => $variationData) {
                $imagePaths = [];

                if ($request->hasFile("variations.{$index}.images")) {
                    foreach ($request->file("variations.{$index}.images") as $imageFile) {
                        // Store original variation image
                        $path = $imageFile->store('products/variations', 'public');
                        $imagePaths[] = $path;

                        $this->storeFixedImage($imageFile, 'products/variations/fixed', 24, 50);
                    }
                }

                $sizePriceStockData = [];
                if (isset($variationData['sizes'])) {
                    foreach ($variationData['sizes'] as $key => $size) {
                        $sizePriceStockData[] = [
                            'size'  => $size,
                            'price' => $variationData['prices'][$key] ?? 0,
                            'stock' => $variationData['stocks'][$key] ?? 0,
                        ];
                    }
                }

                ProductVariation::create([
                    'product_id' => $product->id,
                    'color'      => $variationData['color'],
                    'images'     => $imagePaths, // JSON
                    'data'       => $sizePriceStockData, // JSON
                ]);
            }
        }

        if ($request->filled('specification_label') && $request->filled('specification_description')) {
            $labels = $request->input('specification_label');
            $descriptions = $request->input('specification_description');

            $specifications = [];

            foreach ($labels as $index => $label) {
                if (!empty($label) && !empty($descriptions[$index])) {
                    $specifications[] = [
                        'label'       => $label,
                        'description' => $descriptions[$index],
                    ];
                }
            }

            Productmeta::where('product_id', $product->id)
                ->where('source_key', 'specification')
                ->delete();

            if (!empty($specifications)) {
                Productmeta::create([
                    'product_id'    => $product->id,
                    'source_key'    => 'specification',
                    'source_Values' => json_encode($specifications),
                ]);
            }
        }

        incrementPageCacheVersion();

        return response()->json([
            'message'       => 'Product and its variations added successfully!',
            'redirectUrl'   => route('provider.product')
        ]);
    }

    public function edit($id)
    {
        // Eager load all required relationships correctly
        $product = Service::with([
            'variations',
            'productImages',
            'specificationMeta' // CORRECTED: Eager load the new relationship
        ])->findOrFail($id);

        // Fetch the list of shops
        $shopList = Shops::where("created_by", Auth::id())->get();

        return view('provider.product.edit', compact('product', 'shopList'));
    }

    public function productUpdate(Request $request, $id)
    {
        // --- 1. Validation ---
        $validator = Validator::make($request->all(), [
            'product_name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:products,slug,' . $id,
            'product_code' => 'required|string|max:255',
            'shop_id' => 'required|exists:shops,id',
            'category_id' => 'required|exists:categories,id',
            'description' => 'required|string',
            'price' => 'nullable|numeric',
            'stock' => 'nullable|integer',
            'product_type' => 'required|in:single,variable',
            'variations' => 'nullable|array',
            'variations.*.color' => 'required|string',
            'variations.*.sizes' => 'required|array',
            'variations.*.prices' => 'required|array',
            'variations.*.stocks' => 'required|array',
            'variations.*.images.*' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:2048',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        // --- 2. Find the Product and Update its Main Details ---
        $product = Service::with('variations')->findOrFail($id);

        $product->update([
            'shop_id'            => $request->shop_id,
            'source_name'        => $request->product_name,
            'slug'               => $request->slug,
            'source_code'        => $request->product_code,
            'source_category'    => $request->category_id,
            'source_subcategory' => $request->sub_category_id,
            'source_description' => $request->description,
            'shot_description'   => $request->shot_description,
            'international_shipping' => $request->international_shipping,
            'domestic_shipping'      => $request->domestic_shipping,
            'seo_title'          => $request->meta_title,
            'seo_description'    => $request->meta_description,
            'featured'           => $request->feature ?? 1,
            'product_type'       => $request->product_type,
        ]);
        if ($request->hasFile('product_images')) {
            foreach ($request->file('product_images') as $image) {
                if ($image instanceof UploadedFile) {

                    // Store original
                    $imagePath = $image->store('product_images', 'public');

                    $this->storeFixedImage($image, 'product_images/fixed', 260, 170);
                    $this->storeFixedImage($image, 'product_images/fixed_md', 266, 293);
                    $this->storeFixedImage($image, 'product_images/fixed_xl', 355, 232);

                    // Save original path in DB
                    Productmeta::create([
                        'product_id'    => $product->id,
                        'source_key'    => 'product_image',
                        'source_Values' => $imagePath
                    ]);
                }
            }
        }

        // --- Handle Single vs Variable Logic ---
        if ($request->product_type === 'variable') {
            // Reset product-level price & stock
            $product->update([
                'source_price' => null,
                'source_stock' => null,
            ]);

            // ✅ Process variations normally
            $submittedVariations = $request->input('variations', []);
            $processedVariationIds = [];

            if (!empty($submittedVariations)) {
                foreach ($submittedVariations as $varIndexKey => $variationData) {
                    $sizePriceStockData = [];
                    if (isset($variationData['sizes'])) {
                        foreach ($variationData['sizes'] as $key => $size) {
                            if (empty($size)) continue;
                            $sizePriceStockData[] = [
                                'size'  => $size,
                                'price' => $variationData['prices'][$key] ?? 0,
                                'stock' => $variationData['stocks'][$key] ?? 0,
                            ];
                        }
                    }

                    $variationId = $variationData['id'] ?? null;

                    if ($variationId) {
                        $variation = ProductVariation::find($variationId);
                        if (!$variation) continue;

                        $currentImages = $variation->images ?? [];

                        if (!empty($variationData['removed_images'])) {
                            $imagesToRemove = explode(',', $variationData['removed_images']);
                            foreach ($imagesToRemove as $imgUrl) {
                                Storage::disk('public')->delete($imgUrl);
                                $fixedPath = str_replace('products/variations', 'products/variations/fixed', $imgUrl);
                                Storage::disk('public')->delete($fixedPath);
                            }
                            $currentImages = array_diff($currentImages, $imagesToRemove);
                        }

                        $newImagePaths = [];
                        if ($request->hasFile("variations.{$varIndexKey}.images")) {
                            foreach ($request->file("variations.{$varIndexKey}.images") as $imageFile) {
                                $storedPath = $imageFile->store('products/variations', 'public');
                                $newImagePaths[] = $storedPath;
                                $this->storeFixedImage($imageFile, 'products/variations/fixed', 24, 50);
                            }
                        }

                        $variation->update([
                            'color'  => $variationData['color'],
                            'images' => array_merge(array_values($currentImages), $newImagePaths),
                            'data'   => $sizePriceStockData,
                        ]);

                        $processedVariationIds[] = $variation->id;
                    } else {
                        $imagePaths = [];
                        if ($request->hasFile("variations.{$varIndexKey}.images")) {
                            foreach ($request->file("variations.{$varIndexKey}.images") as $imageFile) {
                                $storedPath = $imageFile->store('products/variations', 'public');
                                $imagePaths[] = $storedPath;
                                $this->storeFixedImage($imageFile, 'products/variations/fixed', 24, 50);
                            }
                        }

                        $newVariation = ProductVariation::create([
                            'product_id' => $product->id,
                            'color'      => $variationData['color'],
                            'images'     => $imagePaths,
                            'data'       => $sizePriceStockData,
                        ]);

                        $processedVariationIds[] = $newVariation->id;
                    }
                }
            }

            // Delete removed variations
            $originalVariationIds = $product->variations->pluck('id');
            $idsToDelete = $originalVariationIds->diff($processedVariationIds);
            if ($idsToDelete->isNotEmpty()) {
                $variationsToDelete = ProductVariation::whereIn('id', $idsToDelete)->get();
                foreach ($variationsToDelete as $varToDelete) {
                    if (is_array($varToDelete->images)) {
                        foreach ($varToDelete->images as $imagePath) {
                            Storage::disk('public')->delete($imagePath);
                            $fixedPath = str_replace('products/variations', 'products/variations/fixed', $imagePath);
                            Storage::disk('public')->delete($fixedPath);
                        }
                    }
                }
                ProductVariation::destroy($idsToDelete);
            }
        } else {
            // --- product_type = single ---
            // Save product-level price & stock
            $product->update([
                'source_price' => $request->price,
                'source_stock' => $request->stock,
            ]);

            // Delete all variations if exist
            if ($product->variations->count()) {
                foreach ($product->variations as $variation) {
                    if (is_array($variation->images)) {
                        foreach ($variation->images as $imagePath) {
                            Storage::disk('public')->delete($imagePath);
                            $fixedPath = str_replace('products/variations', 'products/variations/fixed', $imagePath);
                            Storage::disk('public')->delete($fixedPath);
                        }
                    }
                }
                ProductVariation::where('product_id', $product->id)->delete();
            }
        }

        // --- Handle Gallery Images (same as your code) ---
        $deletedGalleryImageIds = json_decode($request->input('deleted_gallery_image_ids', '[]'), true);
        if (!empty($deletedGalleryImageIds)) {
            foreach ($deletedGalleryImageIds as $imageId) {
                $image = Productmeta::find($imageId);
                if ($image && !empty($image->source_Values)) {
                    $originalPath = $image->source_Values;
                    Storage::disk('public')->delete($originalPath);
                    Storage::disk('public')->delete(str_replace('products/images', 'products/images/fixed', $originalPath));
                    Storage::disk('public')->delete(str_replace('products/images', 'products/images/fixed_xl', $originalPath));
                    Storage::disk('public')->delete(str_replace('products/images', 'products/images/fixed_md', $originalPath));
                    $image->delete();
                }
            }
        }

        // --- Handle Specifications (same as your code) ---
        if ($request->filled('specification_label') && $request->filled('specification_description')) {
            $labels = $request->input('specification_label');
            $descriptions = $request->input('specification_description');
            $specifications = [];

            foreach ($labels as $index => $label) {
                if (!empty($label) && !empty($descriptions[$index])) {
                    $specifications[] = [
                        'label'       => $label,
                        'description' => $descriptions[$index],
                    ];
                }
            }

            Productmeta::where('product_id', $product->id)
                ->where('source_key', 'specification')
                ->delete();

            if (!empty($specifications)) {
                Productmeta::create([
                    'product_id'    => $product->id,
                    'source_key'    => 'specification',
                    'source_Values' => json_encode($specifications),
                ]);
            }
        }

        incrementPageCacheVersion();

        return response()->json([
            'message'     => 'Product updated successfully!',
            'redirectUrl' => route('provider.product', $product->id)
        ]);
    }
    private function storeFixedImage(UploadedFile $image, string $folder, int $width, int $height): void
    {
        $sourcePath = $image->getRealPath();

        // --- START OF FIX ---
        // 1. ROBUST CHECK: Get the actual image type from its content, not its extension.
        $imageInfo = @getimagesize($sourcePath);
        if ($imageInfo === false) {
            // Not a valid image file that getimagesize can read.
            return;
        }

        $mimeType = $imageInfo['mime'];
        $extension = image_type_to_extension($imageInfo[2], true); // e.g., '.jpeg', '.png'
        $extension = ltrim($extension, '.'); // Remove leading dot

        // Use the original hash-based filename
        $fileName = $image->hashName();

        // 2. Create image resource based on the DETECTED file type
        $src = null;
        switch ($mimeType) {
            case 'image/jpeg':
                $src = imagecreatefromjpeg($sourcePath);
                break;
            case 'image/png':
                $src = imagecreatefrompng($sourcePath);
                break;
            case 'image/gif':
                $src = imagecreatefromgif($sourcePath);
                break;
            case 'image/webp':
                // Check if webp is supported on the server
                if (function_exists('imagecreatefromwebp')) {
                    $src = imagecreatefromwebp($sourcePath);
                }
                break;
            default:
                // Unsupported file type for processing
                return;
        }

        if (!$src) {
            // Failed to create image resource, maybe a corrupt file
            return;
        }
        // --- END OF FIX ---

        // Get original dimensions
        $origWidth = imagesx($src);
        $origHeight = imagesy($src);

        // Create new blank canvas
        $dst = imagecreatetruecolor($width, $height);

        // Preserve transparency for PNG, GIF, and WebP
        if (in_array($mimeType, ['image/png', 'image/gif', 'image/webp'])) {
            imagecolortransparent($dst, imagecolorallocatealpha($dst, 0, 0, 0, 127));
            imagealphablending($dst, false);
            imagesavealpha($dst, true);
        }

        // Resize image
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $width, $height, $origWidth, $origHeight);

        // Ensure folder exists
        Storage::disk('public')->makeDirectory($folder);

        // Build final storage path
        $finalPath = Storage::disk('public')->path("{$folder}/{$fileName}");

        // 3. Save the image using the correct function for its type
        switch ($mimeType) {
            case 'image/jpeg':
                imagejpeg($dst, $finalPath, 90);
                break;
            case 'image/png':
                imagepng($dst, $finalPath, 8); // Compression level 0-9
                break;
            case 'image/gif':
                imagegif($dst, $finalPath);
                break;
            case 'image/webp':
                imagewebp($dst, $finalPath, 90);
                break;
        }

        imagedestroy($src);
        imagedestroy($dst);
    }
}
