<?php

namespace Modules\RolesPermissions\app\Repositories\Eloquent;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;
use Modules\RolesPermissions\app\Models\Permission;
use Modules\RolesPermissions\app\Models\Role;
use Modules\RolesPermissions\app\Repositories\Contracts\RolesPermissionsRepositoryInterface;

class RolesPermissionsRepository implements RolesPermissionsRepositoryInterface
{
    private const CACHE_TTL_MINUTES = 10;

    public function index(Request $request): array
    {
        try {
            $userId = Auth::id() ?? $request->user_id;
            $orderBy = $this->normaliseOrderDirection($request->order_by ?? 'asc');
            $status = $request->status ?? '';
            $onlyActive = $request->has('status') && (string) $status === '1';

            $data = Cache::remember(
                $this->getRolesCacheKey($userId, $orderBy, $onlyActive),
                now()->addMinutes(self::CACHE_TTL_MINUTES),
                function () use ($userId, $orderBy, $onlyActive) {
                    $query = Role::select('id', 'role_name', 'status', 'created_by')
                        ->where('created_by', $userId)
                        ->orderBy('id', $orderBy);

                    if ($onlyActive) {
                        $query->where('status', 1);
                    }

                    return $query->get();
                }
            );

            return [
                'code' => '200',
                'message' => __('Role details retrieved successfully.'),
                'data' => $data,
            ];

        } catch (\Exception $e) {
            return [
                'code' => '500',
                'message' => __('An error occurred while retrieving Role.'),
                'error' => $e->getMessage(),
            ];
        }
    }

    public function store(Request $request): array
    {
        $createdBy = Auth::id() ?? $request->created_by;
        $id = $request->id ?? '';

        $data = [
            'role_name' => $request->role_name,
            'created_by' => $createdBy
        ];

        $successMessage = empty($id) ? __('Role created successfully.') : __('Role updated successfully.');
        $errorMessage = empty($id) ? __('Error! while creating role.') : __('Error! while updating role');

        try {
            Role::updateOrCreate(['id' => $id], $data);

            $this->flushRoleCacheForUser($createdBy);

            return [
                'code' => 200,
                'message' => $successMessage,
            ];

        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => $errorMessage,
            ];
        }
    }

    public function destroy(Request $request): array
    {
        $id = $request->id;
        try {
            $role = Role::find($id);

            if (empty($role)) {
                return [
                    'code' => 404,
                    'success' => false,
                    'message' => __('Role not found.')
                ];
            }

            $createdBy = $role->created_by;

            Role::where('id', $id)->delete();

            $this->flushRoleCacheForUser($createdBy);
            $this->forgetPermissionCache($id);

            return [
                'code' => 200,
                'success' => true,
                'message' => __('Role deleted successfully.')
            ];

        } catch (\Exception $e) {
            return [
                'code' => 500,
                'success' => false,
                'message' => __('Error! while deleting role.')
            ];
        }
    }

    public function roleStatusChange(Request $request): array
    {
        $id = $request->id;
        $status = $request->status;
        try {
            $role = Role::find($id);

            if (empty($role)) {
                return [
                    'code' => 404,
                    'success' => false,
                    'message' => __('Role not found.')
                ];
            }

            Role::where('id', $id)->update([
                'status' => $status
            ]);

            $this->flushRoleCacheForUser($role->created_by);

            return [
                'code' => 200,
                'message' => __('Role status updated successfully.')
            ];

        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => __('Error! while changing role status')
            ];
        }
    }

    public function permissionList(Request $request): array
    {
        $id = $request->id ?? $request->role_id;
        try {
            $orderBy = $this->normaliseOrderDirection($request->order_by ?? 'asc');
            $data = $this->getPermissionsForRole($id, $orderBy)->map(function ($permission) {
                return $permission;
            });

            $authId = Auth::id() ?? $request->user_id;
            $user = User::select('user_type')->where('id', $authId)->first();

            if (!$user) {
                return [
                    'code' => '404',
                    'message' => __('User not found.'),
                ];
            }

            if ($data->isEmpty()) {
                $modules = $this->getModulesForUserType($user->user_type);

                if ($modules->isNotEmpty()) {
                    foreach ($modules as $module) {
                        $data->push(
                            ["id" => '',
                            "role_id" => $id,
                            "module" => $module->name,
                            "create" => 0,
                            "view" => 0,
                            "edit" => 0,
                            "delete" => 0,
                            "allow_all" => 0]);
                    }
                }

            } else {
                /** @var \Modules\RolesPermissions\app\Models\Role $role */

                $role = Role::with('permissions')->findOrFail($id);
                $permissions = $role->permissions->pluck('module')->toArray();

                $modules = $this->getModulesForUserType($user->user_type)
                    ->whereNotIn('name', $permissions);

                if ($modules->isNotEmpty()) {
                    foreach ($modules as $module) {
                        $data->push(
                            ["id" => '',
                            "role_id" => $id,
                            "module" => $module->name,
                            "create" => 0,
                            "view" => 0,
                            "edit" => 0,
                            "delete" => 0,
                            "allow_all" => 0]);
                    }
                }
            }

            return [
                'code' => '200',
                'message' => __('Permission details retrieved successfully.'),
                'data' => $data,
            ];

        } catch (\Exception $e) {
            return [
                'code' => '500',
                'message' => __('An error occurred while retrieving Permissions.'),
                'error' => $e->getMessage(),
            ];
        }
    }

    public function permissionUpdate(Request $request): array
    {
        $roleId = $request->role_id;
        $permissions = $request->input('permissions', []);

        try {

            foreach ($permissions as $permission) {
                Permission::updateOrCreate(['id' => $permission['id'], 'role_id' => $roleId ],
                    [
                        'module' => $permission['module'],
                        'create' => $permission['create'],
                        'view' => $permission['view'],
                        'edit' => $permission['edit'],
                        'delete' => $permission['delete'],
                        'allow_all' => $permission['allow_all']
                    ]);
            }

            $this->forgetPermissionCache($roleId);

            return [
                'code' => 200,
                'message' => __('Permission updated successfully.'),
            ];

        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => __('An error occurred while updating Permissions.'),
                'error' => $e->getMessage(),
            ];
        }
    }

    public function checkUniqueRoleName(Request $request): bool
    {
        $id = $request->input('id');
        $userId = $request->input('user_id');

        $validation = Validator::make($request->all(), [
            'role_name' => [
                Rule::unique('roles')->whereNull('deleted_at')->ignore($id)->where('created_by', $userId)
            ],
        ]);

        if ($validation->fails()) {
            return false;
        }

        return true;
    }

    private function normaliseOrderDirection(?string $direction): string
    {
        $direction = strtolower($direction ?? 'asc');

        return in_array($direction, ['asc', 'desc'], true) ? $direction : 'asc';
    }

    private function getRolesCacheKey(int $userId, string $orderBy, bool $onlyActive): string
    {
        return sprintf(
            'roles_permissions.roles.%d.%s.%s',
            $userId,
            $orderBy,
            $onlyActive ? 'active' : 'all'
        );
    }

    private function flushRoleCacheForUser(?int $userId): void
    {
        if (empty($userId)) {
            return;
        }

        foreach (['asc', 'desc'] as $direction) {
            Cache::forget($this->getRolesCacheKey($userId, $direction, true));
            Cache::forget($this->getRolesCacheKey($userId, $direction, false));
        }
    }

    private function getPermissionsCacheKey(int $roleId, string $orderBy): string
    {
        return sprintf('roles_permissions.permissions.%d.%s', $roleId, $orderBy);
    }

    private function forgetPermissionCache(int $roleId): void
    {
        foreach (['asc', 'desc'] as $direction) {
            Cache::forget($this->getPermissionsCacheKey($roleId, $direction));
        }
    }

    private function getModulesCacheKey(string $userType): string
    {
        return sprintf('roles_permissions.modules.%s', $userType);
    }

    private function getModulesForUserType(string $userType)
    {
        return Cache::remember(
            $this->getModulesCacheKey($userType),
            now()->addMinutes(self::CACHE_TTL_MINUTES),
            function () use ($userType) {
                return DB::table('modules')
                    ->where('user_type', $userType)
                    ->whereNull('deleted_at')
                    ->get();
            }
        );
    }

    private function getPermissionsForRole(int $roleId, string $orderBy)
    {
        return Cache::remember(
            $this->getPermissionsCacheKey($roleId, $orderBy),
            now()->addMinutes(self::CACHE_TTL_MINUTES),
            function () use ($roleId, $orderBy) {
                return Permission::where('role_id', $roleId)
                    ->orderBy('id', $orderBy)
                    ->get();
            }
        );
    }
}
