<?php

namespace Modules\Product\app\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Models\User;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Modules\Product\app\Models\Product;
use Modules\Shops\app\Models\Shops;

class Rating extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'ratings';

    protected $fillable = ['id', 'user_id', 'product_id', 'parent_id', 'review', 'rating', 'review_date', 'type'];

    /**
     * The user (customer) who wrote the rating.
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * The product this rating is for.
     */
    public function product()
    {
        return $this->belongsTo(Product::class, 'product_id');
    }

    /**
     * The vendor's reply to this rating.
     * A rating can have only one direct reply from the vendor.
     */
    public function reply()
    {
        return $this->hasMany(Rating::class, 'parent_id', 'id');
    }

    public static function getProductRatingDetails($productId)
    {
        $averageRating = self::where('product_id', $productId)->avg('rating');
        $ratingCount = self::where('product_id', $productId)->count();

        return [
            'average_rating' => $averageRating ? round($averageRating, 1) : 0,
            'rating_count' => $ratingCount
        ];
    }

    /**
     * @return HasMany<Rating, Rating>
     */
    public function replies(): HasMany
    {
        /** @var HasMany<Rating, Review> */
        return $this->hasMany(Rating::class, 'parent_id')->orderBy('created_at', 'desc');
    }
}
