<?php

namespace Modules\Page\app\Repositories\Eloquent;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Modules\GlobalSetting\app\Models\Language;
use Modules\Page\app\Models\Footer;
use Illuminate\Support\Facades\Cache;
use Modules\Page\app\Models\Page;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\GlobalSetting\app\Models\GeneralSetting;
use Illuminate\Support\Facades\Cookie;
use Modules\Blogs\app\Http\Controllers\BlogsController;
use Modules\MenuBuilder\app\Models\Menu;
use Modules\Page\app\Repositories\Contracts\PageRepositoryInterface;
use Illuminate\View\View;
use Modules\GlobalSetting\app\Models\Currency;
use Modules\Page\app\Models\Section;
use Illuminate\Support\Facades\Validator;
use App\CentralLogics\Helpers;
use App\Models\BookingProduct;
use Illuminate\Support\Str;
use Modules\Categories\app\Models\Categories;
use Modules\Product\app\Models\Product;
use Modules\Shops\app\Models\Shops;

class PageRepository implements PageRepositoryInterface
{
    public function index(Request $request): JsonResponse
    {
        try {

            $langCode = App::getLocale();
            if (request()->has('language_code') && !empty($request->language_code)) {
                $langCode = $request->language_code;
            }
            $language = Language::where('code', $langCode)->first();
            $languageId = $language->id;

            if (request()->has('language_id') && !empty($request->language_id) && $request->language_id != $language->id) {
                $languageId = $request->language_id;
            }

            $data = Footer::select('id', 'footer_content', 'status')->where(['language_id' => $languageId])->first();

            if ($data && !empty($data->footer_content)) {
                $data->footer_content = json_decode($data->footer_content);
            }

            return response()->json([
                'code' => 200,
                'message' => __('Footer retrieved successfully.'),
                'data' => $data
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'code' => 500,
                'message' => __('Error! while retrieving footers'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        $sections = $request->input('sections');
        $id = $request->id ?? '';


        try {

            $langCode = App::getLocale();
            $language = Language::where('code', $langCode)->first();
            $languageId = $language->id;

            if (request()->has('language_id') && !empty($request->language_id)) {
                $languageId = $request->language_id;
                $languageData = Language::where('id', $languageId)->first();
                $langCode = $languageData->code;
            }

            $formattedSections = [];
            if (!empty($sections)) {
                foreach ($sections as $section) {
                    $formattedSections[] = [
                        'title' => $section['title'],
                        'footer_content' => $section['footer_content'],
                        'status' => isset($section['status']) ? 1 : 0,
                    ];
                }
            }

            $jsonSections = json_encode($formattedSections);
            Cache::forget('footerList');

            Footer::updateOrCreate(
                ['id' => $id, 'language_id' => $languageId],
                [
                    'footer_content' => $jsonSections,
                    'status' => isset($request->status) ? 1 : 0,
                ]
            );

            $this->incrementPageCacheVersion();

            return response()->json([
                'code' => 200,
                'message' => __('footer_save_success', [], $langCode),
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'code' => 500,
                'message' => __('footer_save_error', [], $langCode),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function getFooterDetails(): JsonResponse
    {
        try {
            $data = Footer::select('footer_content')->where('status', 1)->latest('id')->first();

            $filteredContent = [];

            if ($data && !empty($data->footer_content)) {
                $footerContent = json_decode($data->footer_content, true);

                if (is_array($footerContent)) {
                    $filteredContent = collect($footerContent)->filter(function ($item) {
                        return ($item['status'] ?? 0) == 1;
                    })->values()->toArray();
                }
            }

            return response()->json([
                'code' => 200,
                'message' => __('Footer retrieved successfully.'),
                'data' => $filteredContent
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'code' => 500,
                'message' => __('Error! while getting footers'),
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function pageBuilderApi(Request $request): JsonResponse | view
    {
        $slug = trim($request->path());
        $authUserId = Auth::id() ?? null;
        $isMobileRequest = request()->has('is_mobile') && request()->get('is_mobile') === "yes";
        $layout = 'frontend.theme_1.app';

        if ($slug === 'blogs') {
            return app(BlogsController::class)->blogList($request);
        }
        session(['link' => url()->current()]);

        $language_id = $isMobileRequest ? $request->language_id : getLanguageId(app()->getLocale());

        if (empty($slug) || $slug === '/') {
            $slug = 'home-screen-one';
        }

        $cacheKey = $this->buildPageCacheKey($slug, $language_id, $authUserId, $isMobileRequest, $request->page_name);
        $ttlMinutes = config('cache.page_builder_ttl', 30);

        if ($cachedResponse = Cache::get($cacheKey)) {
            $cachedResult = $this->dispatchCachedPageResponse($cachedResponse);
            if ($cachedResult !== null) {
                return $cachedResult;
            }
        }

        if ($isMobileRequest) {
            $page = Page::where('slug', $request->page_name)->first();
        }
        if (!$isMobileRequest) {
            $page = Page::where('slug', $slug)->where('language_id', $language_id)->first();

            if (!$page) {
                $getPage = Page::select('id')->where('slug', $slug)->first();
                if (!$getPage) {
                    abort(404);
                }
                $page = Page::where('parent_id', $getPage->id)->where('language_id', $language_id)->first();
            }

            if (!$page) {
                abort(404);
            }

            if (!$page) {
                abort(404);
            }
        }

        $pageContentSections = json_decode($page->page_content, true) ?? [];
        $currencySymbol = getDefaultCurrencySymbol();

        if (empty($pageContentSections) || !collect((array)$pageContentSections)->contains(fn($section) => $section['status'] == 1)) {
            $pageContentSections = [];
        } else {
            foreach ($pageContentSections as &$section) {

                // Banner
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[banner_one') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order = $matches[3] ?? 'asc';

                        $banners = DB::table('sections')
                            ->select('id', 'datas')
                            ->where('name', 'Banner One')
                            ->get();

                        // Split last word dynamically
                        $words = explode(' ', $section['section_title']);
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span>' . $lastWord . '</span>';

                        foreach ($banners as &$banner) {
                            $decodedData = json_decode($banner->datas, true);

                            $banner->title = $decodedData['title'] ?? null;
                            $banner->label = $decodedData['label'] ?? null;
                            $banner->services_count = $decodedData['services_count'] ?? null;
                            $banner->review_count = $decodedData['review_count'] ?? null;

                            $thumbnailImagePath = 'storage/uploads/thumbnail_image_banner/' . $decodedData['thumbnail_image'];
                            $backgroundImagePath = 'storage/uploads/background_image_banner/' . $decodedData['background_image'];

                            $banner->customer_images = [
                                asset('front/img/profiles/avatar-01.jpg'),
                                asset('front/img/profiles/avatar-04.jpg'),
                                asset('front/img/profiles/avatar-01.jpg'),
                                asset('front/img/profiles/avatar-04.jpg'),
                            ];

                            $banner->background_image = (isset($decodedData['background_image']) && file_exists(public_path($backgroundImagePath)))
                                ? asset($backgroundImagePath)
                                : null;

                            $banner->thumbnail_image = (isset($decodedData['thumbnail_image']) && file_exists(public_path($thumbnailImagePath)))
                                ? asset($thumbnailImagePath)
                                : null;


                            unset($banner->datas);
                        }

                        $section['section_type'] = 'banner';
                        $section['type'] = 'banner';
                        $section['design'] = 'banner_one';
                        $section['section_content'] = $banners;
                    }
                }

                // Category
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[category') !== false) {
                        preg_match('/type=([a-zA-Z_]+)\s+limit=(\d+)\s+viewall=(yes|no)/', $section['section_content'], $matches);
                        $type = $matches[1] ?? 'all';
                        $limit = $matches[2] ?? 10;
                        $viewAll = $matches[3] ?? 'no';

                        // Split last word dynamically
                        $words = explode(' ', $section['section_title']);
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $query = Categories::query()
                            ->select('id', 'parent_id', 'name', 'image', 'icon', 'description', 'featured', 'slug')
                            ->where('parent_id', 0)
                            ->where('language_id', $language_id)
                            ->where('status', 1)
                            ->whereNull('deleted_at');

                        // Filter by source type
                        if ($type === 'product') {
                            $query->where('source_type', 'product');
                        } else {
                            $query->where('source_type', 'service');
                        }

                        $categories = $query->limit((int) $limit)->get();

                        if ($type === 'product') {
                            $section['section_type'] = 'product_category';
                            $section['type'] = 'category';
                        } else {
                            $section['section_type'] = 'service_category';
                            $section['type'] = 'category';
                        }

                        foreach ($categories as &$category) {
                            $category->image = uploadedAsset($category->image);
                            $category->icon = uploadedAsset($category->icon);

                            $category->product_count = Product::query()
                                ->whereHas('shop')
                                ->where('source_category', $category->id)
                                ->whereNull('deleted_at')
                                ->where('status', 1)
                                ->where('verified_status', 1)
                                ->whereExists(function ($query) {
                                    $query->select(DB::raw(1))
                                        ->from('users')
                                        ->whereColumn('users.id', 'products.user_id')
                                        ->whereNull('users.deleted_at');
                                })
                                ->count();
                        }

                        $sectionTitle = $section['section_title'] ?? '';
                        $words = explode(' ', $sectionTitle);
                        $lastWord = array_pop($words);
                        $remainingTitle = implode(' ', $words);

                        $section['section_title_main'] = $remainingTitle;
                        $section['section_title_last'] = $lastWord;

                        $section['section_content'] = $categories;
                    }
                }

                // Products
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[product') !== false) {
                        preg_match('/type=([a-zA-Z_]+)\s+limit=(\d+)\s+viewall=(yes|no)/', $section['section_content'], $matches);
                        $type = $matches[1] ?? 'all';
                        $limit = $matches[2] ?? 5;
                        $viewAll = $matches[3] ?? 'no';

                        // Split last word dynamically
                        $words = explode(' ', $section['section_title']);
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $bestSellerIds = getBestSellingProductIds($limit);

                        $service = Product::query()
                                ->whereHas('shop')
                                ->select(
                                    'products.id',
                                    'categories.name as category_name',
                                    'products.source_name',
                                    'products.source_price',
                                    'products.featured',
                                    'products.slug',
                                    'products.language_id',
                                    'user_details.user_id',
                                    'user_details.profile_image',
                                    DB::raw("GROUP_CONCAT(products_meta.source_values SEPARATOR ',') as product_images"),
                                    DB::raw("(SELECT AVG(ratings.rating) FROM ratings WHERE ratings.product_id = products.id AND ratings.parent_id = 0) as average_rating"),
                                    DB::raw("(SELECT COUNT(ratings.id) FROM ratings WHERE ratings.product_id = products.id AND ratings.parent_id = 0) as review_count"),
                                    DB::raw("(SELECT COUNT(bookings.id) FROM bookings WHERE bookings.product_id = products.id) as booking_count"),
                                    DB::raw($authUserId
                                    ? "IF(EXISTS (SELECT 1 FROM wishlists
                                                WHERE wishlists.service_id = products.id
                                                AND wishlists.user_id = $authUserId), 1, 0) as is_wishlist"
                                    : "0 as is_wishlist")
                                )
                                ->join('products_meta', function ($join) {
                                    $join->on('products.id', '=', 'products_meta.product_id')
                                        ->where('products_meta.source_key', '=', 'product_image')
                                        ->whereNull('products_meta.deleted_at');
                                })
                                ->join('categories', 'products.source_category', '=', 'categories.id') // Join with categories to get the name
                                ->leftJoin('bookings', 'products.id', '=', 'bookings.product_id') // Join with bookings table to count bookings
                                ->join('user_details', 'products.user_id', '=', 'user_details.user_id')
                                ->where('products.source_type', 'product')
                                ->whereNull('products.deleted_at')
                                ->where('products.language_id', $language_id)
                                ->where('products.status', 1)
                                ->where('products.verified_status', 1)
                                ->whereExists(function ($query) {
                                    $query->select(DB::raw(1))
                                        ->from('users')
                                        ->whereColumn('users.id', 'products.user_id')
                                        ->whereNull('users.deleted_at');
                                })
                                ->groupBy(
                                    'products.id',
                                    'products.source_category',
                                    'categories.name', // Include in group by
                                    'products.source_name',
                                    'products.source_description',
                                    'products.source_price',
                                    'products.source_brand',
                                    'products.source_stock',
                                    'products.featured',
                                    'products.slug',
                                    'products.language_id',
                                    'user_details.user_id',
                                    'user_details.profile_image',
                                );

                            if ($type === 'best_selling') {
                                $service->whereIn('products.id', $bestSellerIds);
                                $sectionTitle = $section['section_title'] ?? '';
                                $words = explode(' ', $sectionTitle);
                                $lastWord = array_pop($words);
                                $remainingTitle = implode(' ', $words);

                                $section['section_title_main'] = $remainingTitle;
                                $section['section_title_last'] = $lastWord;
                            } elseif ($type === 'popular') {
                                $service->havingRaw('booking_count >= 1');
                                $sectionTitle = $section['section_title'] ?? '';
                                $words = explode(' ', $sectionTitle);
                                $lastWord = array_pop($words);
                                $remainingTitle = implode(' ', $words);

                                $section['section_title_main'] = $remainingTitle;
                                $section['section_title_last'] = $lastWord;
                            } else {
                                $sectionTitle = $section['section_title'] ?? '';
                                $words = explode(' ', $sectionTitle);
                                $lastWord = array_pop($words);
                                $remainingTitle = implode(' ', $words);

                                $section['section_title_main'] = $remainingTitle;
                                $section['section_title_last'] = $lastWord;
                            }

                        $service = $service->limit((int) $limit)->get();

                        if ($type === 'best_selling_product') {
                            $section['section_type'] = 'best_selling';
                            $section['type'] = 'product';
                            $section['design'] = 'service_three';
                        } elseif ($type === 'popular') {
                            $section['section_type'] = 'popular_service';
                            $section['type'] = 'service';
                            $section['design'] = 'service_two';
                        } else {
                            $section['section_type'] = 'service';
                            $section['type'] = 'service';
                            $section['design'] = 'service_one';
                        }

                        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
                            foreach ($service as &$item) {
                                $images = explode(',', $item->product_images);
                                $item->product_images = uploadedAsset($images[0], 'default2');
                                $item->average_rating = $item->average_rating ? round($item->average_rating, 2) : 0;
                                $item->review_count = $item->review_count ?: 0; // Set review count to 0 if null
                                $item->booking_count = $item->booking_count ?: 0; // Set booking count to 0 if null
                                $price = $item->source_price;
                                $variationProduct = getProductVariation($item->id);

                                $item->profile_image = uploadedAsset('profile/' . $item->profile_image, 'profile');
                                $item->source_price = $variationProduct['price'] ?? $price ?? 0;
                            }
                        } else {
                            foreach ($service as &$item) {
                                $images = explode(',', $item->product_images);

                                $price = $item->source_price;
                                $variationProduct = getProductVariation($item->id);

                                $item->profile_image = uploadedAsset('profile/' . $item->profile_image, 'profile');
                                $item->source_price = $variationProduct['price'] ?? $price ?? 0;

                                $item->product_images = uploadedAsset($images[0] ?? '', 'default2');
                                $item->average_rating = $item->average_rating ? round($item->average_rating, 2) : 0;
                                $item->review_count = $item->review_count ?: 0; // Set review count to 0 if null
                            }
                        }

                        $section['section_content'] = $service;
                    }
                }

                // Service
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[service') !== false) {
                        preg_match('/type=([a-zA-Z_]+)\s+limit=(\d+)\s+viewall=(yes|no)/', $section['section_content'], $matches);
                        $type = $matches[1] ?? 'all';
                        $limit = $matches[2] ?? 10;
                        $viewAll = $matches[3] ?? 'no';

                        // Split last word dynamically
                        $words = explode(' ', $section['section_title']);
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $service = Product::select(
                                    'products.id',
                                    'products.source_name',
                                    'categories.name as category_name',
                                    'products.source_price',
                                    'products.featured',
                                    'products.slug',
                                    'products.language_id',
                                    DB::raw("GROUP_CONCAT(products_meta.source_values SEPARATOR ',') as product_images"),
                                    DB::raw("(SELECT AVG(ratings.rating) FROM ratings WHERE ratings.product_id = products.id AND ratings.parent_id = 0) as average_rating"),
                                    DB::raw("(SELECT COUNT(ratings.id) FROM ratings WHERE ratings.product_id = products.id AND ratings.parent_id = 0) as review_count"),
                                    DB::raw("(SELECT COUNT(bookings.id) FROM bookings WHERE bookings.product_id = products.id) as booking_count"),
                                    DB::raw($authUserId
                                    ? "IF(EXISTS (SELECT 1 FROM wishlists
                                                WHERE wishlists.service_id = products.id
                                                AND wishlists.user_id = $authUserId), 1, 0) as is_wishlist"
                                    : "0 as is_wishlist"),
                                    'shops.shop_name',
                                    'shops.shop_logo',
                                )
                                ->join('products_meta', function ($join) {
                                    $join->on('products.id', '=', 'products_meta.product_id')
                                        ->where('products_meta.source_key', '=', 'product_image')
                                        ->whereNull('products_meta.deleted_at');
                                })
                                ->join('categories', 'products.source_category', '=', 'categories.id') // Join with categories to get the name
                                ->leftJoin('bookings', 'products.id', '=', 'bookings.product_id') // Join with bookings table to count bookings
                                ->join('shops', 'products.shop_id', '=', 'shops.id')
                                ->where('products.source_type', 'service')
                                ->whereNull('products.deleted_at')
                                ->whereNull('shops.deleted_at')
                                ->where('products.language_id', $language_id)
                                ->where('products.status', 1)
                                ->where('products.verified_status', 1)
                                ->whereExists(function ($query) {
                                    $query->select(DB::raw(1))
                                        ->from('users')
                                        ->whereColumn('users.id', 'products.user_id')
                                        ->whereNull('users.deleted_at');
                                })
                                ->groupBy(
                                    'products.id',
                                    'products.source_category',
                                    'categories.name', // Include in group by
                                    'products.source_name',
                                    'products.source_description',
                                    'products.source_price',
                                    'products.source_brand',
                                    'products.source_stock',
                                    'products.featured',
                                    'products.slug',
                                    'products.language_id',
                                    'shops.shop_name',
                                    'shops.shop_logo'
                                );

                            if ($type === 'featured') {
                                $service->where('products.featured', 1);
                                $sectionTitle = $section['section_title'] ?? '';
                                $words = explode(' ', $sectionTitle);
                                $lastWord = array_pop($words);
                                $remainingTitle = implode(' ', $words);

                                $section['section_title_main'] = $remainingTitle;
                                $section['section_title_last'] = $lastWord;
                            } elseif ($type === 'popular') {
                                $service->havingRaw('booking_count >= 1');
                                $sectionTitle = $section['section_title'] ?? '';
                                $words = explode(' ', $sectionTitle);
                                $lastWord = array_pop($words);
                                $remainingTitle = implode(' ', $words);

                                $section['section_title_main'] = $remainingTitle;
                                $section['section_title_last'] = $lastWord;
                            } else {
                                $sectionTitle = $section['section_title'] ?? '';
                                $words = explode(' ', $sectionTitle);
                                $lastWord = array_pop($words);
                                $remainingTitle = implode(' ', $words);

                                $section['section_title_main'] = $remainingTitle;
                                $section['section_title_last'] = $lastWord;
                            }

                        $service = $service->limit((int) $limit)->get();

                        if ($type === 'featured') {
                            $section['section_type'] = 'featured_service';
                            $section['type'] = 'service';
                            $section['design'] = 'service_three';
                        } elseif ($type === 'popular') {

                            $section['section_type'] = 'popular_service';
                            $section['type'] = 'service';
                            $section['design'] = 'service_two';
                        } else {
                            $section['section_type'] = 'service';
                            $section['type'] = 'service';
                            $section['design'] = 'service_one';
                        }

                        if (request()->has('is_mobile') && request()->get('is_mobile') === "yes") {
                            foreach ($service as &$item) {
                                $images = explode(',', $item->product_images);
                                $item->product_images = uploadedAsset($images[0], 'default2'); // Use only the first image
                                $item->average_rating = $item->average_rating ? number_format($item->average_rating, 1, '.', '') : 0;
                                $item->review_count = $item->review_count ?: 0; // Set review count to 0 if null
                                $item->booking_count = $item->booking_count ?: 0; // Set booking count to 0 if null
                            }
                        } else {
                            foreach ($service as &$item) {
                                $images = explode(',', $item->product_images);

                                // Take only the first image if exists
                                $item->shop_name = $item->shop_name ?? '';
                                $item->shop_image = uploadedAsset($item->shop_logo, 'default_shop_logo');

                                $price = $item->source_price;
                                $variationProduct = getProductVariation($item->id);

                                $item->source_price = number_format($variationProduct['price'] ?? $price ?? 0);

                                $item->product_images = uploadedAsset(Str::replaceFirst('product_images/', 'product_images/fixed/', $images[0] ?? ''), 'default2');
                                $item->average_rating = $item->average_rating ? number_format($item->average_rating, 1, '.', '') : 0;
                                $item->review_count = $item->review_count ?: 0; // Set review count to 0 if null
                                $item->booking_count = $item->booking_count ?: 0; // Set booking count to 0 if null
                            }
                        }
                        $section['section_content'] = $service;
                    }
                }

                // shop
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[shop') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit   = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order   = $matches[3] ?? 'asc';

                        // Split last word dynamically for UI
                        $words     = explode(' ', $section['section_title']);
                        $lastWord  = array_pop($words);
                        $firstPart = implode(' ', $words);

                        $section['section_title'] = $firstPart .
                            ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $shops = Shops::select(
                                'shops.id',
                                'shops.shop_name',
                                'shops.shop_logo',
                                'shops.slug',
                                'shops.address',
                                'shops.status',
                                'c.name as city',
                                'st.name as state',
                                'co.name as country',
                                'shops.language_id',
                                'shops.created_at'
                            )
                            ->withCount([
                                'products as product_count' => function ($q) {
                                    $q->where('source_type', 'product')
                                    ->whereNull('deleted_at');
                                },
                                'products as service_count' => function ($q) {
                                    $q->where('source_type', 'service')
                                    ->whereNull('deleted_at');
                                },
                                'reviews as review_count' => function ($q) {
                                    $q->whereNull('deleted_at');
                                }
                            ])
                            ->withAvg('ratings as average_rating', 'rating')
                            ->leftJoin('cities as c', 'c.id', '=', 'shops.city')
                            ->leftJoin('states as st', 'st.id', '=', 'shops.state')
                            ->leftJoin('countries as co', 'co.id', '=', 'shops.country')
                            ->where('status', 1)
                            ->whereNull('deleted_at')
                            ->where('language_id', $language_id)
                            ->orderBy('created_at', $order)
                            ->limit((int) $limit)
                            ->get()
                            ->map(function ($shop) {
                                // Add full logo url
                                $shop->shop_logo = uploadedAsset($shop->shop_logo, 'default_shop_logo');
                                $shop->average_rating = number_format($shop->average_rating, 1, '.', '');

                                return $shop;
                            });

                        // Update section meta
                        $section['section_type']    = 'shop';
                        $section['design']          = 'shop_one';
                        $section['section_content'] = $shops;
                    }
                }

                // Raeted Servies
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[rated_service') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order = $matches[3] ?? 'asc';

                        $rated_service = Product::query()
                                ->whereHas('shop')
                                ->select(
                                    'products.id',
                                    'products.source_category',
                                    'categories.name as category_name', // Fetch the category name
                                    'products.source_name',
                                    'products.source_description',
                                    'products.source_price',
                                    'products.source_brand',
                                    'products.source_stock',
                                    'products.featured',
                                    'products.slug',
                                    'products.language_id',
                                    DB::raw("GROUP_CONCAT(products_meta.source_values SEPARATOR ',') as product_images"),
                                    DB::raw("(SELECT AVG(ratings.rating) FROM ratings WHERE ratings.product_id = products.id AND ratings.parent_id = 0) as average_rating")
                                )
                                ->join('products_meta', function ($join) {
                                    $join->on('products.id', '=', 'products_meta.product_id')
                                        ->where('products_meta.source_key', '=', 'product_image')
                                        ->whereNull('products_meta.deleted_at');
                                })
                                ->join('categories', 'products.source_category', '=', 'categories.id') // Join with categories to get the name
                                ->where('products.source_type', 'service')
                                ->where('products.language_id', $language_id)
                                ->whereNull('products.deleted_at')
                                ->where('products.status', 1)
                                ->where('products.verified_status', 1)
                                ->whereExists(function ($query) {
                                    $query->select(DB::raw(1))
                                        ->from('users')
                                        ->whereColumn('users.id', 'products.user_id')
                                        ->whereNull('users.deleted_at');
                                })
                                ->groupBy(
                                    'products.id',
                                    'products.source_category',
                                    'categories.name', // Include in group by
                                    'products.source_name',
                                    'products.source_description',
                                    'products.source_price',
                                    'products.source_brand',
                                    'products.source_stock',
                                    'products.featured',
                                    'products.slug',
                                    'products.language_id'
                                )
                                ->havingRaw('average_rating >= 4.0') // Filter products with average rating >= 4.0
                                ->limit((int) $limit)
                                ->get();

                        foreach ($rated_service as &$item) {
                            // Process product images
                            $images = explode(',', $item->product_images);
                            $item->product_images = array_map(fn($image) => uploadedAsset($image, 'default2'), $images);
                            $item->average_rating = $item->average_rating ? round($item->average_rating, 2) : 0;

                            $price = $item->source_price;
                            $variationProduct = getProductVariation($item->id);

                            $item->source_price = number_format($variationProduct['price'] ?? $price ?? 0);
                        }


                        $sectionTitle = $section['section_title'] ?? '';
                        $words = explode(' ', $sectionTitle);
                        $lastWord = array_pop($words);
                        $remainingTitle = implode(' ', $words);

                        $section['section_title_main'] = $remainingTitle;
                        $section['section_title_last'] = $lastWord;
                        $section['section_type'] = 'rated_service';
                        $section['design'] = 'rated_service_one';
                        $section['section_content'] = $rated_service;
                    }
                }

                // Popular Provider
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[popular_provider') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order = $matches[3] ?? 'asc';


                        $request = new \Illuminate\Http\Request([
                            'type' => '2',
                            'listtype' => 'popular',
                        ]);

                        $getUserListResponse = app()->call('App\Http\Controllers\UserController@getuserlist', ['request' => $request]);

                        $responseData = json_decode($getUserListResponse->getContent(), true);

                        if ($responseData['code'] === 200) {
                            $itemCount = count($responseData['data']);

                            if ($itemCount >= 4) {
                                $itemCount = $itemCount > 8 ? 8 : 4;

                                $popular_provider = collect((array) $responseData['data'])->take($itemCount);

                                if ($order === 'desc') {
                                    $popular_provider = $popular_provider->sortByDesc('created_at');
                                } else {
                                    $popular_provider = $popular_provider->sortBy('created_at');
                                }
                            } else {
                                $popular_provider = collect();
                            }
                        }

                        $section['section_type'] = 'popular_provider';
                        $section['type'] = 'provider';
                        $section['design'] = 'provider_one';
                        $section['section_content'] = $popular_provider;
                    }
                }

                // How It Works
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[how_it_work') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order = $matches[3] ?? 'asc';

                        $how_it_works = GlobalSetting::select('id', 'key', 'value', 'group_id')
                                ->where(['group_id' => 14, 'language_id' => $language_id])
                                ->orderBy('created_at', $order)
                                ->limit((int) $limit)
                                ->get();

                        $sectionTitle = $section['section_title'] ?? '';
                        $words = explode(' ', $sectionTitle);
                        $lastWord = array_pop($words);
                        $remainingTitle = implode(' ', $words);

                        $section['section_title_main'] = $remainingTitle;
                        $section['section_title_last'] = $lastWord;
                        $section['section_type'] = 'how_it_works';
                        $section['design'] = 'how_it_works_one';
                        $section['section_content'] = $how_it_works;
                    }
                }

                // Advertisement
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[advertisement') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order = $matches[3] ?? 'asc';

                        $how_it_works = DB::table('general_settings')->select('id', 'key', 'value', 'group_id')
                                ->where(['group_id' => 14, 'language_id' => $language_id])
                                ->orderBy('created_at', $order)
                                ->limit((int) $limit)
                                ->get();

                        $sectionTitle = $section['section_title'] ?? '';
                        $words = explode(' ', $sectionTitle);
                        $lastWord = array_pop($words);
                        $remainingTitle = implode(' ', $words);

                        $section['section_title_main'] = $remainingTitle;
                        $section['section_title_last'] = $lastWord;
                        $section['section_type'] = 'advertisement';
                        $section['design'] = 'advertisement_one';
                        $section['section_content'] = $how_it_works;
                    }
                }

                // FAQ
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[faq') !== false) {
                        $stripedContent = strip_tags($section['section_content']);
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)(?:\s+order=(asc|desc))?/', $stripedContent, $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';
                        $order = $matches[3] ?? 'asc';

                        // Split last word dynamically
                        $words = explode(' ', $section['section_title']);
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $faqs = DB::table('faqs')->select('id', 'question', 'answer', 'status')->where('status', 1)
                                ->whereNull('deleted_at')
                                ->where('language_id', $language_id)
                                ->orderBy('created_at', $order)
                                ->limit((int) $limit)
                                ->get();

                        $section['section_type'] = 'faq';
                        $section['design'] = 'faq_one';
                        $section['section_content'] = $faqs;
                    }
                }

                //Blog
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[blog') !== false) {
                        preg_match('/type=([a-zA-Z_]+)\s+limit=(\d+)\s+viewall=(yes|no)/', $section['section_content'], $matches);
                        $type = $matches[1] ?? 'all';
                        $limit = $matches[2] ?? 10;
                        $viewAll = $matches[3] ?? 'no';

                        // Split last word dynamically
                        $title = $section['section_title'] ?? '';
                        $words = explode(' ', trim($title));
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $blogs = DB::table('blog_posts')->select('id', 'title', 'image', 'slug', 'category', 'updated_at')
                                ->when($type === 'all', function ($query) {
                                    return $query;
                                })
                                ->limit((int) $limit)
                                ->where('status', 1)
                                ->where('language_id', $language_id)
                                ->whereNull('deleted_at')
                                ->get();

                        foreach ($blogs as &$blog) {
                            $blog->image = uploadedAsset('blogs/' . $blog->image, 'default2');
                            $carbonDate = \Carbon\Carbon::parse($blog->updated_at);

                            // Separate date and month
                            $blog->day = $carbonDate->format('d');              // e.g. "26"
                            $blog->month = strtoupper($carbonDate->format('M')); // e.g. "JUN"

                            unset($blog->updated_at); // optional, if you don't want original
                        }

                        $sectionTitle = $section['section_title'] ?? '';
                        $words = explode(' ', $sectionTitle);
                        $lastWord = array_pop($words);
                        $remainingTitle = implode(' ', $words);

                        $section['section_title_main'] = $remainingTitle;
                        $section['section_title_last'] = $lastWord;
                        $section['section_type'] = 'blog';
                        $section['design'] = 'blog_one';
                        $section['section_content'] = $blogs;
                    }
                }

                //Testimonial
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && strpos($section['section_content'], '[testimonial') !== false) {
                        preg_match('/limit=(\d+)\s+viewall=(yes|no)/', $section['section_content'], $matches);
                        $limit = $matches[1] ?? 10;
                        $viewAll = $matches[2] ?? 'no';

                        // Split last word dynamically
                        $words = explode(' ', $section['section_title']);
                        $lastWord = array_pop($words);
                        $firstPart = implode(' ', $words);

                        // Final HTML
                        $section['section_title'] = $firstPart . ' <span class="text-secondary text-decoration-underline">' . $lastWord . '</span>';

                        $testimonials = DB::table('testimonials')->select('id', 'client_name', 'client_image', 'position', 'description', 'order_by', 'status', 'updated_at')
                                ->limit((int) $limit)
                                ->where('status', 1)
                                ->whereNull('deleted_at')
                                ->get();

                        foreach ($testimonials as &$testimonial) {
                            $testimonial->client_image = asset('storage/testimonials/' . $testimonial->client_image);
                        }

                        $sectionTitle = $section['section_title'] ?? '';
                        $words = explode(' ', $sectionTitle);
                        $lastWord = array_pop($words);
                        $remainingTitle = implode(' ', $words);

                        $section['section_title_main'] = $remainingTitle;
                        $section['section_title_last'] = $lastWord;
                        $section['section_type'] = 'testimonial';
                        $section['design'] = 'testimonial_one';
                        $section['section_content'] = $testimonials;
                    }
                }

                // Facts
                if (is_array($section) && ($section['status'] ?? 0) == 1) {
                    $content = $section['section_content'] ?? '';

                    if (is_string($content) && strpos($content, '[facts') !== false) {

                        $userCount = 1200;   // example static value
                        $clientsSatisfaction = 34; // example static value
                        $teamMembers = 45; // example static value
                        $bookingCount = 1976;     // static

                        $facts = [
                            ["key" => "happy_customers", "value" => $userCount, 'count' => 'K'],
                            ["key" => "clients_satisfaction", "value" => $clientsSatisfaction, 'count' => '%'],
                            ["key" => "team_members", "value" => $teamMembers, 'count' => '+'],
                            ["key" => "booking_count", "value" => $bookingCount, 'count' => '+'],
                        ];

                        // Set section attributes
                        $section['section_type'] = 'facts_only';
                        $section['design'] = 'facts_counter';
                        $section['facts'] = $facts;
                    }
                }

                // about_company
                if (is_array($section) && ($section['status'] ?? 0) == 1) {
                    $content = $section['section_content'] ?? '';
                    if (is_string($content) && strpos($content, '[about_company') !== false) {

                        // Set section attributes
                        $section['section_type'] = 'about_company';
                        $section['design'] = 'about_company';
                    }
                }

                // Newsletter
                if (is_array($section) && ($section['status'] ?? 0) == 1) {
                    $content = $section['section_content'] ?? '';
                    if (is_string($content) && strpos($content, '[newsletter') !== false) {

                        // Set section attributes
                        $section['section_type'] = 'newsletter';
                        $section['design'] = 'newsletter';
                    }
                }

                // Marquee Section (Static Data)
                if (is_array($section) && ($section['status'] ?? 0) == 1) {
                    $content = $section['section_content'] ?? '';

                    if (is_string($content) && strpos($content, '[marquee') !== false) {

                        // Static marquee items
                        $keywordsJsonArray = [
                            ['text' => 'Service Categories'],
                            ['text' => 'Smart Booking System'],
                            ['text' => 'Real Reviews & Ratings'],
                            ['text' => 'Skilled & Certified Professionals'],
                            ['text' => 'Clear & Honest Pricing'],
                            ['text' => '24/7 Emergency Services'],
                            ['text' => 'Wide Service Categories'],
                            ['text' => 'Smart Booking System'],
                            ['text' => 'Real Reviews & Ratings'],
                        ];

                        $section['section_type'] = 'marquee_section';
                        $section['type'] = 'marquee_section';
                        $section['design'] = 'marquee_one';
                        $section['section_content'] = $keywordsJsonArray;
                    }
                }

                // Business With Us
                if ($section['status'] == 1) {
                    if (isset($section['section_content']) && is_string($section['section_content']) && strpos($section['section_content'], '[business_with_us') !== false) {
                        preg_match('/link=([^\s\]]+)/', $section['section_content'], $matches);
                        $link = $matches[1] ?? 'https://example.com';

                        $section['section_type'] = 'business_with_us';
                        $section['design'] = 'business_with_us_one';
                        $section['section_content'] = [
                            'link' => $link,
                        ];
                    }
                }

                if (isset($section['section_content']) && is_string($section['section_content'])) {
                    if (preg_match('/\[[^\]]+\]/', $section['section_content']) === 0) {
                        $section['section_content'] = $section['section_content'];
                        $section['section_type'] = 'multiple_section';
                    }
                }
            }
        }

        if (!empty($page->about_us)) {
            $pageContentSections[] = ['section_type' => 'about_us', 'about_us' => $page->about_us, 'status' => 1]; //about us
        }

        if (!empty($page->terms_conditions)) {
            $pageContentSections[] = ['section_type' => 'terms_conditions', 'terms_conditions' => $page->terms_conditions, 'status' => 1]; //Terms and Conditions
        }

        if (!empty($page->privacy_policy)) {
            $pageContentSections[] = ['section_type' => 'privacy_policy', 'privacy_policy' => $page->privacy_policy,  'status' => 1]; //Privacy Policy
        }

        if (!empty($page->contact_us)) {
            $contact = GlobalSetting::whereIn('key', ['phone_no', 'site_address', 'site_email'])->pluck('value');
            $content = '';
            if (!empty($contact)) {
                if (!empty($contact[0]) && !empty($contact[1]) && !empty($contact[2])) {
                    $content = str_replace(
                        ['{{company_phonenumber}}', '{{company_address}}', '{{company_email}}'],
                        [$contact[0], $contact[1], $contact[2]],
                        $page->contact_us
                    );
                }
            }
            $pageContentSections[] = ['section_type' => 'contact_us', 'contact_us' => $content, 'status' => 1]; //Contact us
        }
        $email = "";
        if (Auth::check()) {
            $email = Auth::user()->email;
        }

        $data = [
            'page_name' => $page->page_title,
            'currency' => $currencySymbol,
            'language_id' => $page->language_id,
            'content_sections' => $pageContentSections,
            'seo_tag' => $page->seo_tag,
            'seo_title' => $page->seo_title,
            'seo_description' => $page->seo_description,
            'email' => $email,
            'status' => $page->status,
        ];

        $content_sections = collect((array) $data['content_sections']);
        $maintenance = GlobalSetting::where('key', 'maintenance')->first();
        $maintenanceContent = GlobalSetting::where('key', 'maintenance_content')->first();

        if ($maintenance && $maintenance->value == '1') {
            Cache::put(
                $cacheKey,
                [
                    'type' => 'maintenance',
                    'maintenanceContent' => $maintenanceContent,
                ],
                now()->addMinutes($ttlMinutes)
            );

            return view('user.partials.maintenance', compact('maintenanceContent'));
        }

        $authID = Auth::id();

        $user_details = DB::table('user_details')->where('user_id', $authID)->first();

        $city_name = '';
        $address = '';

        if ($user_details) {
            if (!empty($user_details->city)) {
                $city = DB::table('cities')->where('id', $user_details->city)->first();
                if ($city) {
                    $city_name = $city->name;
                }
            }

            if (!empty($user_details->address)) {
                $address = $user_details->address;
            }
        }

        if ($city_name || $address) {
            $location_text = trim($city_name . ' - ' . $address, ' -');
        } else {
            $location_text = '';
        }

        if ($slug != 'home-screen-one') {
            $layout = 'frontend.theme_2.app';
        }

        if ($slug === 'delete-my-account') {
            $viewData = compact('data', 'content_sections', 'currencySymbol', 'location_text', 'layout');

            Cache::put(
                $cacheKey,
                [
                    'type' => 'view',
                    'view' => 'delete-my-account',
                    'data' => $viewData,
                ],
                now()->addMinutes($ttlMinutes)
            );

            return view('delete-my-account', $viewData);
        }

        if ($isMobileRequest) {
            $responsePayload = ['code' => "200", 'message' => __('Page details retrieved successfully.'), 'data' => $data];

            Cache::put(
                $cacheKey,
                [
                    'type' => 'json',
                    'payload' => $responsePayload,
                    'status' => 200,
                ],
                now()->addMinutes($ttlMinutes)
            );

            return response()->json($responsePayload, 200);
        } else {
            $defaultTheme = GeneralSetting::where('key', 'default_theme')->first();
            $theme = $defaultTheme ? $defaultTheme->value : 1;
            $viewPath = 'frontend.home.home_' . $theme;
            if (!view()->exists($viewPath)) {
                $viewPath = 'frontend.home.home_1';
            }

            $viewData = compact('data', 'content_sections', 'currencySymbol', 'location_text', 'layout');

            Cache::put(
                $cacheKey,
                [
                    'type' => 'view',
                    'view' => $viewPath,
                    'data' => $viewData,
                ],
                now()->addMinutes($ttlMinutes)
            );

            return view($viewPath, $viewData);
        }
    }

    protected function dispatchCachedPageResponse(array $cachedResponse): View|JsonResponse|null
    {
        $type = $cachedResponse['type'] ?? null;

        if ($type === 'maintenance') {
            return view('user.partials.maintenance', ['maintenanceContent' => $cachedResponse['maintenanceContent'] ?? null]);
        }

        if ($type === 'json') {
            $status = $cachedResponse['status'] ?? 200;
            return response()->json($cachedResponse['payload'] ?? [], $status);
        }

        if ($type === 'view') {
            return view($cachedResponse['view'] ?? 'frontend.home.home_1', $cachedResponse['data'] ?? []);
        }

        return null;
    }

    protected function buildPageCacheKey(string $slug, ?int $languageId, ?int $authUserId, bool $isMobileRequest, ?string $pageName = null): string
    {
        $version = $this->getPageCacheVersion();
        $context = $isMobileRequest ? 'mobile' : 'web';
        $identifier = $isMobileRequest ? ($pageName ?: $slug) : $slug;
        $identifier = Str::slug($identifier ?: 'home-screen-one');
        $languagePart = $languageId ?? 'default';
        $userPart = $authUserId ?? 'guest';

        return sprintf('page_builder:v%s:%s:%s:%s', $version, $context, $identifier, $languagePart) . ':user:' . $userPart;
    }

    protected function getPageCacheVersion(): int
    {
        return Cache::rememberForever('page_builder_cache_version', static function () {
            return 1;
        });
    }

    protected function incrementPageCacheVersion(): void
    {
        $version = Cache::get('page_builder_cache_version', 1);
        Cache::forever('page_builder_cache_version', $version + 1);
    }

    public function deletePage(Request $request): JsonResponse
    {
        $page = Page::find($request->id);

        if (!$page) {
            return response()->json(['success' => false, 'message' => 'Page not found']);
        }

        // Find the Menu entry associated with the Page's language_id
        $menu = Menu::where("language_id", $page->language_id)->first();

        if ($menu) {
            // Decode the JSON from the menus column
            $menus = json_decode($menu->menus, true);

            // Filter out the menu item where page_id matches the deleted page
            $menus = array_filter($menus, function ($menuItem) use ($page) {
                return $menuItem['page_id'] != $page->id;
            });

            // Re-index array to avoid gaps in keys
            $menus = array_values($menus);

            // Update the menu table with the new JSON data
            $menu->menus = json_encode($menus);
            $menu->save();
        }
        Cache::forget('menuList');

        $this->incrementPageCacheVersion();

        // Delete the page after updating menus
        $page->delete();

        $this->incrementPageCacheVersion();

        return response()->json(['success' => true, 'message' => 'Page deleted successfully']);
    }

    public function indexSection(Request $request): JsonResponse
    {
        $orderBy = $request->input('order_by', 'asc');
        $sortBy = $request->input('sort_by', 'id');

        $sections = Section::orderBy($sortBy, $orderBy)->where("status", 1)->get();

        $data = [];
        $baseUrl = asset('storage/uploads');

        foreach ($sections as $section) {
            $decodedDatas = json_decode($section->datas, true);

            if (isset($decodedDatas['background_image'])) {
                $decodedDatas['background_image'] = $baseUrl . '/background_image_banner/' . $decodedDatas['background_image'];
            }

            if (isset($decodedDatas['thumbnail_image'])) {
                $decodedDatas['thumbnail_image'] = $baseUrl . '/thumbnail_image_banner/' . $decodedDatas['thumbnail_image'];
            }

            $data[] = array_merge([
                'id' => $section->id,
                'name' => $section->name,
                'status' => $section->status,
            ], $decodedDatas);
        }


        return response()->json(['code' => '200', 'message' => __('Section details retrieved successfully.'), 'data' => $data], 200);
    }

    public function getPageDetails(Request $request): JsonResponse
    {
        $orderBy = $request->input('order_by', 'asc');
        $sortBy = $request->input('sort_by', 'id');

        $sections = Section::orderBy($sortBy, $orderBy)->get();

        $data = [];
        $baseUrl = asset('storage/uploads');

        foreach ($sections as $section) {
            $decodedDatas = json_decode($section->datas, true);

            if (isset($decodedDatas['background_image'])) {
                $decodedDatas['background_image'] = $baseUrl . '/background_image_banner/' . $decodedDatas['background_image'];
            }

            if (isset($decodedDatas['thumbnail_image'])) {
                $decodedDatas['thumbnail_image'] = $baseUrl . '/thumbnail_image_banner/' . $decodedDatas['thumbnail_image'];
            }

            $data[] = array_merge([
                'id' => $section->id,
                'name' => $section->name,
                'status' => $section->status,
            ], $decodedDatas);
        }

        $pages = Page::all()->map(function ($page) {
            $page->encrypted_id = encrypt($page->id);
            return $page;
        });


        return response()->json(['code' => '200', 'message' => __('Section details retrieved successfully.'), 'data' => $data, 'meta' => $pages], 200);
    }

    public function getDetails(Request $request): JsonResponse
    {
        $id = $request->id;

        $page = '';

        if (request()->has('language_id') && !empty($request->language_id)) {

            $page = Page::where(['parent_id' => $id, 'language_id' => $request->language_id])->first();
            if (empty($page)) {
                $pageData = Page::select('parent_id')->where(['id' => $id])->first();
                $page = Page::where(['id' => $pageData->parent_id, 'language_id' => $request->language_id])->first();
            }
            if (empty($page)) {
                $page = Page::where(['id' => $id, 'language_id' => $request->language_id])->first();
            }
        }

        if ($page) {
            return response()->json([
                'code' => 200,
                'data' => $page,
            ]);
        } else {
            return response()->json([
                'code' => 200,
                'message' => [],
            ], 200);
        }
    }

    public function indexBuilderList(Request $request): JsonResponse
    {
        $orderBy = $request->input('order_by', 'asc');
        $sortBy = $request->input('sort_by', 'id');

        $language_id = Language::select('id', 'code')->where('code', $request->language_code)->first();

        $pages = Page::orderBy($sortBy, $orderBy)->where('language_id', $language_id->id)->get();

        $data = [];

        foreach ($pages as $page) {
            $data[] = [
                'id' => $page->id,
                'page_title' => $page->page_title,
                'slug' => $page->slug,
                'page_content' => $page->page_content,
                'status' => $page->status,
                'encrypted_id' => customEncrypt($page->id, Page::$pageSecretKey),
            ];
        }

        return response()->json(['code' => '200', 'message' => __('Page builder details retrieved successfully.'), 'data' => $data], 200);
    }

    public function storeSection(Request $request): JsonResponse
    {
        if ($request->section_id == 1) {
            $rules['title'] = 'required';
            $rules['label'] = 'required';
        } elseif ($request->section_id == 2) {
            $rules['category'] = 'required';
        } elseif ($request->section_id == 3) {
            $rules['feature_category'] = 'required';
        } elseif ($request->section_id == 4) {
            $rules['popular_category'] = 'required';
        } elseif ($request->section_id == 5) {
            $rules['service'] = 'required';
        } elseif ($request->section_id == 6) {
            $rules['feature_service'] = 'required';
        } elseif ($request->section_id == 7) {
            $rules['popular_service'] = 'required';
        } elseif ($request->section_id == 8) {
            $rules['product'] = 'required';
        } elseif ($request->section_id == 9) {
            $rules['feature_product'] = 'required';
        } elseif ($request->section_id == 10) {
            $rules['popular_product'] = 'required';
        } elseif ($request->section_id == 11) {
            $rules['faq'] = 'required';
        } elseif ($request->section_id == 12) {
            $rules['service_package'] = 'required';
        } elseif ($request->section_id == 13) {
            $rules['about_as'] = 'required';
        } elseif ($request->section_id == 14) {
            $rules['testimonial'] = 'required';
        } elseif ($request->section_id == 15) {
            $rules['how_it_work'] = 'required';
        } elseif ($request->section_id == 16) {
            $rules['blog'] = 'required';
        } else {
            return response()->json(['message' => 'Invalid section ID'], 400);
        }

        $messages = [
            'how_it_work.required' => __('The how it work field is required.'),
        ];
        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $backgroundImagePath = null;
        $thumbnailImagePath = null;

        if ($request->hasFile('background_image')) {
            $backgroundImagePath = Helpers::upload('background_image_banner/', $request->background_image->extension(), $request->file('background_image'));
        }
        if ($request->hasFile('thumbnail_image')) {
            $thumbnailImagePath = Helpers::upload('thumbnail_image_banner/', $request->thumbnail_image->extension(), $request->file('thumbnail_image'));
        }

        $data = [];

        if ($request->section_id == 1) {
            $bannerSectionData = Section::where('id', $request->section_id)->first();
            $oldBackgroundImage = null;
            $oldThumbnailImage  = null;

            if ($bannerSectionData) {
                $data = json_decode($bannerSectionData->datas, true);

                $oldBackgroundImage = $data['background_image'] ?? null;
                $oldThumbnailImage  = $data['thumbnail_image'] ?? null;
            }

            $data = [
                'title' => $request->title,
                'label' => $request->label,
                'show_search' => $request->show_search,
                'show_location' => $request->show_location,
                'popular_search' => $request->popular_search,
                'provider_count' => $request->provider_count,
                'services_count' => $request->services_count,
                'review_count' => $request->review_count,
                'background_image' => $backgroundImagePath ?? $oldBackgroundImage,
                'thumbnail_image' => $thumbnailImagePath ?? $oldThumbnailImage,
            ];
        } elseif ($request->section_id == 2) {
            $data = [
                'category' => $request->category,
            ];
        } elseif ($request->section_id == 3) {
            $data = [
                'feature_category' => $request->feature_category,
            ];
        } elseif ($request->section_id == 4) {
            $data = [
                'popular_category' => $request->popular_category,
            ];
        } elseif ($request->section_id == 5) {
            $data = [
                'service' => $request->service,
            ];
        } elseif ($request->section_id == 6) {
            $data = [
                'feature_service' => $request->feature_service,
            ];
        } elseif ($request->section_id == 7) {
            $data = [
                'popular_service' => $request->popular_service,
            ];
        } elseif ($request->section_id == 8) {
            $data = [
                'product' => $request->product,
            ];
        } elseif ($request->section_id == 9) {
            $data = [
                'feature_product' => $request->feature_product,
            ];
        } elseif ($request->section_id == 10) {
            $data = [
                'popular_product' => $request->popular_product,
            ];
        } elseif ($request->section_id == 11) {
            $data = [
                'faq' => $request->faq,
            ];
        } elseif ($request->section_id == 12) {
            $data = [
                'service_package' => $request->service_package,
            ];
        } elseif ($request->section_id == 13) {
            $data = [
                'about_as' => $request->about_as,
            ];
        } elseif ($request->section_id == 14) {
            $data = [
                'testimonial' => $request->testimonial,
            ];
        } elseif ($request->section_id == 15) {
            $data = [
                'how_it_work' => $request->how_it_work,
            ];
        } elseif ($request->section_id == 16) {
            $data = [
                'blog' => $request->blog,
            ];
        }

        $id = $request->section_id;

        $section = Section::updateOrCreate(
            ['id' => $id],
            ['datas' => json_encode($data)]
        );

        if (!$section) {
            return response()->json(['message' => 'Something went wrong while saving the section!'], 500);
        }

        $this->incrementPageCacheVersion();

        return response()->json(['code' => 200, 'message' => 'Section saved successfully!'], 200);
    }

    public function pageBuilderStore(Request $request): JsonResponse
    {
        $sections = [];
        $titles = $request->input('section_title');
        $labels = $request->input('section_label');
        $contents = $request->input('page_content');
        $statuses = $request->input('page_status', []);

        for ($i = 0; $i < count($titles); $i++) {
            $sections[] = [
                'section_title' => $titles[$i],
                'section_label' => $labels[$i],
                'section_content' => $contents[$i],
                'status' => isset($statuses[$i]) ? 1 : 0,
            ];
        }

        $slug = Str::slug($request->slug);
        $language_id = Language::select('id', 'code')->where('code', $request->currentLang)->first();

        $data = [
            'page_title' => $request->page_title,
            'slug' => $slug,
            'page_content' => json_encode($sections),
            'seo_tag' => $request->tag,
            'seo_title' => $request->seo_title,
            'seo_description' => $request->seo_description,
            'language_id' => $language_id->id,
            'status' => $request->status
        ];

        $save = Page::create($data);

        if (!$save) {
            return response()->json(['message' => __('Something went wrong while saving!')], 500);
        }

        $this->incrementPageCacheVersion();

        return response()->json(['code' => 200, 'message' => __('page_create_success'), 'data' => []], 200);
    }

    public function pageBuilderUpdate(Request $request): JsonResponse
    {
        try {
            $id = $request->id ?? '';
            $languageId = $request->input('language_id');

            // Fixed duplicate assignment
            $langCode = Language::find($languageId)->code ?? 'en';

            $titles = $request->input('section_title', []);
            $labels = $request->input('section_label', []);
            $contents = $request->input('page_content', []);
            $statuses = $request->input('page_status', []);

            $sections = [];

            for ($i = 0; $i < count($titles); $i++) {
                $sections[] = [
                    'section_title' => $titles[$i] ?? '',
                    'section_label' => $labels[$i] ?? '',
                    'section_content' => $contents[$i] ?? '',
                    'status' => isset($statuses[$i]) ? (int)$statuses[$i] : 0,
                ];
            }

            $slug = Str::slug($request->slug);

            $data = [
                'page_title' => $request->page_title,
                'slug' => $slug,
                'page_content' => json_encode($sections),
                'about_us' => $request->about_us,
                'terms_conditions' => $request->terms_conditions,
                'privacy_policy' => $request->privacy_policy,
                'contact_us' => $request->contact_us,
                'seo_tag' => $request->tag,
                'seo_title' => $request->seo_title,
                'seo_description' => $request->seo_description,
                'status' => $request->status
            ];

            $data['language_id'] = $request->language_id;

            $existingPage = Page::where('id', $id)
                ->where('language_id', $request->language_id)
                ->first();

            $existingLangPage = Page::where('parent_id', $id)
                ->where('language_id', $request->language_id)
                ->first();

            $existingParentPage = '';
            $parentPage = Page::where('id', $id)->first();
            if ($parentPage) {
                $existingParentPage = Page::where(['id' => $parentPage->parent_id, 'language_id' => $request->language_id])->first();
            }

            if ($existingPage) {
                $pageSlugExists = Page::where('id', '!=', $existingPage->id)->where('slug', $slug);
                if ($pageSlugExists->exists()) {
                    return response()->json(['code' => 422, 'message' => __('slug_exists', [], $langCode), 'data' => []], 422);
                }
                Page::where('id', $id)
                    ->where('language_id', $request->language_id)
                    ->update($data);
            } elseif ($existingLangPage) {
                $pageSlugExists = Page::where('id', '!=', $existingLangPage->id)->where('slug', $slug);
                if ($pageSlugExists->exists()) {
                    return response()->json(['code' => 422, 'message' => __('slug_exists', [], $langCode), 'data' => []], 422);
                }
                Page::where('parent_id', $id)
                    ->where('language_id', $request->language_id)
                    ->update($data);
            } elseif ($existingParentPage) {
                $pageSlugExists = Page::where('id', '!=', $existingParentPage->id)->where('slug', $slug);
                if ($pageSlugExists->exists()) {
                    return response()->json(['code' => 422, 'message' => __('slug_exists', [], $langCode), 'data' => []], 422);
                }
                Page::where('id', $parentPage->parent_id)
                    ->where('language_id', $request->language_id)
                    ->update($data);
            } else {
                $pageSlugExists = Page::where('slug', $slug);
                if ($pageSlugExists->exists()) {
                    return response()->json(['code' => 422, 'message' => __('slug_exists', [], $langCode), 'data' => []], 422);
                }
                $data['parent_id'] = $id;
                Page::create($data);
            }
            $this->incrementPageCacheVersion();
            return response()->json(['code' => 200, 'message' => __('page_update_success'), 'data' => []], 200);
        } catch (\Exception $e) {
            return response()->json(['code' => 500, 'message' => __('error_occurred_update_data', [], $langCode)], 500);
        }
    }

    public function delete(Request $request): JsonResponse
    {
        $request->validate([
            'id' => 'required|integer|exists:sections,id',
        ]);

        $id = $request->input('id');

        $faq = Section::where('id', $id)->first();

        $faq->delete();
        $this->incrementPageCacheVersion();

        return response()->json(['code' => '200', 'success' => true, 'message' => 'Section deleted successfully.'], 200);
    }
}
