<?php

namespace Modules\Newsletter\app\Repositories\Eloquent;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Modules\GlobalSetting\app\Models\Templates;
use Modules\Newsletter\app\Models\EmailSubscription;
use Modules\Newsletter\app\Repositories\Contracts\NewsletterRepositoryInterface;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Modules\Communication\app\Http\Controllers\EmailController;

class NewsletterRepository implements NewsletterRepositoryInterface
{
    public function index(Request $request)
    {
        $orderBy = $request->order_by ?? 'desc';
        return EmailSubscription::orderBy('id', $orderBy)->get();
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'subscriber_email' => 'required|email|unique:email_subscriptions,email',
        ], [
            'subscriber_email.required' => __('The email field is required.'),
            'subscriber_email.email' => __('Please provide a valid email address.'),
            'subscriber_email.unique' => __('This email is already subscribed.'),
        ]);

        if ($validator->fails()) {
            return ['error' => true, 'messages' => $validator->messages()];
        }

        $data = [
            'email' => $request->subscriber_email,
            'status' => 1,
        ];

        EmailSubscription::create($data);

        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', 9)
            ->first();

        if ($template) {
            $settingData = getCommonSettingData(['company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website']);
            $companyName = $settingData['company_name'] ?? '';
            $companyWebsite = $settingData['website'] ?? '';
            $companyPhone = $settingData['phone_no'] ?? '';
            $companyEmail = $settingData['site_email'] ?? '';
            $contact = $companyEmail . ' | ' . $companyPhone;

            $tempdata = [
                '{{customer_name}}' => $data['email'],
                '{{user_name}}' => $data['email'],
                '{{email_id}}' => $data['email'],
                '{{company_name}}' => $companyName,
                '{{website_link}}' => $companyWebsite,
                '{{contact}}' => $contact,
            ];

            $subject = str_replace(array_keys($tempdata), array_values($tempdata), $template->subject);
            $content = str_replace(array_keys($tempdata), array_values($tempdata), $template->content);

            $data = [
                'to_email' => $data['email'],
                'subject' => $subject,
                'content' => $content,
            ];

            try {
                $emailController = app(EmailController::class);
                $emailController->sendEmail(new Request($data));
            } catch (\Exception $e) {
                Log::error('Email sending failed to ' . $data['to_email'] . ': ' . $e->getMessage());
            }

        }

        return ['error' => false];
    }

    public function destroy(Request $request)
    {
        return EmailSubscription::where('id', $request->id)->delete();
    }

    public function subscriberStatusChange(Request $request)
    {
        return EmailSubscription::where('id', $request->id)->update([
            'status' => $request->status
        ]);
    }

    public function sendNewsletterEmail(Request $request)
    {
        $template = Templates::select('subject', 'content')
            ->where('type', 1)
            ->where('notification_type', 9)
            ->first();

        $emailIds = $request->email_ids ?? [];
        $isError = false;

        if ($template) {
            $settingData = getCommonSettingData(['company_name', 'site_email', 'phone_no', 'site_address', 'postal_code', 'website']);
            $companyName = $settingData['company_name'] ?? '';
            $companyWebsite = $settingData['website'] ?? '';
            $companyPhone = $settingData['phone_no'] ?? '';
            $companyEmail = $settingData['site_email'] ?? '';
            $contact = $companyEmail . ' | ' . $companyPhone;

            foreach ($emailIds as $emailId) {
                $tempdata = [
                    '{{customer_name}}' => $emailId,
                    '{{user_name}}' => $emailId,
                    '{{email_id}}' => $emailId,
                    '{{company_name}}' => $companyName,
                    '{{website_link}}' => $companyWebsite,
                    '{{contact}}' => $contact,
                ];
                $subject = str_replace(array_keys($tempdata), array_values($tempdata), $template->subject);
                $content = str_replace(array_keys($tempdata), array_values($tempdata), $template->content);

                $data = [
                    'to_email' => $emailId,
                    'subject' => $subject,
                    'content' => $content,
                ];

                try {
                    $emailController = app(EmailController::class);
                    $emailController->sendEmail(new Request($data));
                    $isError = false;
                } catch (\Exception $e) {
                    $isError = true;
                    Log::error('Email sending failed to ' . $data['to_email'] . ': ' . $e->getMessage());
                }
            }
        }

        return ['error' => $isError];
    }
}
