<?php

namespace Modules\MenuBuilder\app\Repositories\Eloquent;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Modules\GlobalSetting\app\Models\Language;
use Modules\MenuBuilder\app\Models\Menu;
use Modules\MenuBuilder\app\Repositories\Contracts\MenuBuilderRepositoryInterface;

class MenuBuilderRepository implements MenuBuilderRepositoryInterface
{
    public function index(Request $request): array
    {
        try {
            $language = $this->resolveLanguage($request);

            if (!$language) {
                return [
                    'code' => 404,
                    'message' => __('language_not_found'),
                    'data' => [],
                ];
            }

            $cacheKey = $this->menuCacheKey($language->id);
            $data = Cache::rememberForever($cacheKey, function () use ($language) {
                $menu = Menu::select('id', 'menus', 'language_id')
                    ->where(['language_id' => $language->id])
                    ->first();

                if ($menu && !empty($menu->menus) && is_string($menu->menus)) {
                    $menu->menus = json_decode($menu->menus);
                }

                return $menu;
            });

            return [
                'code' => 200,
                'message' => __('Website menus retrieved successfully.'),
                'data' => $data ?? [],
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => __('An error occurred while retrieving website menus.'),
                'error' => $e->getMessage(),
            ];
        }
    }

    public function store(Request $request): array
    {
        $menus = $request->input('menus');
        $id = $request->id;

        try {

            $language = $this->resolveLanguage($request);
            $languageId = $language?->id;
            $langCode = $language?->code ?? App::getLocale();

            if (!$languageId) {
                return [
                    'code' => 404,
                    'message' => __('language_not_found'),
                    'data' => [],
                ];
            }

            if (empty($id)) {
                Menu::create([
                    'menus' => $menus,
                    'language_id' => $languageId
                ]);
            } else {
                Menu::where(['id' => $id, 'language_id' => $languageId])->update([
                    'menus' => $menus,
                    'language_id' => $languageId
                ]);
            }
            Cache::forget($this->menuCacheKey($languageId));

            return [
                'code' => 200,
                'message' => __('menu_save_success', [], $langCode),
                'data' => json_encode($menus),
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => __('menus_save_error', [], $langCode),
                'error' => $e->getMessage()
            ];
        }
    }

    public function getBuiltMenus(Request $request): array
    {
        $orderBy = $request->order_by ?? 'asc';

        try {

            $language = $this->resolveLanguage($request);

            if (!$language) {
                return [
                    'code' => 404,
                    'message' => __('language_not_found'),
                    'data' => [],
                ];
            }

            $cacheKey = $this->builtMenuCacheKey($language->id, $orderBy);
            $data = Cache::rememberForever($cacheKey, function () use ($language, $orderBy) {
                return DB::table('pages')
                    ->select('id', 'page_title', 'slug')
                    ->where(['language_id' => $language->id])
                    ->orderBy('id', $orderBy)
                    ->get();
            });

            return [
                'code' => 200,
                'message' => __('Built in menus retrieved successfully.'),
                'data' => $data,
            ];
        } catch (\Exception $e) {
            return [
                'code' => 500,
                'message' => __('An error occurred while retrieving built in menus.'),
                'error' => $e->getMessage(),
            ];
        }
    }

    private function resolveLanguage(Request $request): ?Language
    {
        $langCode = $request->input('language_code', App::getLocale());
        $language = Language::where('code', $langCode)->first();

        if ($request->filled('language_id')) {
            $language = Language::find($request->language_id) ?? $language;
        }

        return $language;
    }

    private function menuCacheKey(int $languageId): string
    {
        return "menu_builder.menus.{$languageId}";
    }

    private function builtMenuCacheKey(int $languageId, string $orderBy): string
    {
        return "menu_builder.built_menus.{$languageId}.{$orderBy}";
    }
}
