<?php

namespace Modules\GlobalSetting\app\Repositories\Eloquent;

use Illuminate\Support\Facades\Storage;
use Modules\GlobalSetting\app\Models\Language;
use Modules\GlobalSetting\Entities\GlobalSetting;
use Modules\GlobalSetting\app\Repositories\Contracts\GlobalSettingInterface;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\UploadedFile;


class GlobalSettingRepository implements GlobalSettingInterface
{
    private const CACHE_TTL_MINUTES = 30;

    public function __construct(protected GlobalSetting $model) {}

    public function index(array $filters = [])
    {
        return $this->model->filter($filters)->get();
    }

    public function getByGroup(int $groupId)
    {
        return $this->rememberGroupCache($groupId, 'default', function () use ($groupId) {
            $query = $this->model->where('group_id', $groupId);

            if ($groupId === 1) {
                $query->orWhere('key', 'sso_status');
            }

            $settings = $query->get();

            return $settings->map(function ($setting) {
                if ($setting->key === 'invoice_company_logo') {
                    $setting->value = GlobalSetting::file($setting->value);
                }
                return $setting;
            });
        });
    }

    public function getByGroupAndLanguage(int $groupId, int $languageId, bool $single = false)
    {
        $suffix = 'language_' . $languageId . ($single ? '_single' : '_list');

        return $this->rememberGroupCache($groupId, $suffix, function () use ($groupId, $languageId, $single) {
            $query = $this->model->where('group_id', $groupId)
                ->where('language_id', $languageId);

            $result = $single ? $query->first() : $query->get();

            if (!$single && $result instanceof \Illuminate\Support\Collection) {
                return $result->map(function ($setting) {
                    if ($setting->key === 'invoice_company_logo') {
                        $setting->value = GlobalSetting::file($setting->value);
                    }

                    return $setting;
                });
            }

            if ($result && $result instanceof GlobalSetting && $result->key === 'invoice_company_logo') {
                $result->value = GlobalSetting::file($result->value);
            }

            return $result;
        });
    }

    public function create(array $data)
    {
        $setting = $this->model->create($data);

        if (isset($setting->group_id)) {
            $this->clearGroupCache((int) $setting->group_id);
        }

        if (isset($setting->key)) {
            $this->clearSettingCache($setting->key);
        }

        return $setting;
    }

    public function update(int $id, array $data)
    {
        $setting = $this->model->findOrFail($id);
        $setting->update($data);
        $this->clearGroupCache((int) $setting->group_id);
        $this->clearSettingCache($setting->key);
        return $setting;
    }

    public function delete(int $id)
    {
        $setting = $this->model->findOrFail($id);
        $setting->delete();
        $this->clearGroupCache((int) $setting->group_id);
        $this->clearSettingCache($setting->key);
        return true;
    }

    public function updateGeneralSettings(array $data)
    {
        DB::beginTransaction();
        try {
            if (!array_key_exists("save_single_vendor_status", $data)) {
                $this->persistSetting('save_single_vendor_status', 'off', $data['group_id']);
            }

            if (isset($data['sso_status'])) {
                $this->persistSetting('sso_status', $data['sso_status'], 4);
            }

            foreach ($data as $key => $value) {
                if ($key != 'group_id') {
                    $this->persistSetting($key, $value, $data['group_id']);
                }
            }

            Cache::forget('singlevendor');
            $this->clearGroupCache($data['group_id']);

            if (isset($data['sso_status'])) {
                $this->clearGroupCache(4);
            }

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function updatePaymentSettings(array $data)
    {
        DB::beginTransaction();
        try {
            $groupId = $data['group_id'] ?? 13;

            foreach ($data as $key => $value) {
                if ($key !== 'group_id') {
                    $this->persistSetting($key, $value, $groupId);
                }
            }

            $this->clearGroupCache($groupId);

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function updateInvoiceSettings(array $data)
    {
        DB::beginTransaction();
        try {
            if ( $data['invoice_logo'] != "undefined" && isset($data['invoice_logo']) && $data['invoice_logo']->isValid()) {
                $path = $data['invoice_logo']->store('invoice-logos', 'public');
                $this->persistSetting('invoice_company_logo', $path, $data['group_id']);
            }

            foreach ($data as $key => $value) {
                if ($key != 'group_id' && $key != 'invoice_logo') {
                    $this->persistSetting($key, $value, $data['group_id']);
                }
            }

            $this->clearGroupCache($data['group_id']);

            DB::commit();
            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function updateEnvVariables(array $data)
    {
        $path = base_path('.env');
        if (!File::exists($path)) {
            return false;
        }

        $envContent = File::get($path);
        $updated = false;

        foreach ($data as $key => $value) {
            $pattern = "/^{$key}=.*/m";
            if (preg_match($pattern, $envContent)) {
                $envContent = preg_replace($pattern, "{$key}={$value}", $envContent);
                $updated = true;
            } else {
                $envContent .= "\n{$key}={$value}";
                $updated = true;
            }
        }

        if ($updated) {
            return File::put($path, $envContent) !== false;
        }

        return false;
    }

    public function getSettingByKey(string $key)
    {
        return $this->rememberSettingCache($key, function () use ($key) {
            return $this->model->where('key', $key)->first();
        });
    }
    public function updateOrCreate(array $attributes, array $values)
    {
        $setting = $this->model->updateOrCreate($attributes, $values);

        if ($setting) {
            if (isset($setting->group_id)) {
                $this->clearGroupCache((int) $setting->group_id);
            }

            if (isset($setting->key)) {
                $this->clearSettingCache($setting->key);
            }
        }

        return $setting;
    }

    public function deleteByKey(string $key)
    {
        $setting = $this->model->where('key', $key)->first();

        if ($setting) {
            $this->clearGroupCache((int) $setting->group_id);
            $this->clearSettingCache($setting->key);
        }

        return $this->model->where('key', $key)->delete();
    }

    public function getFile(string $path)
    {
        return Storage::url($path);
    }

    public function updateTaxStatus(string $key, string $status)
    {
        return $this->model->where('key', $key)->update(['value' => $status]);
    }

    public function updateMultiple(array $settings, int $groupId, ?int $languageId = null)
    {
        foreach ($settings as $key => $value) {
            $attributes = ['key' => $key];

            if ($languageId) {
                $attributes['language_id'] = $languageId;
            }

            $dataToUpdate = [
                'value' => $value,
                'group_id' => $groupId,
            ];

            if ($languageId) {
                $dataToUpdate['language_id'] = $languageId;
            }

            $this->model->updateOrCreate($attributes, $dataToUpdate);
            $this->clearSettingCache($key, $languageId);
        }

        $suffixes = ['default'];

        if ($languageId) {
            $suffixes = array_merge($suffixes, $this->languageCacheSuffixes($languageId));
        }

        $this->clearGroupCache($groupId, $suffixes);
    }

     public function updateLogoSettings(array $data)
    {
        try {
            // Exclude non-logo fields
            $settings = collect($data)->except([
                '_token', 'logo', 'favicon', 'icon', 'dark_logo', 'mobile_icon'
            ])->toArray();

            // Update logo only if file exists
            if (!empty($data['logo']) && $data['logo'] instanceof UploadedFile) {
                $this->handleLogoUpdate($data['logo'], 'site_logo', $data['group_id']);
            }

            if (!empty($data['favicon']) && $data['favicon'] instanceof UploadedFile) {
                $this->handleLogoUpdate($data['favicon'], 'site_favicon', $data['group_id']);
            }

            if (!empty($data['icon']) && $data['icon'] instanceof UploadedFile) {
                $this->handleLogoUpdate($data['icon'], 'site_icon', $data['group_id']);
            }

            if (!empty($data['mobile_icon']) && $data['mobile_icon'] instanceof UploadedFile) {
                $this->handleLogoUpdate($data['mobile_icon'], 'site_mobile_icon', $data['group_id']);
            }

            if (!empty($data['dark_logo']) && $data['dark_logo'] instanceof UploadedFile) {
                $this->handleLogoUpdate($data['dark_logo'], 'site_dark_logo', $data['group_id']);
            }

            $this->clearLogoCache();
            $this->clearGroupCache($data['group_id'], ['default', 'logo']);

            return [
                'success' => true,
                'message' => __('Global setting updated successfully.')
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error updating logo settings: ' . $e->getMessage()
            ];
        }
    }

    public function getLogoSettings(int $groupId)
    {
        return $this->rememberGroupCache($groupId, 'logo', function () use ($groupId) {
            $settings = GlobalSetting::where('group_id', $groupId)->get();

            return $settings->map(function ($setting) {
                if (in_array($setting->key, ['site_logo', 'site_favicon', 'site_icon', 'site_dark_logo', 'site_mobile_icon'])) {
                    $setting->value = $this->getFileUrl($setting->value);
                }
                return $setting;
            });
        });
    }

    public function updateCustomSettings(array $data)
    {
        try {
            $settings = collect($data)->except(['_token'])->toArray();

            foreach ($settings as $key => $value) {
                $this->updateOrCreateSetting($key, [
                    'value' => $value ?? '',
                    'group_id' => $data['group_id']
                ]);
                $this->clearSettingCache($key);
            }

            // Handle custom CSS/JS files if present
            if (isset($data['custom_setting_content'])) {
                $cssFilePath = public_path('assets/css/custom.css');
                file_put_contents($cssFilePath, $data['custom_setting_content']);
            }

            if (isset($data['custom_setting_content1'])) {
                $jsFilePath = public_path('assets/js/custom.js');
                file_put_contents($jsFilePath, $data['custom_setting_content1']);
            }

            $this->clearGroupCache($data['group_id'], ['default', 'custom']);

            return [
                'success' => true,
                'message' => __('Global setting updated successfully.')
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error updating custom settings: ' . $e->getMessage()
            ];
        }
    }

    public function getCustomSettings(int $groupId)
    {
        return $this->rememberGroupCache($groupId, 'custom', function () use ($groupId) {
            $settings = GlobalSetting::where('group_id', $groupId)->get();

            return $settings->map(function ($setting) {
                if ($setting->key === 'invoice_company_logo') {
                    $setting->value = $this->getFileUrl($setting->value);
                }
                return $setting;
            });
        });
    }

    public function updateOrCreateSetting(string $key, array $data)
    {
        $setting = GlobalSetting::updateOrCreate(
            ['key' => $key],
            $data
        );

        $this->clearSettingCache($key);

        if (isset($data['group_id'])) {
            $this->clearGroupCache($data['group_id']);
        }

        return $setting;
    }

    public function deleteFileIfExists(string $filePath)
    {
        if ($filePath && Storage::disk('public')->exists($filePath)) {
            Storage::disk('public')->delete($filePath);
        }
    }

    public function storeFile($file, string $directory): string
    {
        if ($file instanceof UploadedFile) {
            return $file->store($directory, 'public');
        }

        throw new \InvalidArgumentException("Invalid file type provided for storage.");
    }

     protected function handleLogoUpdate(UploadedFile $file, string $key, int $groupId)
    {
        $oldSetting = GlobalSetting::where('key', $key)->first();

        if ($oldSetting && $oldSetting->value) {
            $this->deleteFileIfExists($oldSetting->value);
        }

        $path = $this->storeFile($file, $this->getDirectoryForKey($key));

        $this->updateOrCreateSetting($key, [
            'value'    => $path,
            'group_id' => $groupId
        ]);
    }

    protected function getDirectoryForKey(string $key): string
    {
        return match ($key) {
            'site_logo', 'site_dark_logo' => 'logos',
            'site_favicon'                => 'favicons',
            'site_icon', 'site_mobile_icon' => 'icons',
            default                        => 'uploads',
        };
    }

    protected function getFileUrl(?string $path): ?string
    {
        return $path ? Storage::disk('public')->url($path) : null;
    }

    protected function clearLogoCache()
    {
        Cache::forget('logoPath');
        Cache::forget('faviconPath');
        Cache::forget('darkLogoPath');
        Cache::forget('smallLogoPath');
        Cache::forget('iconMobilePath');
    }

    public function updateCopyrightSettings(array $data)
    {
        try {
            $language = Language::select('id', 'code')->findOrFail($data['language_id']);
            $key = 'copyright_' . $language->code;

            $this->updateOrCreateSettingWithLanguage(
                [
                    'group_id' => $data['group_id'],
                    'language_id' => $data['language_id'],
                    'key' => $key,
                ],
                ['value' => $data['copyright']]
            );

            Cache::forget('copyRight_' . $data['language_id']);
            $this->clearGroupCache($data['group_id'], array_merge(
                ['default'],
                $this->languageCacheSuffixes($data['language_id'])
            ));
            $this->clearSettingCache($key, $data['language_id']);

            return [
                'success' => true,
                'message' => __('Copyright setting updated successfully.'),
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error updating copyright settings: ' . $e->getMessage(),
            ];
        }
    }

    public function updateCookiesSettings(array $data)
    {
        try {
            $language = Language::select('id', 'code')->findOrFail($data['language_id']);
            $keys = [
                'cookies_content_text',
                'cookies_position',
                'agree_button_text',
                'decline_button_text',
                'show_decline_button',
                'lin_for_cookies_page'
            ];

            foreach ($keys as $key) {
                $dynamicKey = $key . '_' . $language->code;
                $value = $data[$key] ?? null;

                $this->updateOrCreateSettingWithLanguage(
                    [
                        'group_id' => $data['group_id'],
                        'key' => $dynamicKey,
                        'language_id' => $data['language_id'],
                    ],
                    ['value' => $value]
                );
                $this->clearSettingCache($dynamicKey, $data['language_id']);
            }

            $this->clearGroupCache($data['group_id'], array_merge(
                ['default'],
                $this->languageCacheSuffixes($data['language_id'])
            ));

            return [
                'success' => true,
                'message' => __('Cookies settings updated successfully.')
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error updating cookies settings: ' . $e->getMessage()
            ];
        }
    }

    public function updateOrCreateSettingWithLanguage(array $conditions, array $data)
    {
        $setting = GlobalSetting::updateOrCreate($conditions, $data);

        if (isset($conditions['key'])) {
            $this->clearSettingCache($conditions['key'], $conditions['language_id'] ?? null);
        }

        if (isset($conditions['group_id'])) {
            $suffixes = ['default'];

            if (isset($conditions['language_id'])) {
                $suffixes = array_merge($suffixes, $this->languageCacheSuffixes($conditions['language_id']));
            }

            $this->clearGroupCache($conditions['group_id'], $suffixes);
        }

        return $setting;
    }

    protected function persistSetting(string $key, $value, int $groupId, ?int $languageId = null): void
    {
        $attributes = ['key' => $key];

        if ($languageId !== null) {
            $attributes['language_id'] = $languageId;
        }

        $values = [
            'value' => $value,
            'group_id' => $groupId,
        ];

        if ($languageId !== null) {
            $values['language_id'] = $languageId;
        }

        $this->model->updateOrCreate($attributes, $values);
        $this->clearSettingCache($key, $languageId);
    }

    protected function cacheKey(string $type, ...$segments): string
    {
        $normalized = array_map(static function ($segment) {
            if ($segment === null || $segment === '') {
                return 'null';
            }

            if (is_scalar($segment)) {
                return (string) $segment;
            }

            return md5(serialize($segment));
        }, $segments);

        return implode(':', array_merge(['global_setting', $type], $normalized));
    }

    protected function rememberCached(string $key, callable $callback)
    {
        return Cache::remember($key, now()->addMinutes(self::CACHE_TTL_MINUTES), $callback);
    }

    protected function rememberGroupCache(int $groupId, string $suffix, callable $callback)
    {
        return $this->rememberCached($this->cacheKey('group', $groupId, $suffix), $callback);
    }

    protected function rememberSettingCache(string $key, callable $callback, ?int $languageId = null)
    {
        return $this->rememberCached($this->cacheKey('setting', $key, $languageId ?? 'all'), $callback);
    }

    protected function clearGroupCache(int $groupId, array $suffixes = ['default']): void
    {
        foreach ($suffixes as $suffix) {
            Cache::forget($this->cacheKey('group', $groupId, $suffix));
        }
    }

    protected function clearSettingCache(string $key, ?int $languageId = null): void
    {
        Cache::forget($this->cacheKey('setting', $key, $languageId ?? 'all'));
    }

    protected function languageCacheSuffixes(int $languageId): array
    {
        return [
            'language_' . $languageId . '_list',
            'language_' . $languageId . '_single',
        ];
    }
}
