<?php

namespace Modules\Faq\app\Repositories\Eloquent;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Modules\Faq\app\Models\Faq;
use Modules\Faq\app\Repositories\Contracts\FaqRepositoryInterface;
use Modules\GlobalSetting\app\Models\Language;
use Modules\Faq\app\Support\FaqCache;

class FaqRepository implements FaqRepositoryInterface
{
    private const SORTABLE_COLUMNS = ['id', 'order_by', 'created_at'];
    private const DEFAULT_SORT_COLUMN = 'id';
    private const DEFAULT_SORT_DIRECTION = 'desc';

    public function getAll(Request $request)
    {
        $orderBy = $this->resolveSortDirection($request->input('order_by', self::DEFAULT_SORT_DIRECTION));
        $sortBy = $this->resolveSortColumn($request->input('sort_by', self::DEFAULT_SORT_COLUMN));

        $languageId = $this->resolveLanguageId($request);

        $cacheKey = sprintf('list:%d:%s:%s', $languageId, $sortBy, $orderBy);

        return FaqCache::remember($cacheKey, function () use ($languageId, $sortBy, $orderBy) {
            return Faq::query()
                ->select(['id', 'question', 'answer', 'status'])
                ->where('language_id', $languageId)
                ->orderBy($sortBy, $orderBy)
                ->get()
                ->map(fn (Faq $faq) => [
                    'id' => $faq->id,
                    'question' => $faq->question,
                    'answer' => $faq->answer,
                    'status' => $faq->status,
                ])->all();
        });
    }

    public function store(Request $request)
    {
        $languageId = $this->resolveLanguageId($request);

        $exists = Faq::query()
            ->where('question', $request->question)
            ->where('language_id', $languageId)
            ->whereNull('deleted_at')
            ->exists();

        if ($exists) {
            return ['exists' => true];
        }

        $faq = Faq::create([
            'question' => $request->question,
            'answer' => $request->answer,
            'status' => $request->boolean('status') ? 1 : 0,
            'order_by' => $this->nextOrderValue(),
            'language_id' => $languageId,
        ]);

        if ($faq) {
            FaqCache::flush();
            incrementPageCacheVersion();
        }

        return $faq;
    }

    public function update(Request $request)
    {
        $faqId = (int) $request->edit_id;
        $languageId = $this->resolveLanguageId($request);

        $data = [
            'question' => $request->edit_question,
            'answer' => $request->edit_answer,
            'status' => $request->boolean('status') ? 1 : 0,
        ];

        $faq = Faq::query()->where('id', $faqId)->where('language_id', $languageId)->first();
        incrementPageCacheVersion();

        if ($faq) {
            $updated = $faq->update($data);
            if ($updated) {
                FaqCache::flush();
            }

            return $updated;
        }

        $faqLang = Faq::query()->where('parent_id', $faqId)->where('language_id', $languageId)->first();
        if ($faqLang) {
            $updated = $faqLang->update($data);
            if ($updated) {
                FaqCache::flush();
            }

            return $updated;
        }

        $parent = Faq::find($faqId);
        if (!$parent) {
            return false;
        }

        $rootId = $parent->parent_id ?: $parent->id;
        $root = $parent->parent_id ? Faq::find($parent->parent_id) : $parent;

        $data['parent_id'] = $rootId;
        $data['language_id'] = $languageId;
        $data['order_by'] = $root?->order_by ?? $this->nextOrderValue();

        $newTranslation = Faq::create($data);
        if ($newTranslation) {
            FaqCache::flush();
        }

        return $newTranslation;
    }

    public function delete(Request $request)
    {
        $faq = Faq::findOrFail($request->id);
        $deleted = $faq->delete();

        if (!$faq->parent_id) {
            Faq::where('parent_id', $faq->id)->delete();
        }

        FaqCache::flush();
        incrementPageCacheVersion();

        return $deleted;
    }

    public function getById(Request $request)
    {
        $id = (int) $request->id;
        $languageId = $this->resolveLanguageId($request);

        $faq = Faq::query()
            ->where('language_id', $languageId)
            ->where(function ($query) use ($id) {
                $query->where('id', $id)
                    ->orWhere('parent_id', $id);
            })
            ->first();

        if ($faq) {
            return $faq;
        }

        $parentId = Faq::query()->where('id', $id)->value('parent_id');
        if ($parentId) {
            $fallback = Faq::query()
                ->where('language_id', $languageId)
                ->where('id', $parentId)
                ->first();

            if ($fallback) {
                return $fallback;
            }
        }

        return Faq::find($id);
    }

    private function resolveLanguageId(Request $request): int
    {
        if ($request->filled('language_id')) {
            return (int) $request->input('language_id');
        }

        $languageCode = $request->input('language_code') ?? App::getLocale();
        if ($languageCode) {
            $languageId = Language::where('code', $languageCode)->value('id');
            if ($languageId) {
                return (int) $languageId;
            }
        }

        $defaultId = Language::where('is_default', 1)->value('id');
        if ($defaultId) {
            return (int) $defaultId;
        }

        return (int) (Language::value('id') ?? 1);
    }

    private function resolveSortColumn(string $column): string
    {
        return in_array($column, self::SORTABLE_COLUMNS, true) ? $column : self::DEFAULT_SORT_COLUMN;
    }

    private function resolveSortDirection(string $direction): string
    {
        return strtolower($direction) === 'asc' ? 'asc' : self::DEFAULT_SORT_DIRECTION;
    }

    private function nextOrderValue(): int
    {
        $maxOrder = Faq::query()->whereNull('parent_id')->max('order_by');

        return $maxOrder ? ((int) $maxOrder + 1) : 1;
    }
}
