<?php

namespace Modules\Coupon\app\Repositories\Eloquent;

use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Modules\Coupon\app\Models\Coupon;
use Modules\Coupon\app\Repositories\Contracts\CouponRepositoryInterface;

class CouponRepository implements CouponRepositoryInterface
{
    public function getCouponsByUser($userId, $isValid, $orderBy)
    {
        $condition = (int) $isValid === 1 ? '>=' : '<';
        $currentDate = Carbon::today();
        $cacheKey = $this->userCouponsCacheKey($userId, $isValid, $orderBy);

        return Cache::remember($cacheKey, now()->addMinutes(10), function () use ($userId, $condition, $currentDate, $orderBy) {
            return Coupon::where(['created_by' => $userId])
                ->where('end_date', $condition, $currentDate)
                ->orderBy('id', $orderBy)
                ->get()
                ->map(function ($coupon) {
                    if ($coupon->product_id) {
                        $coupon->product_id = explode(',', $coupon->product_id);
                    }
                    if ($coupon->category_id) {
                        $coupon->category_id = explode(',', $coupon->category_id);
                    }
                    if ($coupon->subcategory_id) {
                        $coupon->subcategory_id = explode(',', $coupon->subcategory_id);
                    }

                    return $coupon;
                });
        });
    }

    public function findCoupon($id)
    {
        $cacheKey = $this->couponDetailsCacheKey($id);

        return Cache::remember($cacheKey, now()->addMinutes(10), function () use ($id) {
            $coupon = Coupon::where('id', $id)->first();

            if ($coupon) {
                $coupon->product_id = $coupon->product_id ? explode(',', $coupon->product_id) : [];
                $coupon->category_id = $coupon->category_id ? explode(',', $coupon->category_id) : [];
                $coupon->subcategory_id = $coupon->subcategory_id ? explode(',', $coupon->subcategory_id) : [];
            }

            return $coupon;
        });
    }

    public function createOrUpdateCoupon(array $data, $id = null)
    {
        $coupon = Coupon::updateOrCreate(['id' => $id], $data);

        $this->flushCouponCaches($coupon);

        return $coupon;
    }

    public function deleteCoupon($id)
    {
        $coupon = Coupon::find($id);

        if (!$coupon) {
            return false;
        }

        $deleted = (bool) $coupon->delete();

        if ($deleted) {
            $this->flushCouponCaches($coupon);
        }

        return $deleted;
    }

    public function updateCouponStatus($id, $status)
    {
        $updated = Coupon::where('id', $id)->update(['status' => $status]);

        if ($updated) {
            $coupon = Coupon::find($id);
            if ($coupon) {
                $this->flushCouponCaches($coupon);
            }
        }

        return $updated;
    }

    public function checkCodeUnique($code, $id = null)
    {
        return Coupon::where('code', $code)
            ->when($id, function ($query) use ($id) {
                return $query->where('id', '!=', $id);
            })
            ->doesntExist();
    }

    protected function userCouponsCacheKey($userId, $isValid, $orderBy): string
    {
        return sprintf('coupon_user_%s_valid_%s_order_%s', $userId, (int) $isValid, strtolower($orderBy));
    }

    protected function couponDetailsCacheKey($id): string
    {
        return "coupon_details_{$id}";
    }

    protected function flushCouponCaches(?Coupon $coupon): void
    {
        if (!$coupon) {
            return;
        }

        Cache::forget($this->couponDetailsCacheKey($coupon->id));

        $userId = $coupon->created_by;

        if (!$userId) {
            return;
        }

        foreach ([0, 1] as $isValid) {
            foreach (['asc', 'desc'] as $order) {
                Cache::forget($this->userCouponsCacheKey($userId, $isValid, $order));
            }
        }
    }
}