<?php

namespace Modules\Communication\Tests\Unit;

use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Modules\Communication\app\Services\FirebaseNotificationService;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

#[CoversClass(FirebaseNotificationService::class)]
class FirebaseNotificationServiceTest extends TestCase
{
    /** @var list<string> */
    private array $temporaryFiles = [];

    protected function tearDown(): void
    {
        foreach ($this->temporaryFiles as $file) {
            if (is_file($file)) {
                unlink($file);
            }
        }

        parent::tearDown();
    }

    public function testReturnsErrorWhenServiceAccountMissing(): void
    {
        $service = new FirebaseNotificationService(
            fcmUrl: 'https://example.com/messages:send',
            serviceAccountPath: '/tmp/does-not-exist.json',
        );

        $response = $service->sendNotification('Title', 'Body', 'device-token');

        $this->assertArrayHasKey('error', $response);
        $this->assertStringContainsString('Service account file not found at:', $response['error']);
    }

    public function testReturnsErrorWhenAccessTokenIsMissing(): void
    {
        $path = $this->createTemporaryServiceAccountFile();

        $service = new FirebaseNotificationService(
            httpClient: $this->createMock(ClientInterface::class),
            tokenResolver: static fn (string $path): array => [],
            fcmUrl: 'https://example.com/messages:send',
            serviceAccountPath: $path,
        );

        $response = $service->sendNotification('Title', 'Body', 'device-token');

        $this->assertSame(['error' => 'Unable to fetch access token.'], $response);
    }

    public function testReturnsErrorWhenRequestFails(): void
    {
        $path = $this->createTemporaryServiceAccountFile();

        /** @var ClientInterface&MockObject $httpClient */
        $httpClient = $this->createMock(ClientInterface::class);

        $httpClient->expects($this->once())
            ->method('request')
            ->willThrowException(new RequestException(
                'network error',
                $this->createMock(RequestInterface::class),
            ));

        $service = new FirebaseNotificationService(
            httpClient: $httpClient,
            tokenResolver: static fn (string $path): array => ['access_token' => 'token'],
            fcmUrl: 'https://example.com/messages:send',
            serviceAccountPath: $path,
        );

        $response = $service->sendNotification('Title', 'Body', 'device-token');

        $this->assertSame(['error' => 'network error'], $response);
    }

    public function testReturnsDecodedResponseOnSuccess(): void
    {
        $path = $this->createTemporaryServiceAccountFile();

        $responseBody = $this->createMock(StreamInterface::class);
        $responseBody->method('__toString')->willReturn(json_encode(
            ['name' => 'projects/example/messages/123'],
            JSON_THROW_ON_ERROR,
        ));

        /** @var ResponseInterface&MockObject $response */
        $response = $this->createMock(ResponseInterface::class);
        $response->expects($this->once())->method('getBody')->willReturn($responseBody);

        /** @var ClientInterface&MockObject $httpClient */
        $httpClient = $this->createMock(ClientInterface::class);

        $capturedOptions = [];

        $httpClient->expects($this->once())
            ->method('request')
            ->with(
                'POST',
                'https://example.com/messages:send',
                $this->callback(function (array $options) use (&$capturedOptions): bool {
                    $capturedOptions = $options;

                    return true;
                }),
            )
            ->willReturn($response);

        $service = new FirebaseNotificationService(
            httpClient: $httpClient,
            tokenResolver: static fn (string $path): array => ['access_token' => 'token'],
            fcmUrl: 'https://example.com/messages:send',
            serviceAccountPath: $path,
        );

        $result = $service->sendNotification(
            'Sample Title',
            'Sample Body',
            'device-token',
            ['custom_key' => 'custom_value'],
        );

        $this->assertSame(['name' => 'projects/example/messages/123'], $result);
        $this->assertArrayHasKey('json', $capturedOptions);
        $this->assertSame('Bearer token', $capturedOptions['headers']['Authorization']);
        $this->assertSame('application/json', $capturedOptions['headers']['Content-Type']);
        $this->assertSame('Sample Title', $capturedOptions['json']['message']['notification']['title']);
        $this->assertSame('Sample Body', $capturedOptions['json']['message']['notification']['body']);
        $this->assertSame('device-token', $capturedOptions['json']['message']['token']);
        $this->assertSame(['custom_key' => 'custom_value'], $capturedOptions['json']['message']['data']);
    }

    private function createTemporaryServiceAccountFile(): string
    {
        $path = tempnam(sys_get_temp_dir(), 'service-account-');

        if ($path === false) {
            $this->fail('Unable to create temporary service account file.');
        }

        file_put_contents($path, '{}');

        $this->temporaryFiles[] = $path;

        return $path;
    }
}
