<?php

namespace Modules\Communication\app\Services;

use Google\Auth\Credentials\ServiceAccountCredentials;
use GuzzleHttp\Client;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

class FirebaseNotificationService
{
    protected ClientInterface $httpClient;

    protected string $fcmUrl;

    protected string $serviceAccountPath;

    /**
     * @var callable(string $serviceAccountPath): array<string, string>
     */
    private $tokenResolver;

    public function __construct(
        ?ClientInterface $httpClient = null,
        ?callable $tokenResolver = null,
        ?string $fcmUrl = null,
        ?string $serviceAccountPath = null,
    ) {
        $this->httpClient = $httpClient ?? new Client(['timeout' => 10]);

        $fcmConfig = config('communication.services.fcm', []);

        $this->fcmUrl = $fcmUrl
            ?? (string) ($fcmConfig['url'] ?? 'https://fcm.googleapis.com/v1/projects/unknown/messages:send');

        $this->serviceAccountPath = $serviceAccountPath
            ?? (string) ($fcmConfig['service_account'] ?? storage_path('firebase/firebase_service_account.json'));

        $this->tokenResolver = $tokenResolver ?? static function (string $serviceAccountPath): array {
            $credentials = ServiceAccountCredentials::fromJsonFile($serviceAccountPath);

            return $credentials->fetchAuthToken();
        };
    }

    /**
     * @param array<string, string> $data
     *
     * @return array<string, mixed>
     */
    public function sendNotification(string $title, string $body, string $deviceToken, array $data = []): array
    {
        if (!is_file($this->serviceAccountPath) || !is_readable($this->serviceAccountPath)) {
            return ['error' => sprintf('Service account file not found at: %s', $this->serviceAccountPath)];
        }

        $token = ($this->tokenResolver)($this->serviceAccountPath);

        if (!isset($token['access_token'])) {
            return ['error' => 'Unable to fetch access token.'];
        }

        $payload = [
            'message' => [
                'token' => $deviceToken,
                'notification' => [
                    'title' => $title,
                    'body' => $body,
                ],
                'data' => $data,
            ],
        ];

        try {
            $response = $this->httpClient->request('POST', $this->fcmUrl, [
                'json' => $payload,
                'headers' => [
                    'Authorization' => 'Bearer ' . $token['access_token'],
                    'Content-Type' => 'application/json',
                ],
            ]);

            $body = (string) $response->getBody();

            return json_decode($body, true) ?? [];
        } catch (GuzzleException $exception) {
            return ['error' => $exception->getMessage()];
        }
    }
}
