<?php

namespace Modules\Chat\app\Repositories\Eloquent;

use App\Models\Bookings;
use App\Models\User;
use App\Models\UserDetail;
use App\Services\MqttService;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use JsonException;
use Modules\Chat\app\Models\Message;
use Modules\Chat\app\Repositories\Contracts\ChatRepositoryInterface;
use Modules\Chat\app\Transformers\ChatResource;
use Modules\Leads\app\Models\ProviderFormsInput;
use Modules\Product\app\Models\Product;
use Throwable;

class ChatRepository implements ChatRepositoryInterface
{
    private const DEFAULT_PER_PAGE = 10;
    private const MAX_PER_PAGE = 50;

    public function adminChat(Request $request): array
    {
        return $this->buildUserListing($request, static fn (Builder $query): Builder => $query);
    }

    public function providerChat(Request $request): array
    {
        $authId = (int) (Auth::id() ?? 0);
        $relatedUserIds = $this->getRelatedUsers($authId);

        return $this->buildUserListing(
            $request,
            static function (Builder $query) use ($authId, $relatedUserIds): Builder {
                return $query->where(function (Builder $inner) use ($authId, $relatedUserIds) {
                    if (!empty($relatedUserIds)) {
                        $inner->whereIn('id', $relatedUserIds);
                    }

                    $inner->orWhereHas('userDetail', static function (Builder $subQuery) use ($authId): void {
                        $subQuery->where('parent_id', $authId);
                    });
                });
            },
            [
                'chatUserId' => $this->decryptChatUserId($request),
            ]
        );
    }

    public function userChat(Request $request): array
    {
        $authId = (int) (Auth::id() ?? 0);
        $relatedUserIds = $this->getRelatedUsers($authId);

        return $this->buildUserListing(
            $request,
            static function (Builder $query) use ($relatedUserIds): Builder {
                if (empty($relatedUserIds)) {
                    return $query->whereRaw('0 = 1');
                }

                return $query->whereIn('id', $relatedUserIds);
            },
            [
                'chatUserId' => $this->decryptChatUserId($request),
            ]
        );
    }

    public function sendChat(Request $request): array
    {
        $validator = Validator::make($request->all(), [
            'sender_id' => ['required', 'integer'],
            'receiver_id' => ['required', 'integer', 'different:sender_id'],
            'messageType' => ['nullable', 'in:text,file'],
            'message' => ['nullable', 'string'],
            'topic' => ['required', 'string'],
            'file' => ['required_if:messageType,file', 'file'],
        ]);

        if ($validator->fails()) {
            return [
                'success' => false,
                'message' => __('validation.failed'),
                'errors' => $validator->errors()->toArray(),
            ];
        }

        $type = $request->input('messageType', 'text');
        $senderId = (int) $request->input('sender_id');
        $receiverId = (int) $request->input('receiver_id');

        try {
            $message = $type === 'file'
                ? $this->storeFileMessage($request, $senderId, $receiverId)
                : $this->storeTextMessage($request, $senderId, $receiverId);

            if (!$message) {
                return [
                    'success' => false,
                    'message' => __('Message content cannot be empty'),
                ];
            }

            $mqttPayload = [
                'sender_id' => $senderId,
                'receiver_id' => $receiverId,
                'message_id' => $message->id,
                'message' => $message->type === 'file' ? $message->file : $message->content,
                'type' => $message->type,
            ];

            $resource = ChatResource::make($message)->resolve();

            $mqtt = new MqttService();
            $encodedPayload = json_encode($mqttPayload, JSON_THROW_ON_ERROR);
            $mqtt->publish($request->input('topic'), $encodedPayload);
            $mqtt->disconnect();

            return [
                'success' => true,
                'message' => __('Message sent successfully'),
                'data' => $resource,
            ];
        } catch (JsonException $exception) {
            return [
                'success' => false,
                'message' => __('Failed to encode message payload'),
                'error' => $exception->getMessage(),
            ];
        } catch (Throwable $exception) {
            return [
                'success' => false,
                'message' => __('Failed to send message'),
                'error' => $exception->getMessage(),
            ];
        }
    }

    public function fetchMessages(Request $request): array
    {
        $authUserId = (int) (Auth::id() ?? 0);
        $messagePartnerId = (int) $request->input('user_id', 0);

        if ($authUserId === 0 || $messagePartnerId === 0) {
            return [
                'status' => false,
                'code' => 422,
                'messages' => [],
                'next_offset' => null,
                'last_offset' => null,
                'last_message' => null,
            ];
        }

        $perPage = $this->resolvePerPage($request->input('per_page'));
        $conversationQuery = $this->buildConversationQuery($authUserId, $messagePartnerId);

        $totalMessages = (clone $conversationQuery)->count();

        if ($totalMessages === 0) {
            return [
                'status' => true,
                'code' => 200,
                'messages' => [],
                'next_offset' => null,
                'last_offset' => null,
                'last_message' => null,
            ];
        }

        $offset = $this->resolveOffset($request->input('offset'), $totalMessages, $perPage);

        $messages = (clone $conversationQuery)
            ->orderBy('id')
            ->offset($offset)
            ->limit($perPage)
            ->get();

        $lastMessage = (clone $conversationQuery)
            ->orderByDesc('id')
            ->first();

        return [
            'status' => true,
            'code' => 200,
            'messages' => ChatResource::collection($messages)->resolve(),
            'next_offset' => $offset === 0 ? null : max(0, $offset - $perPage),
            'last_offset' => $offset,
            'last_message' => $lastMessage ? [
                'id' => $lastMessage->id,
                'message' => Str::limit($lastMessage->content, 20, '...'),
                'created_at' => optional($lastMessage->created_at)->diffForHumans(),
            ] : null,
        ];
    }

    public function getRelatedUsers(int $user_id): array
    {
        $authUser = User::find($user_id);
        if (!$authUser) {
            return [];
        }

        $adminUserIds = User::where('user_type', 1)->pluck('id')->toArray();
        $relatedUserIds = [];

        if ($authUser->user_type === 3) {
            $bookedServices = Bookings::where('user_id', $authUser->id)->pluck('product_id');
            $sellerIds = Product::whereIn('id', $bookedServices)->pluck('user_id')->toArray();
            // $providerIds = ProviderFormsInput::whereHas('userFormInput', static function ($query) use ($user_id): void {
            //     $query->where('user_id', $user_id)->whereNull('deleted_at');
            // })->pluck('provider_id')->unique()->toArray();

            $relatedUserIds = array_merge($sellerIds);
        } elseif ($authUser->user_type === 2) {
            $myServices = Product::where('user_id', $authUser->id)->pluck('id');
            $buyerIds = Bookings::whereIn('product_id', $myServices)->pluck('user_id')->toArray();
            $relatedUserIds = $buyerIds;
        } elseif ($authUser->user_type === 4) {
            $providerIds = UserDetail::where('user_id', $authUser->id)
                ->whereNotNull('parent_id')
                ->pluck('parent_id')
                ->toArray();

            $myServices = Product::where('user_id', $authUser->id)->pluck('id');
            $buyerIds = Bookings::where(function ($query) use ($myServices, $authUser): void {
                $query->whereIn('product_id', $myServices)
                    ->orWhere('staff_id', $authUser->id);
            })->pluck('user_id')->toArray();

            $relatedUserIds = array_merge($providerIds, $buyerIds);
        }

        $relatedUserIds = array_merge($relatedUserIds, $adminUserIds);

        return array_values(array_unique(array_map('intval', $relatedUserIds)));
    }

    /**
     * @param  Request  $request
     * @param  callable(Builder):Builder  $constraints
     * @param  array<string, mixed>  $additionalData
     * @return array<string, mixed>
     */
    private function buildUserListing(Request $request, callable $constraints, array $additionalData = []): array
    {
        $authId = (int) (Auth::id() ?? 0);
        $perPage = $this->resolvePerPage($request->input('per_page'));

        $query = User::with('userDetail')
            ->select('users.id', 'users.email', 'users.name')
            ->where('users.id', '!=', $authId)
            ->where('status', 1);

        $query = $constraints($query) ?? $query;

        $this->applySearchFilter($query, (string) $request->input('search', ''));

        $paginator = $query->paginate($perPage);
        $formattedUsers = $paginator->getCollection()->map(function (User $user): User {
            return $this->formatUser($user);
        });

        $paginator->setCollection($formattedUsers);

        return array_merge([
            'users' => $formattedUsers->values()->all(),
            'current_page' => $paginator->currentPage(),
            'last_page' => $paginator->lastPage(),
            'sender' => Auth::user(),
        ], $additionalData);
    }

    private function resolvePerPage($perPage): int
    {
        $perPage = is_numeric($perPage) ? (int) $perPage : self::DEFAULT_PER_PAGE;

        if ($perPage <= 0) {
            $perPage = self::DEFAULT_PER_PAGE;
        }

        return min($perPage, self::MAX_PER_PAGE);
    }

    private function applySearchFilter(Builder $query, string $search): void
    {
        $search = trim($search);
        if ($search === '') {
            return;
        }

        $words = array_filter(preg_split('/\s+/', $search) ?: []);

        if (empty($words)) {
            return;
        }

        $query->where(function (Builder $builder) use ($words): void {
            foreach ($words as $word) {
                $builder->where(function (Builder $inner) use ($word): void {
                    $inner->where('users.name', 'like', "%{$word}%")
                        ->orWhereHas('userDetail', static function (Builder $subQuery) use ($word): void {
                            $subQuery->where('first_name', 'like', "%{$word}%")
                                ->orWhere('last_name', 'like', "%{$word}%");
                        });
                });
            }
        });
    }

    private function formatUser(User $user): User
    {
        $firstName = optional($user->userDetail)->first_name;
        $lastName = optional($user->userDetail)->last_name;
        $displayName = $firstName ? trim($firstName . ' ' . ($lastName ?? '')) : $user->name;
        $user->name = ucwords($displayName);

        $profilePath = optional($user->userDetail)->profile_image;

        if ($profilePath && Storage::disk('public')->exists('profile/' . $profilePath)) {
            $user->profile_image = Storage::url('profile/' . $profilePath);
        } else {
            $user->profile_image = asset('assets/img/profile-default.png');
        }

        return $user;
    }

    private function decryptChatUserId(Request $request): string
    {
        if (!$request->filled('user_id')) {
            return '';
        }

        try {
            return (string) customDecrypt($request->user_id, User::$userSecretKey);
        } catch (Throwable $exception) {
            return '';
        }
    }

    private function storeTextMessage(Request $request, int $senderId, int $receiverId): ?Message
    {
        $content = trim((string) $request->input('message', ''));

        if ($content === '') {
            return null;
        }

        return Message::create([
            'from_user_id' => $senderId,
            'to_user_id' => $receiverId,
            'type' => 'text',
            'content' => $content,
        ]);
    }

    private function storeFileMessage(Request $request, int $senderId, int $receiverId): ?Message
    {
        if (!$request->hasFile('file')) {
            return null;
        }

        $file = $request->file('file');
        $path = $file->store('chat', 'public');

        return Message::create([
            'from_user_id' => $senderId,
            'to_user_id' => $receiverId,
            'type' => 'file',
            'file' => $path,
            'mime_type' => $file->getClientMimeType(),
            'size' => (string) $file->getSize(),
            'content' => $file->getClientOriginalName(),
        ]);
    }

    private function buildConversationQuery(int $authUserId, int $messagePartnerId): Builder
    {
        return Message::query()
            ->where(function (Builder $query) use ($authUserId, $messagePartnerId): void {
                $query->where('from_user_id', $authUserId)
                    ->where('to_user_id', $messagePartnerId);
            })
            ->orWhere(function (Builder $query) use ($authUserId, $messagePartnerId): void {
                $query->where('from_user_id', $messagePartnerId)
                    ->where('to_user_id', $authUserId);
            });
    }

    private function resolveOffset($offset, int $totalMessages, int $perPage): int
    {
        if (!is_numeric($offset)) {
            return max(0, $totalMessages - $perPage);
        }

        $offset = (int) $offset;

        if ($offset < 0) {
            return 0;
        }

        return min($offset, max(0, $totalMessages - 1));
    }
}
