<?php

namespace Modules\Categories\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\View\View;
use Modules\Categories\app\Models\Categories;
use Modules\Categories\app\Support\CategoryCache;

class FrontendCategoryController extends Controller
{
    public function serviceCategories(): View
    {
        return view('categories::frontend.service_categories');
    }

    public function productCategories(): View
    {
        return view('categories::frontend.product_categories');
    }

    public function getAllCategories(Request $request): JsonResponse
    {
        try {
            $languageId = getLanguageId(app()->getLocale());

            $validator = Validator::make($request->all(), [
                'per_page' => 'nullable|integer|min:1|max:100',
                'type' => 'nullable|in:service,product',
                'sort_by' => 'nullable|in:newest,oldest,name_asc,name_desc',
                'page' => 'nullable|integer|min:1',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'code' => 422,
                    'message' => $validator->errors()->first(),
                    'errors' => $validator->errors(),
                ], 422);
            }

            $perPage = $request->get('per_page', 8);
            $type = $request->get('type', 'service');
            $sortBy = $request->get('sort_by', 'newest');

            $query = Categories::withCount([
                    'products as product_count' => function ($q) {
                        $q->where('source_type', 'product')
                        ->whereNull('deleted_at')
                        ->whereHas('shop');
                    },
                    'products as service_count' => function ($q) {
                        $q->where('source_type', 'service')
                        ->whereNull('deleted_at')
                        ->whereHas('shop');
                    }
                ])
                ->where('status', 1)
                ->where('source_type', $type)
                ->where('parent_id', 0)
                ->where('language_id', $languageId);

            if ($sortBy == 'newest') {
                $query->orderBy('id', 'DESC');
            } elseif ($sortBy == 'oldest') {
                $query->orderBy('id', 'ASC');
            } elseif ($sortBy == 'name_asc') {
                $query->orderBy('name', 'ASC');
            } elseif ($sortBy == 'name_desc') {
                $query->orderBy('name', 'DESC');
            } else {
                $query->orderBy('id', 'DESC');
            }

            $paginatedData = $query->paginate($perPage);
            $paginatedData->getCollection()->transform(function ($category) {
                $category->image = uploadedAsset($category->image);
                $category->icon = uploadedAsset($category->icon);
                return $category;
            });

            $finalData = [
                'categories' => $paginatedData->items(),
                'current_page' => $paginatedData->currentPage(),
                'next_page' => $paginatedData->hasMorePages() ? $paginatedData->currentPage() + 1 : null,
                'total'         => $paginatedData->total(),
                'per_page'      => $paginatedData->perPage(),
            ];

            return response()->json([
                'status' => 'success',
                'code' => 200,
                'data' => $finalData,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'code' => 500,
                'message' => __('default_retrieve_error'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}
